/************************************************************************/
/*									*/
/*		gprofinter.c						*/
/*									*/
/*	Interactive GPROF printing routines				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


/*
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

#ifndef lint
static char sccsid[] = "@(#)printgprof.c	5.1 (Berkeley) 6/4/85";
#endif not lint

#include "gprof.h"


/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	int		countcmp();
static	int		alphacmp();
static	void		incrprofline();




/************************************************************************/
/*									*/
/*	printinterprof -- print interactive profile information 	*/
/*									*/
/************************************************************************/


printinterprof(order,file,fct,line)
   SORT_ORDER order;
   char * file;
   char * fct;
   char * line;
{
   register nltype     *np;
   nltype	       **sortednlp;
   int		       index;
   int aname;
   int filefg,fctfg,ufg,pfile,pfct,pline;
   int timecmp();

   actime = 0.0;

   if (file[0] == 0 || file[0] == '*') file = NULL;
   if (fct[0] == 0 || fct[0] == '*') fct = NULL;
   if (line[0] == 0 || line[0] == '*') line = NULL;

   if (file != NULL && file[0] == '?') {
      file = NULL;
      pfile = TRUE;
    }
   else pfile = FALSE;
   if (fct != NULL && fct[0] == '?') {
      fct = NULL;
      pfct = TRUE;
    }
   else pfct = FALSE;
   if (line != NULL && line[0] == '?') {
      line = NULL;
      pline = TRUE;
    }
   else pline = FALSE;

   filefg = (file == NULL);
   fctfg = (fct == NULL);

   sortednlp = (nltype **) calloc( nname , sizeof(nltype *) );
   if ( sortednlp == (nltype **) 0 ) {
      fprintf( stderr , "[printprof] ran out of memory for time sorting\n" );
    }

   aname = 0;
   for ( index = 0 ; index < nname ; index += 1 ) {
      ufg = FALSE;
      switch (nl[index].type) {
	 case TYPE_NONE :
	    break;
	 case TYPE_FILE :
	    if (file != NULL) filefg = (strcmp(nl[index].name,file) == 0);
	    ufg = pfile;
	    break;
	 case TYPE_FUNCT :
	 case TYPE_CYCLE :
	    if (fct != NULL) fctfg = (strcmp(nl[index].name,fct) == 0);
	    ufg = pfct;
	    break;
	 case TYPE_LINE :
	    ufg = pline;
	    break;
       };
      if (filefg && fctfg && ufg) sortednlp[aname++] = &nl[ index ];
    }

   switch (order) {
      case SORT_SYMS :
	 break;
      case SORT_TIME :
	 qsort( sortednlp , aname , sizeof(nltype *) , timecmp );
	 break;
      case SORT_COUNT :
	 qsort( sortednlp , aname , sizeof(nltype *) , countcmp );
	 break;
      case SORT_ALPHA :
	 qsort( sortednlp , aname , sizeof(nltype *) , alphacmp );
	 break;
    };

   for ( index = 0 ; index < aname ; index += 1 ) {
      np = sortednlp[ index ];
      incrprofline( np );
    }

   actime = 0.0;
   cfree( sortednlp );
}




/************************************************************************/
/*									*/
/*	countcmp -- compare two symbols by call counts			*/
/*									*/
/************************************************************************/


static int
countcmp( npp1 , npp2 )
    nltype **npp1, **npp2;
{
    long	calldiff;

    calldiff = (*npp2) -> ncall - (*npp1) -> ncall;
    if ( calldiff > 0 )
	return 1;
    if ( calldiff < 0 )
	return -1;
    return( strcmp( (*npp1) -> name , (*npp2) -> name ) );
}





/************************************************************************/
/*									*/
/*	alphacmp -- compare two symbols alphabetically			*/
/*									*/
/************************************************************************/


static int
alphacmp( npp1 , npp2 )
    nltype **npp1, **npp2;
{
    return( strcmp( (*npp1) -> name , (*npp2) -> name ) );
}





/************************************************************************/
/*									*/
/*	incrprofline -- output line for incremental profiling		*/
/*									*/
/************************************************************************/


static void
incrprofline( np )
    nltype     *np;
{

    if ( zflag && np -> ncall == 0 && np -> time == 0 ) {
	return;
    }
    if (np->type == TYPE_FUNCT || np->type == TYPE_CYCLE)
       actime += np -> time;

    if (minflag != 0 && np->time/totime*10000 < minflag) return;

    printf( "%5.1f %10.2f %8.2f" ,
	100 * np -> time / totime , actime / hz , np -> time / hz );
    if ( np -> ncall != 0 ) {
	printf( " %8d %8.2f %8.2f  " , np -> ncall ,
	    1000 * np -> time / hz / np -> ncall ,
	    1000 * ( np -> time + np -> childtime ) / hz / np -> ncall );
    } else {
	printf( " %8.8s %8.8s %8.8s  " , "" , "" , "" );
    }
    printname( np );
    printf( "\n" );
}





/************************************************************************/
/*									*/
/*	printintergprof -- do interactive gprof output			*/
/*									*/
/************************************************************************/



printintergprof(nlp,file,fct)
   nltype ** nlp;
   char * file;
   char * fct;
{
   int filefg,fctfg;
   nltype * parentp;
   int index;
   int ufg;

   if (file[0] == 0 || file[0] == '*') file = NULL;
   if (fct[0] == 0 || fct[0] == '*') fct = NULL;

   if (file != NULL && file[0] == '?') file = NULL;
   if (fct != NULL && fct[0] == '?') fct = NULL;

   filefg = (file == NULL);
   fctfg = (fct == NULL);

   for ( index = 0 ; index < nname + ncycle ; index ++ ) {
      parentp = nlp[ index ];

      ufg = FALSE;
      switch (nl[index].type) {
	 case TYPE_NONE :
	    break;
	 case TYPE_FILE :
	    if (file != NULL) filefg = (strcmp(parentp->name,file) == 0);
	    break;
	 case TYPE_FUNCT :
	 case TYPE_CYCLE :
	    if (fct != NULL) fctfg = (strcmp(parentp->name,fct) == 0);
	    ufg = TRUE;
	    break;
	 case TYPE_LINE :
	    break;
       };
      if (!filefg || !fctfg || !ufg) continue;

      if ( zflag && parentp -> ncall == 0 && parentp -> selfcalls == 0 &&
	      parentp -> propself == 0 && parentp -> propchild == 0 )
	 continue;

      if ( ! parentp -> printflag) continue;

      if ( parentp -> name == 0 && parentp -> cycleno != 0 ) {
	 printcycle( parentp );
	 printmembers( parentp );
       }
      else {
	 printparents( parentp );
	 gprofline( parentp );
	 printchildren( parentp );
       }
      printf( "\n" );
      printf( "\n" );
    }
};





/************************************************************************/
/*									*/
/*	printintergraph -- output graphing information for profiling	*/
/*									*/
/************************************************************************/


#define COUNT_PER_SYMBOL	30

printintergraph(lo,hi,cnt)
   float lo,hi;
   int cnt;
{
   long lopc,hipc,sval;
   int incr;
   int i,j,k,l;
   int fctct,filct,linct;
   int scale;
   unsigned long pcl,pch,addr;
   UNIT ccnt;
   double time,ttime;
   char * typ;
   int fg;

   i = (highpc - lowpc);
   lopc = ((int) (i*lo)) + lowpc;
   hipc = ((int) (i*hi)) + lowpc;

   fctct = 0;
   filct = 0;
   linct = 0;
   for (j = 0; j < nname; ++j) {
      if (nl[j].svalue < lopc || nl[j+1].svalue > hipc) continue;
      switch (nl[j].type) {
	 case TYPE_FILE :
	    filct++;
	    break;
	 case TYPE_FUNCT :
	    fctct++;
	    break;
	 case TYPE_LINE :
	    linct++;
	    break;
       };
    };

   scale = (highpc - lowpc)/nsamples;
   incr = (hipc-lowpc)/cnt;
   incr /= scale;
   if (incr < 1) incr = 1;

   if (fctct > cnt/COUNT_PER_SYMBOL) {
      fctct = 0;
      linct = 0;
    }
   else if (linct > cnt/COUNT_PER_SYMBOL) linct = 0;

   printf("RANGE 0x%x 0x%x\n",lopc,hipc);

   k = 0;
   time = 0;
   ttime = 0;
   addr = 0;
   l = 0;
   for (i = 0, j = 1; i < nsamples; ++i) {
      pcl = lowpc + scale*i;
      pch = lowpc + scale*(i+1);
      if (pch < lopc || pcl > hipc) continue;

      ccnt = samples[i];
      time += ccnt;
      ttime += ccnt;
      if (k == 0) addr = pcl;
      k = (k+1) % incr;
      if (k == 0) {
	 printf("SAMPLE 0x%x %f\n",addr,time/totime);
	 ++l;
	 time = 0;
       };
      for (j = j-1; j < nname; j++) {
	 sval = nl[j].svalue;
	 if (pcl > sval) continue;
	 if (pch < sval) break;
	 switch (nl[j].type) {
	    case TYPE_FILE :
	       typ = "FILE";
	       break;
	    case TYPE_FUNCT :
	       typ = (fctct > 0 ? "FUNCT" : NULL);
	       break;
	    case TYPE_LINE :
	       typ = (linct > 0 ? "LINE" : NULL);
	       break;
	    default :
	       typ = NULL;
	       break;
	  };
	 if (typ != NULL) {
	    printf("SYMBOL 0x%x %s %s\n",sval,typ,nl[j].name);
	  };
       };
    };

   printf("TOTAL %d %f\n",l,ttime/totime);
};





/* end of gprofinter.c */
