/************************************************************************/
/*									*/
/*		fprofdata.c						*/
/*									*/
/*	Data maintenance routines for profile viewer			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "fprof_local.h"




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		free_line();
static	Integer 	linenumber();
static	Integer 	find_line();
static	Integer 	lineinfo();
static	Integer 	find_info();
static	Integer 	linelevel();
static	Integer 	find_level();



/************************************************************************/
/*									*/
/*	FPROF_data_init -- module initialization			*/
/*									*/
/************************************************************************/


void
FPROF_data_init()
{
};





/************************************************************************/
/*									*/
/*	FPROF_data_clear -- clear data for profile			*/
/*	free_line -- free a FPROF_LINE structure			*/
/*									*/
/************************************************************************/


void
FPROF_data_clear(fw)
   FPROF_WIN fw;
{
   free_line(fw->data);
   fw->data = NULL;

   FPROF_edit_clear(fw,1);
};




static void
free_line(fl)
   FPROF_LINE fl;
{
   if (fl == NULL) return;

   free_line(fl->son);
   free_line(fl->next);

   SFREE(fl->name);
   free(fl);
};





/************************************************************************/
/*									*/
/*	FPROF_data_gather -- gather data for line			*/
/*									*/
/************************************************************************/


void
FPROF_data_gather(fw,fl,buf)
   FPROF_WIN fw;
   FPROF_LINE fl;
   String buf;
{
   Integer i,j,lnno;
   FPROF_LINE fs,nl;
   Character ebuf[1024];

   if (buf == NULL) {
      if (fl != NULL && fl->son != NULL) {
	 i = linenumber(fw,fl);
	 if (fl->next != NULL) j = linenumber(fw,fl->next);
	 else j = 0;
	 free_line(fl->son);
	 fl->son = NULL;
	 if (j == 0) FPROF_edit_clear(fw,i+1);
	 else FPROF_edit_remove(fw,i+1,j-1);
       };
      return;
    };

   if (fl != NULL) {
      for (fs = fl->son; fs != NULL && fs->next != NULL; fs = fs->next);
    }
   else {
      for (fs = fw->data; fs != NULL && fs->next != NULL; fs = fs->next);
    };

   if (fs == NULL && fl == NULL) lnno = 0;
   else if (fs == NULL) lnno = linenumber(fw,fl);
   else lnno = linenumber(fw,fs);

   nl = PALLOC(FPROF_LINE_INFO);
   nl->next = NULL;
   nl->son = NULL;
   nl->name = NULL;

   if (fl == NULL && fs == NULL) fw->data = nl;
   else if (fs == NULL) fl->son = nl;
   else fs->next = nl;

   i = strlen(buf)-1;
   while (isspace(buf[i])) --i;
   while (!isspace(buf[i])) --i;
   ++i;

   nl->name = SALLOC(&buf[i]);

   strcpy(ebuf,buf);
   j = linelevel(fw,fl);
   ebuf[i] = 0;

   for (i = 0; i < j; ++i) {
      strcat(ebuf,"  ");
    };

   strcat(ebuf,nl->name);

   FPROF_edit_insert(fw,lnno+1,ebuf);
};





/************************************************************************/
/*									*/
/*	FPROF_data_expand -- handle expand request for data		*/
/*									*/
/************************************************************************/


Boolean
FPROF_data_expand(fw,line)
   FPROF_WIN fw;
   Integer line;
{
   FPROF_LINE par[4];

   lineinfo(fw,line,par);

   if (par[0] == NULL) return FALSE;

   if (par[1] == NULL) {
      FPROF_run_profile(fw,par[0],par[0]->name,NULL);
    }
   else if (par[2] == NULL) {
      FPROF_run_profile(fw,par[1],par[0]->name,par[1]->name);
    }
   else return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	linenumber -- find line number of FPROF_LINE block		*/
/*	find_line -- actually search for line number			*/
/*									*/
/************************************************************************/


static Integer
linenumber(fw,fl)
   FPROF_WIN fw;
   FPROF_LINE fl;
{
   Integer i;

   i = find_line(fw->data,fl,1);

   if (i >= 0) return -1;

   return -i;
};





static Integer
find_line(fl,find,line)
   FPROF_LINE fl;
   FPROF_LINE find;
   Integer line;
{
   Integer i;

   while (fl != NULL) {
      if (fl == find) return -line;

      i = find_line(fl->son,find,line+1);
      if (i < 0) return i;

      line = i;
      fl = fl->next;
    };

   return line;
};





/************************************************************************/
/*									*/
/*	lineinfo -- find information for line number			*/
/*									*/
/************************************************************************/


static Integer
lineinfo(fw,line,fls)
   FPROF_WIN fw;
   Integer line;
   FPROF_LINE fls[];
{
   Integer i;

   i = find_info(fw,fw->data,line,1,fls);

   fls[i] = NULL;

   return i;
};





static Integer
find_info(fw,fl0,line,i,par)
   FPROF_WIN fw;
   FPROF_LINE fl0;
   Integer line;
   Integer i;
   FPROF_LINE par[];
{
   FPROF_LINE fl;
   Integer i0,i1;

   for (fl = fl0; fl != NULL; fl = fl->next) {
      i0 = linenumber(fw,fl);
      if (fl->next == NULL) break;
      i1 = linenumber(fw,fl->next);
      if (line < i1) break;
    };

   if (fl == NULL) return 0;

   par[i-1] = fl;
   if (line == i0)  return i;

   i = find_info(fw,fl->son,line,i+1,par);

   return i;
};





/************************************************************************/
/*									*/
/*	linelevel -- find level of FPROF_LINE block			*/
/*	find_level -- actually search for level 			*/
/*									*/
/************************************************************************/


static Integer
linelevel(fw,fl)
   FPROF_WIN fw;
   FPROF_LINE fl;
{
   Integer i;

   if (fl == NULL) return 0;

   i = find_level(fw->data,fl,1);

   return i;
};





static Integer
find_level(fl,find,lvl)
   FPROF_LINE fl;
   FPROF_LINE find;
   Integer lvl;
{
   Integer i;

   while (fl != NULL) {
      if (fl == find) return lvl;

      i = find_level(fl->son,find,lvl+1);
      if (i >= 0) return i;

      fl = fl->next;
    };

   return -1;
};





/* end of fprofdata.c */

