/************************************************************************/
/*									*/
/*		formwin.c						*/
/*									*/
/*	Window manipulation routines for FORM cm front end		*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/



#include "form_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	void		check_windows();
static	int		form_control();
static	void		form_remove();
static	Boolean 	check_use_proj();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define DISP_WIN	2



static	LEAF_DATA	form_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DISP_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	FORM_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
FORM_win_init()
{
   all_windows = NULL;
};





/************************************************************************/
/*									*/
/*	FORM_create -- create a new form graph window			*/
/*									*/
/************************************************************************/


FORM_WIN
FORM_create(w,name)
   ASH_WINDOW w;
   String name;
{
   register FORM_WIN fw;
   Character buf[256];

   if (name == NULL) name = FORM_request_project(w);
   else name = SALLOC(name);

   if (name == NULL) return NULL;

   sprintf(buf,"formview: %s",name);
   ASHset_window_name(w,buf);
   ASHset_window_id(w,"FORM");
   ASHset_window_defaults(w);

   fw = PALLOC(FORM_WIN_INFO);
   fw->window = w;
   fw->project = name;

   fw->root = NULL;
   fw->selection = NULL;
   fw->nodes = NULL;
   fw->last_trans = NULL;
   fw->trans_max = 0;
   fw->trans_cnt = 0;
   fw->exec_btn = NULL;

   PROTECT;
   all_windows = CONS(fw,all_windows);
   UNPROTECT;

   if (!setup_window(fw)) return NULL;

   if (ASHinput_lock(fw->window,TRUE) != ASH_LOCK_GRANTED) return NULL;

   FORM_disp_setup(fw);
   FORM_menu_setup(fw);

   FORM_proj_setup(fw);

   FORM_disp_reset(fw,TRUE);

   if (fw->disp_win != NULL) ASHvisible(fw->disp_win,TRUE);

   ASHinput_unlock(fw->window);

   return fw;
};





/************************************************************************/
/*									*/
/*	FORM_create_trans -- create a new form transcript window	*/
/*									*/
/************************************************************************/


FORM_WIN
FORM_create_trans(w,name)
   ASH_WINDOW w;
   String name;
{
   register FORM_WIN fw;
   Character buf[256];

   if (name != NULL) {
      name = SALLOC(name);
      sprintf(buf,"form transcript: %s",name);
      ASHset_window_name(w,buf);
    }
   else {
      ASHset_window_name(w,"form transcript");
    };

   ASHset_window_id(w,"FORM");
   ASHset_window_defaults(w);

   fw = PALLOC(FORM_WIN_INFO);
   fw->window = w;
   fw->project = name;

   fw->root = NULL;
   fw->selection = NULL;
   fw->nodes = NULL;
   fw->last_trans = NULL;
   fw->trans_max = 0;
   fw->trans_cnt = 0;
   fw->exec_btn = NULL;
   fw->selection = NULL;
   fw->dispproj = NULL;
   fw->dispitem = NULL;
   fw->ignore = NULL;
   fw->region = NULL;
   fw->disp_win = NULL;
   fw->menu_win = NULL;

   PROTECT;
   all_windows = CONS(fw,all_windows);
   UNPROTECT;

   if (ASHinput_lock(fw->window,TRUE) != ASH_LOCK_GRANTED) return NULL;

   if (fw->project != NULL) fw->root = FORM_proj_find(fw->project);

   FORM_trans_setup(fw,FALSE);

   if (fw->text_win != NULL) ASHvisible(fw->text_win,TRUE);
   else fw = NULL;

   ASHinput_unlock(fw->window);

   return fw;
};





/************************************************************************/
/*									*/
/*	FORM_find_window -- find form window given ash window		*/
/*	FORM_find_project -- find form window given project name	*/
/*									*/
/************************************************************************/


FORM_WIN
FORM_find_window(w)
   ASH_WINDOW w;
{
   Sequence l;
   FORM_WIN fw;

   forin (fw,FORM_WIN,l,all_windows) {
      if (fw->window == w) break;
      else if (fw->disp_win == w) break;
      else if (fw->menu_win == w) break;
      else if (fw->text_win == w) break;
    };

   return fw;
};





FORM_WIN
FORM_find_project(pro)
   String pro;
{
   Sequence l;
   FORM_WIN fw;

   forin (fw,FORM_WIN,l,all_windows) {
      if (STREQL(pro,fw->project)) break;
    };

   return fw;
};





/************************************************************************/
/*									*/
/*	FORM_win_select -- handle selection of a particular item	*/
/*									*/
/************************************************************************/


void
FORM_win_select(fw,itm,upd)
   FORM_WIN fw;
   FORM_ITEM itm;
   Boolean upd;
{
   Boolean fg;

   if (itm != NULL && upd) {
      fg = FORM_item_attr_get(itm,"CHECKED").bool_value;
      if (!fg) itm = FORM_back_scan(itm);
    }

   FORM_disp_select(fw,itm);
};






/************************************************************************/
/*									*/
/*	FORM_win_update -- handle update request for item		*/
/*									*/
/************************************************************************/


void
FORM_win_update(what)
   String what;
{
   FORM_WIN fw;
   Sequence l,la,lb,pl;
   FORM_ITEM fp,fi;
   Boolean fg;

   forin (fw,FORM_WIN,l,all_windows) {
      fg = FALSE;

      forin (fp,FORM_ITEM,la,fw->dispproj) {
	 forin (fi,FORM_ITEM,lb,fp->items) {
	    if (STREQL(fi->name,what)) break;
	  };
	 if (fi != NULL) {
	    if (ASHinput_lock(fw->window,TRUE) == ASH_LOCK_GRANTED) {
	       FORM_proj_update(fi);
	       fg = TRUE;
	       ASHinput_unlock(fw->window);
	     };
	  };
       };

      if (fg) {
	 if (ASHinput_lock(fw->window,TRUE) == ASH_LOCK_GRANTED) {
	    FORM_disp_show(fw);
	    ASHinput_unlock(fw->window);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	FORM_win_output -- handle transcript output request for item	*/
/*									*/
/************************************************************************/


void
FORM_win_output(proj,text)
   FORM_ITEM proj;
   String text;
{
   FORM_WIN fw;
   Sequence l;
   Integer ln;

   forin (fw,FORM_WIN,l,all_windows) {
      if (check_use_proj(fw,proj)) {
	 FORM_trans_output(fw,text);
	 if (fw->disp_win == NULL) continue;
	 if (text == NULL) {
	    if (fw->last_trans == NULL) {
	       fw->trans_max = 1024;
	       fw->last_trans = (String) malloc(fw->trans_max);
	     };
	    fw->trans_cnt = 0;
	    fw->last_trans[fw->trans_cnt] = 0;
	  }
	 else {
	    ln = strlen(text);
	    if (ln+1+fw->trans_cnt > fw->trans_max) {
	       while (ln+1+fw->trans_cnt > fw->trans_max) fw->trans_max *= 2;
	       fw->last_trans = (String) realloc(fw->last_trans,fw->trans_max);
	     };
	    strcpy(&fw->last_trans[fw->trans_cnt],text);
	    fw->trans_cnt += ln;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of an form window		*/
/*									*/
/************************************************************************/


static Boolean
setup_window(fw)
   FORM_WIN fw;
{
   LEAFsetup_window(fw->window,form_leaf,fw);

   LEAFset_control(fw->window,form_control);
   LEAFredraw(fw->window);

   fw->disp_win = LEAFinq_window(fw->window,DISP_WIN,0);
   if (fw->disp_win == NULL) {
      fprintf(stderr,"FORM: Window too small to run form graph viewer\n");
      ASHremove(fw->window);
      return FALSE;
   };

   fw->menu_win = LEAFinq_window(fw->window,MENU_WIN,0);
   fw->text_win = NULL;
   fw->edit_info = NULL;

   LEAFset_refresh(fw->window,check_windows);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   FORM_WIN fw;
   ASH_WINDOW dw;

   fw = FORM_find_window(w);

   if (fw == NULL) return;

   dw = LEAFinq_window(w,DISP_WIN,0);

   if (dw != fw->disp_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	form_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
form_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register FORM_WIN fw;

   fw = FORM_find_window(w);

   if (fw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      form_remove(fw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	form_remove -- remove given form				*/
/*									*/
/************************************************************************/


static void
form_remove(fw)
   FORM_WIN fw;
{
   PROTECT;

   all_windows = REMOB(fw,all_windows);

   UNPROTECT;

   fw->window = NULL;
   if (fw->text_win != NULL && ASHinq_valid_window(fw->text_win))
      ASHremove(fw->text_win);
};





/************************************************************************/
/*									*/
/*	check_use_proj -- check if window uses project			*/
/*									*/
/************************************************************************/


static Boolean
check_use_proj(fw,itm)
   FORM_WIN fw;
   FORM_ITEM itm;
{
   FORM_ITEM p,fp;
   Sequence l,la,pl;
   Boolean fg;

   if (fw->disp_win != NULL) pl = fw->dispproj;
   else if (fw->root != NULL) pl = NULL;
   else if (fw->disp_win == NULL) return TRUE;
   else return FALSE;

   fg = FALSE;
   for (p = itm; p != NULL; p = p->project) {
      if (fw->root == p) fg = TRUE;
      else if (fw->selection != NULL && fw->selection->project == p) fg = TRUE;
      else {
	 forin (fp,FORM_ITEM,la,pl) if (fp == p) break;
	 fg = (fp != NULL);
       };
      if (fg) break;
    };

   return fg;
};






/* end of formwin.c */
