/************************************************************************/
/*									*/
/*		formmsg.c						*/
/*									*/
/*	Message interface for FORM front end				*/
/*									*/
/************************************************************************/


#include "form_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_SEARCH_LEVEL	3





/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _FORM_MONITOR *	FORM_MONITOR;
typedef struct _FORM_ERR *	FORM_ERR;



typedef struct _FORM_MONITOR {		/* data for monitoring output	*/
   String last_file;
   String last_proc;
   Integer last_line;
} FORM_MONITOR_INFO;





typedef struct _FORM_ERR {
   String text;
   String type;
   String format;
   PMAT_PAT pattern;
} FORM_ERR_INFO;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Boolean 	form_service;
static	Sequence	monitor;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		form_system_msg();
static	void		form_make_msg();
static	void		form_compile_msg();
static	void		form_compilego_msg();
static	void		form_command_msg();
static	void		form_wd_msg();
static	void		form_file_msg();
static	void		form_done_msg();
static	void		form_item();
static	FORM_ITEM	search_for_target();
static	FORM_ITEM	find_target();





/************************************************************************/
/*									*/
/*	FORM_msg_init -- module initialization				*/
/*									*/
/************************************************************************/


void
FORM_msg_init()
{
   AUXD hdl;
   FORM_ERR ei;

   form_service = FALSE;

   monitor = NULL;
   hdl = AUXDget_handle(NULL,"FORM");

   for (hdl = AUXDget_handle(hdl,"MONITOR"); hdl != NULL; hdl = AUXDnext_handle(hdl)) {
      ei = PALLOC(FORM_ERR_INFO);
      ei->text = AUXDget_info(hdl,"TEXT");
      if (ei->text == NULL) continue;
      ei->type = AUXDget_info(hdl,"EVENT");
      ei->format = AUXDget_info(hdl,"FORMAT");
      ei->pattern = PMATmake_pattern(ei->text,6,NULL);
      monitor = APPEND(ei,monitor);
    };
};





/************************************************************************/
/*									*/
/*	FORM_msg_start -- start message services			*/
/*									*/
/************************************************************************/


void
FORM_msg_start()
{
   AUXD hdl;
   String msg;

   form_service = MSGservice_register("FORM");

   if (!form_service) return;

   hdl = AUXDget_handle(NULL,"FORM");

   msg = AUXDget_info(hdl,"SYSTEM_MSG");
   if (msg != NULL) MSGregister(msg,form_system_msg,1,NULL);

   msg = AUXDget_info(hdl,"MAKE_MSG");
   if (msg != NULL) MSGregister(msg,form_make_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMPILE_MSG");
   if (msg != NULL) MSGregister(msg,form_compile_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMPILEGO_MSG");
   if (msg != NULL) MSGregister(msg,form_compilego_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMMAND_MSG");
   if (msg != NULL) MSGregister(msg,form_command_msg,2,NULL);

   msg = AUXDget_info(hdl,"FILEWD_MSG");
   if (msg != NULL) MSGregister(msg,form_wd_msg,0,NULL);

   msg = AUXDget_info(hdl,"FILEGET_MSG");
   if (msg != NULL) MSGregister(msg,form_file_msg,3,NULL);

   msg = AUXDget_info(hdl,"MAKEDONE_MSG");
   if (msg != NULL) MSGregister(msg,form_done_msg,3,NULL);
};





/************************************************************************/
/*									*/
/*	FORM_msg_start_output -- begin scanning output			*/
/*	FORM_msg_end_output -- finish scanning output			*/
/*	FORM_msg_check_output -- scan an output line			*/
/*									*/
/************************************************************************/


Universal
FORM_msg_start_output()
{
   FORM_MONITOR fm;

   fm = PALLOC(FORM_MONITOR_INFO);

   fm->last_file = NULL;
   fm->last_proc = NULL;
   fm->last_line = 0;

   return (Universal) fm;
};





void
FORM_msg_end_output(fm)
   FORM_MONITOR fm;
{
   if (fm->last_file != NULL) SFREE(fm->last_file);
   if (fm->last_proc != NULL) SFREE(fm->last_proc);
   free(fm);
};






void
FORM_msg_check_output(fm,msg)
   FORM_MONITOR fm;
   String msg;
{
   Sequence l;
   Integer i,j;
   Universal args[10];
   Character buf[1024],buf1[1024];
   String err;
   String s,t,p;
   FORM_ERR ep;

   forin (ep,FORM_ERR,l,monitor) {
      if (PMATmatch(msg,ep->pattern,args) >= 0) {
	 if (ep->format == NULL) {
	    if (args[0] != NULL) {
	       if (fm->last_file != NULL) SFREE(fm->last_file);
	       fm->last_file = SALLOC(args[0]);
	       fm->last_line = 0;
	     };
	    if (args[3] != NULL) {
	       if (fm->last_proc != NULL) SFREE(fm->last_proc);
	       fm->last_proc = SALLOC(args[3]);
	       fm->last_line = 0;
	     };
	    if (((Integer) args[1]) > 0) fm->last_line = (Integer) args[1];
	    err = (String) args[2];
	  }
	 else {
	    s = buf;
	    for (t = ep->format; *t != 0; ++t) {
	       if (*t != '%') *s++ = *t;
	       else {
		  ++t;
		  if (*t < '1' || *t > '6') *s++ = *t;
		  else {
		     j = *t - '1';
		     p = (String) args[j];
		     if (p == NULL) {
			switch (j) {
			   case 0:
			      p = fm->last_file;
			      break;
			   case 1 :
			      sprintf(buf1,"%d",fm->last_line);
			      p = buf1;
			      break;
			   case 3 :
			      p = fm->last_proc;
			      break;
			 };
		      };
		     if (p != NULL) while (*p != 0) *s++ = *p++;
		   };
		};
	     };
	    *s = 0;
	    err = buf;
	  };

	 if (ep->type != NULL) {
	    if (fm->last_file != NULL && fm->last_line > 0 && err != NULL) {
	       MSGsenda("BUILD %s %s %d %S",
			   ep->type,fm->last_file,fm->last_line,err);
	     }
	    else if (fm->last_file != NULL && err != NULL) {
	       MSGsenda("BUILD %s %s %d %S",
			   ep->type,fm->last_file,9999999,err);
	     };
	  };

	 PMATfree_args(ep->pattern,args);
	 break;
       };
    };
};






/************************************************************************/
/*									*/
/*	FORM_msg_check_trans -- check transcript error messages 	*/
/*									*/
/************************************************************************/


void
FORM_msg_check_trans(fw,tline,btn)
   FORM_WIN fw;
   Integer tline;
   String btn;
{
   Sequence l;
   Integer i,ln;
   Universal args[10];
   String s;
   FORM_ERR ep;
   Character lbuf[10240],fbuf[1024];
   String file;
   Integer line;
   Boolean fail;

   file = NULL;
   line = 0;
   fail = FALSE;

   for (ln = tline; ln > 0; --ln) {
      EDTinq_line_text(fw->edit_info,ln,lbuf);
      for (s = lbuf; isspace(*s); ++s);
      if (*s == 0) continue;
      if (strncmp(lbuf,END_LINE,20) == 0) break;
      strcat(lbuf,"\n");

      forin (ep,FORM_ERR,l,monitor) {
	 if (PMATmatch(lbuf,ep->pattern,args) >= 0) {
	    if (ep->format == NULL) {
	       if (args[0] != NULL) {
		  if (file == NULL) {
		     strcpy(fbuf,args[0]);
		     file = fbuf;
		   }
		  else if (STRNEQ(file,args[0])) fail = TRUE;
		};
	       if (line == 0) line = (Integer) args[1];
	     };

	    PMATfree_args(ep->pattern,args);
	    break;
	  };
       };

      if (fail) break;

      if (file != NULL && line > 0) {
	 MSGsenda("XREF SET %s %d %s",file,line,btn);
       };
    };
};






/************************************************************************/
/*									*/
/*	form_system_msg -- handle message indicating new system 	*/
/*									*/
/************************************************************************/


static void
form_system_msg(sys,rid)
   String sys;
   Integer rid;
{
   if (rid >= 0) MSGreply(rid,NULL);
};





/************************************************************************/
/*									*/
/*	form_make_msg -- handle MAKE request for file			*/
/*									*/
/************************************************************************/


static void
form_make_msg(name,rid)
   String name;
   Integer rid;
{
   FORM_ITEM proj,itm;

   proj = FORM_proj_find(".");

   if (name == NULL || STREQL(name,"*") || name[0] == 0) itm = proj;
   else itm = FORM_item_define(proj,name);

   form_item(itm,rid);
};






/************************************************************************/
/*									*/
/*	form_compile_msg -- handle COMPILE request for file		*/
/*									*/
/************************************************************************/


static void
form_compile_msg(name,rid)
   String name;
   Integer rid;
{
   FORM_ITEM itm;

   itm = FORM_proj_find_file(name);
   itm = search_for_target(itm,".o");

   form_item(itm,rid);
};





/************************************************************************/
/*									*/
/*	form_compilego_msg -- handle COMPGO request for file		*/
/*									*/
/************************************************************************/


static void
form_compilego_msg(name,rid)
   String name;
   Integer rid;
{
   FORM_ITEM itm;

   itm = FORM_proj_find_file(name);
   itm = search_for_target(itm,NULL);

   form_item(itm,rid);
};






/************************************************************************/
/*									*/
/*	form_command_msg -- handle COMMAND message			*/
/*									*/
/************************************************************************/


static void
form_command_msg(sys,cmd,rid)
   String sys;
   String cmd;
   Integer rid;
{
   FORM_ITEM itm,proj;

   proj = FORM_proj_find_file(sys);

   if (!proj->is_project) proj = proj->project;

   if (cmd == NULL || STREQL(cmd,"*") || cmd[0] == 0) itm = proj;
   else itm = FORM_item_define(proj,cmd);

   form_item(itm,rid);
};





/************************************************************************/
/*									*/
/*	form_wd_msg -- handle WD change messages			*/
/*									*/
/************************************************************************/


static void
form_wd_msg(rid)
   Integer rid;
{
   if (rid >= 0) MSGreply(rid,NULL);

   FORM_proj_find(MSGinq_wd());
};





/************************************************************************/
/*									*/
/*	form_file_msg -- handle FILE get (RCS/SCSS) requests		*/
/*									*/
/************************************************************************/


static void
form_file_msg(file,rid)
   String file;
   Integer rid;
{
   FORM_ITEM itm;
   Boolean fg;

   itm = FORM_proj_find_file(file);

   if (itm->is_project || itm == NULL) fg = FALSE;

   if (fg) fg = FORM_item_attr_get(itm->project,"USE_SCC").bool_value;

   if (fg && itm->out_links != NULL) form_item(itm,rid);
   else if (rid >= 0) MSGreply(rid,NULL);
};





/************************************************************************/
/*									*/
/*	form_done_msg -- handle make completed successfully message	*/
/*									*/
/************************************************************************/


static void
form_done_msg(what,outfile,sts,rid)
   String what;
   String outfile;
   Integer sts;
   Integer rid;
{
   if (rid >= 0) MSGreply(rid,NULL);

   FORM_win_update(what);
};





/************************************************************************/
/*									*/
/*	form_item -- do a build for a single item			*/
/*									*/
/************************************************************************/


static void
form_item(itm,rid)
   FORM_ITEM itm;
   Integer rid;
{
   if (itm == NULL) MSGreply(rid,NULL);
   else {
      FORM_back_exec_start(itm,rid);
    };
};





/************************************************************************/
/*									*/
/*	search_for_target -- look for target given source item		*/
/*									*/
/************************************************************************/


static FORM_ITEM
search_for_target(itm,ext)
   FORM_ITEM itm;
   String ext;
{
   Character buf[1024],bufa[1024];
   String s;
   FORM_ITEM tgt,fi;
   FORM_LINK lnk;
   Sequence l;
   Integer ln;

   s = rindex(itm->name,'/');
   if (s == NULL) s = itm->name;
   else ++s;
   strcpy(buf,s);

   s = rindex(buf,'.');
   if (s != NULL) *s = 0;
   strcpy(bufa,buf);
   if (ext != NULL) strcat(bufa,ext);

   tgt = find_target(itm,bufa,0);
   if (tgt != NULL) return tgt;

   if (ext == NULL) {
      strcat(bufa,".o");
      tgt = find_target(itm,bufa,0);
      if (tgt != NULL) itm = tgt;
    };

   tgt = NULL;
   ln = strlen(buf);
   forin (lnk,FORM_LINK,l,itm->out_links) {
      fi = lnk->to;
      if (tgt == NULL) tgt = fi;
      s = fi->name;
      if (strncmp(s,buf,ln) == 0 && (s[ln] == 0 || s[ln] == '.')) {
	 tgt = fi;
	 break;
       };
    };

   if (tgt == NULL) {
      if (ext != NULL) strcat(buf,ext);
      tgt = FORM_item_define(itm->project,buf);
    };

   return tgt;
};




/************************************************************************/
/*									*/
/*	find_target -- recursive search for target			*/
/*									*/
/************************************************************************/


static FORM_ITEM
find_target(base,name,lvl)
   FORM_ITEM base;
   String name;
   Integer lvl;
{
   FORM_ITEM proj,itm;
   Sequence l;
   FORM_LINK lnk;

   proj = base->project;

   forin (itm,FORM_ITEM,l,proj->items) {
      if (STREQL(name,itm->name)) return itm;
    };

   if (lvl > MAX_SEARCH_LEVEL) return NULL;

   forin (lnk,FORM_LINK,l,base->out_links) {
      itm = lnk->to;
      if (STREQL(name,itm->name)) return itm;
    };

   forin (lnk,FORM_LINK,l,base->out_links) {
      itm = find_target(lnk->to,name,lvl+1);
      if (itm != NULL) return itm;
    };

   return NULL;
};





/* end of formmsg.c */
