/************************************************************************/
/*									*/
/*		formitem.c						*/
/*									*/
/*	Routines for managing items and links in FORM system		*/
/*									*/
/************************************************************************/


#include "form_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Sequence		all_roots;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Sequence		list_attr_set();
static	FORM_ATTR_VALUE 	list_attr_get();
static	Boolean 		list_attr_test();




/************************************************************************/
/*									*/
/*	FORM_item_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FORM_item_init()
{
   all_roots = NULL;
}






/************************************************************************/
/*									*/
/*	FORM_item_define -- define a new item				*/
/*									*/
/************************************************************************/


FORM_ITEM
FORM_item_define(proj,name)
   FORM_ITEM proj;
   String name;
{
   FORM_ITEM itm;
   Sequence l;

   if (proj != NULL) {
      forin (itm,FORM_ITEM,l,proj->items) {
	 if (STREQL(name,itm->name)) return itm;
       };
    }
   else {
      forin (itm,FORM_ITEM,l,all_roots) {
	 if (STREQL(name,itm->name)) return itm;
       };
    };

   itm = PALLOC(FORM_ITEM_INFO);
   itm->name = SALLOC(name);
   itm->project = proj;
   itm->in_links = NULL;
   itm->out_links = NULL;
   itm->attrs = NULL;
   itm->items = NULL;
   itm->is_project = FALSE;
   itm->locked = FALSE;

   if (proj == NULL) {
      itm->is_project = TRUE;
      all_roots = APPEND(itm,all_roots);
      FORM_proj_register(itm);
    }
   else {
      if (!proj->is_project) {
	 proj->is_project = TRUE;
	 FORM_proj_register(proj);
       };
      proj->items = APPEND(itm,proj->items);
    };

   return itm;
};






/************************************************************************/
/*									*/
/*	FORM_item_free -- free an item					*/
/*	FORM_item_free_contents -- free the contents of an item 	*/
/*	FORM_item_free_attrs -- free attributes of an item		*/
/*									*/
/************************************************************************/


void
FORM_item_free(itm)
   FORM_ITEM itm;
{
   Sequence l;
   FORM_LINK fl;
   FORM_ATTR fa;
   FORM_ITEM sitm;

   FORM_item_wait_lock(itm,FALSE);

   FORM_item_free_contents(itm);
   FORM_item_free_attrs(itm);

   SFREE(itm->name);
   itm->name = NULL;

   while (itm->out_links != NULL) {
      FORM_link_free(CAR(FORM_LINK,itm->out_links));
    };

   while (itm->in_links != NULL) {
      FORM_link_free(CAR(FORM_LINK,itm->in_links));
    };

   if (itm->project != NULL) {
      itm->project->items = REMOB(itm,itm->project->items);
    };

   all_roots = REMOB(itm,all_roots);
   if (itm->is_project) FORM_proj_remove(itm);

   free(itm);
};





void
FORM_item_free_contents(itm)
   FORM_ITEM itm;
{
   Sequence l;
   FORM_ITEM sitm;

   FORM_item_wait_lock(itm,FALSE);

   forin (sitm,FORM_ITEM,l,itm->items) {
      FORM_item_free(sitm);
    };
   LFREE(itm->items);
   itm->items = NULL;
};





void
FORM_item_free_attrs(itm)
   FORM_ITEM itm;
{
   Sequence l;
   FORM_ATTR fa;

   FORM_item_wait_lock(itm,FALSE);

   forin (fa,FORM_ATTR,l,itm->attrs) {
      FORM_attr_free(fa);
    };
   LFREE(itm->attrs);
   itm->attrs = NULL;
};





/************************************************************************/
/*									*/
/*	FORM_item_attr_set -- add attribute to an item			*/
/*									*/
/************************************************************************/


void
FORM_item_attr_set(itm,name,value)
   FORM_ITEM itm;
   String name;
   FORM_ATTR_VALUE value;
{
   itm->attrs = list_attr_set(itm->attrs,name,value);
};





/************************************************************************/
/*									*/
/*	FORM_item_attr_get -- get attribute value for item		*/
/*									*/
/************************************************************************/


FORM_ATTR_VALUE
FORM_item_attr_get(itm,name)
   FORM_ITEM itm;
   String name;
{
   return list_attr_get(itm->attrs,name);
};





/************************************************************************/
/*									*/
/*	FORM_item_attr_test -- check if attribute defined for item	*/
/*									*/
/************************************************************************/


Boolean
FORM_item_attr_test(itm,name)
   FORM_ITEM itm;
   String name;
{
   return list_attr_test(itm->attrs,name);
};





/************************************************************************/
/*									*/
/*	FORM_item_wait_lock -- wait for item to be unlocked		*/
/*									*/
/************************************************************************/


void
FORM_item_wait_lock(itm,set)
   FORM_ITEM itm;
   Boolean set;
{
   while (itm->locked) {
      RIPidler(0);
    };

   itm->locked = set;
};





/************************************************************************/
/*									*/
/*	FORM_link_define -- add a link between two items		*/
/*									*/
/************************************************************************/


FORM_LINK
FORM_link_define(from,to)
   FORM_ITEM from;
   FORM_ITEM to;
{
   Sequence l;
   FORM_LINK fl;

   forin (fl,FORM_LINK,l,from->out_links) {
      if (fl->to == to) break;
    };

   if (fl == NULL) {
      fl = PALLOC(FORM_LINK_INFO);
      fl->from = from;
      fl->to = to;
      fl->attrs = NULL;
      fl->project = to->project;
      from->out_links = APPEND(fl,from->out_links);
      to->in_links = APPEND(fl,to->in_links);
    };

   return fl;
};





/************************************************************************/
/*									*/
/*	FORM_link_free -- remove link					*/
/*									*/
/************************************************************************/


void
FORM_link_free(lnk)
   FORM_LINK lnk;
{
   Sequence l;
   FORM_ATTR at;

   lnk->from->out_links = REMOB(lnk,lnk->from->out_links);
   lnk->to->in_links = REMOB(lnk,lnk->to->in_links);

   forin (at,FORM_ATTR,l,lnk->attrs) {
      FORM_attr_free(at);
    };
   LFREE(lnk->attrs);
   lnk->attrs = NULL;

   free(lnk);
};





/************************************************************************/
/*									*/
/*	FORM_link_attr_set -- add attribute to an link			*/
/*									*/
/************************************************************************/


void
FORM_link_attr_set(lnk,name,value)
   FORM_LINK lnk;
   String name;
   FORM_ATTR_VALUE value;
{
   lnk->attrs = list_attr_set(lnk->attrs,name,value);
};





/************************************************************************/
/*									*/
/*	FORM_link_attr_get -- get attribute value for link		*/
/*									*/
/************************************************************************/


FORM_ATTR_VALUE
FORM_link_attr_get(lnk,name)
   FORM_LINK lnk;
   String name;
{
   return list_attr_get(lnk->attrs,name);
};





/************************************************************************/
/*									*/
/*	FORM_link_attr_test -- check if attribute defined for link	*/
/*									*/
/************************************************************************/


Boolean
FORM_link_attr_test(lnk,name)
   FORM_LINK lnk;
   String name;
{
   return list_attr_test(lnk->attrs,name);
};





/************************************************************************/
/*									*/
/*	list_attr_set -- add/set attribute in attribute list		*/
/*									*/
/************************************************************************/


static Sequence
list_attr_set(lst,name,value)
   Sequence lst;
   String name;
   FORM_ATTR_VALUE value;
{
   FORM_ATTR_ID id;
   Sequence l;
   FORM_ATTR attr;

   id = FORM_attr_find_id(name);
   if (id == NULL) return lst;

   forin (attr,FORM_ATTR,l,lst) {
      if (attr->id == id) break;
    };

   if (attr == NULL) {
      attr = FORM_attr_define(id,value);
      lst = APPEND(attr,lst);
    };

   attr->value = value;

   return lst;
};





/************************************************************************/
/*									*/
/*	list_attr_get -- get attribute value from attr list		*/
/*									*/
/************************************************************************/


static FORM_ATTR_VALUE
list_attr_get(lst,name)
   Sequence lst;
   String name;
{
   FORM_ATTR_ID id;
   Sequence l;
   FORM_ATTR attr;

   id = FORM_attr_find_id(name);
   if (id == NULL) return FORM_attr_value_null();

   forin (attr,FORM_ATTR,l,lst) {
      if (attr->id == id) break;
    };

   if (attr == NULL) return id->dflt;

   return attr->value;
};





/************************************************************************/
/*									*/
/*	list_attr_test -- check if attribute defined in attr list	*/
/*									*/
/************************************************************************/


static Boolean
list_attr_test(lst,name)
   Sequence lst;
   String name;
{
   FORM_ATTR_ID id;
   FORM_ATTR attr;
   Sequence l;

   id = FORM_attr_find_id(name);
   if (id == NULL) return;

   forin (attr,FORM_ATTR,l,lst) {
      if (attr->id == id) break;
    };

   return (attr != NULL);
};





/* end of formitem.c */
