/************************************************************************/
/*									*/
/*		formback.c						*/
/*									*/
/*	Generic backend routines for FORM interface			*/
/*									*/
/************************************************************************/


#include "form_local.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/errno.h>
#include <sys/wait.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_FLAGS	64
#define MAX_ARGS	64
#define BUF_SIZE	1024





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence		all_backends;
static	FORM_BACKEND		default_backend;
static	Sequence		all_scans;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		load_backends();
static	FORM_BACKEND	find_backend();
static	void		form_scan_reader();
static	void		finish_scan();
static	void		form_exec_reader();
static	void		form_output_line();
static	void		finish_exec();






/************************************************************************/
/*									*/
/*	FORM_back_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FORM_back_init()
{
   all_backends = NULL;
   default_backend = NULL;
   all_scans = NULL;
};





/************************************************************************/
/*									*/
/*	FORM_back_load -- load resource definitions			*/
/*									*/
/************************************************************************/


void
FORM_back_load()
{
   load_backends();
};





/************************************************************************/
/*									*/
/*	FORM_back_scan -- scan for project or item			*/
/*									*/
/************************************************************************/


FORM_ITEM
FORM_back_scan(itm)
   FORM_ITEM itm;
{
   FORM_ITEM proj;
   String dir;
   FORM_BACKEND fb;
   Integer pid;
   Integer op[2];
   Integer nul;
   Boolean okfg;
   fd_set rmask;
   struct stat sinfo;
   FORM_SCAN fs;
   String s;
   FORM_ITEM rslt;

   if (itm->is_project) {
      proj = itm;
      itm = NULL;
    }
   else {
      proj = itm->project;
    };

   FORM_item_wait_lock(proj,TRUE);
   okfg = TRUE;

   s = FORM_item_attr_get(proj,"BACKEND").string_value;
   fb = find_backend(s);
   if (fb == NULL) okfg = FALSE;

   dir = FORM_item_attr_get(proj,"PATHNAME").string_value;
   if (dir == NULL) dir = proj->name;
   if (access(dir,5) < 0 || stat(dir,&sinfo) < 0 || (sinfo.st_mode & S_IFDIR) == 0) {
      FORM_error("Can't access directory %s",dir);
      okfg = FALSE;
    };

   if (okfg) {
      fprintf(stderr,"FORM:  Scanning for %s of %s\n",
		 (itm == NULL ? "<DEFAULT>" : itm->name), dir);

      pipe(op);

      pid = vfork();

      if (pid < 0) {
	 FORM_error("Can't fork to run backend");
	 okfg = FALSE;
	 close(op[0]);
	 close(op[1]);
       }
      else if (pid == 0) {
	 nul = open("/dev/null",0);
	 dup2(nul,0);
	 dup2(op[1],1);
	 close(op[0]);
	 (*fb->exec_scan_rtn)(fb,dir,proj,itm);
	 okfg = FALSE;
	 _exit(1);
       };
    };

   if (okfg) {
      fs = PALLOC(FORM_SCAN_INFO);
      fs->project = proj;
      fs->item = itm;
      fs->backend = fb;
      fs->pid = pid;
      fs->pipe = op[0];
      fs->data = NULL;
      fs->xtra[0] = 0;
      fs->done = FALSE;
      fs->rid = -1;
      fs->outfile = NULL;
      fs->outname = NULL;
      (*fb->scan_rtn)(fs,NULL);
      all_scans = CONS(fs,all_scans);

      FD_ZERO(&rmask);
      FD_SET(op[0],&rmask);
      CMPXregister_add(op[0]+1,&rmask,NULL,NULL,form_scan_reader);
      close(op[1]);
      close(nul);

      while (!fs->done) {
	 RIPidler(0);
       };

      rslt = fs->item;
      free(fs);
    }
   else {
      rslt = NULL;
    };

   proj->locked = FALSE;

   return rslt;
};





/************************************************************************/
/*									*/
/*	FORM_back_exec_start -- start execution for item		*/
/*									*/
/************************************************************************/


Boolean
FORM_back_exec_start(itm,rid)
   FORM_ITEM itm;
   Integer rid;
{
   FORM_ITEM proj;
   String dir;
   FORM_BACKEND fb;
   Integer pid;
   Integer op[2];
   Integer nul;
   Boolean okfg;
   fd_set rmask;
   struct stat sinfo;
   FORM_SCAN fs;
   Character buf[256];

   if (itm->is_project) {
      proj = itm;
      itm = NULL;
    }
   else {
      proj = itm->project;
    };

   okfg = TRUE;
   FORM_item_wait_lock(proj,TRUE);

   fb = find_backend(FORM_item_attr_get(proj,"BACKEND"));
   if (fb == NULL) okfg = FALSE;

   dir = FORM_item_attr_get(proj,"PATHNAME").string_value;
   if (dir == NULL) dir = proj->name;
   if (access(dir,5) < 0 || stat(dir,&sinfo) < 0 || (sinfo.st_mode & S_IFDIR) == 0) {
      FORM_error("Can't access directory %s",dir);
      okfg = FALSE;
    };

   if (okfg) {
      fprintf(stderr,"FORM:  Making %s of %s\n",
		 (itm == NULL ? "<DEFAULT>" : itm->name), dir);

      pipe(op);

      pid = vfork();

      if (pid < 0) {
	 FORM_error("Can't fork to run backend");
	 close(op[0]);
	 close(op[1]);
	 okfg = FALSE;
       }
      else if (pid == 0) {
	 nul = open("/dev/null",0);
	 dup2(nul,0);
	 dup2(op[1],1);
	 dup2(1,2);
	 close(op[0]);
	 (*fb->exec_build_rtn)(fb,dir,proj,itm);
	 okfg = FALSE;
	 _exit(1);
       };
    };

   if (okfg) {
      fs = PALLOC(FORM_SCAN_INFO);
      fs->project = proj;
      fs->item = itm;
      fs->backend = fb;
      fs->pid = pid;
      fs->pipe = op[0];
      fs->data = FORM_msg_start_output();
      fs->xtra[0] = 0;
      fs->rid = rid;
      if (rid < 0) fs->outfile = NULL;
      else {
	 sprintf(buf,"/tmp/formoutXXXXXX");
	 mktemp(buf);
	 fs->outfile = fopen(buf,"w");
	 if (fs->outfile != NULL) fs->outname = SALLOC(buf);
       };
      all_scans = CONS(fs,all_scans);
      FORM_win_output(fs->project,NULL);

      FD_ZERO(&rmask);
      FD_SET(op[0],&rmask);
      CMPXregister_add(op[0]+1,&rmask,NULL,NULL,form_exec_reader);
      close(op[1]);
    };

   if (!okfg) {
      proj->locked = FALSE;
      if (rid >= 0) MSGreply(rid,NULL);
    };

   return okfg;
};





/************************************************************************/
/*									*/
/*	load_backends -- get defined backends from AUXD file		*/
/*									*/
/************************************************************************/


static void
load_backends()
{
   AUXD hdl,db;
   String nm,s;
   FORM_BACKEND fb;
   Character buf[1024];
   String flgs[MAX_FLAGS];
   Integer i,ct;
   Function_Ptr irtn;

   hdl = AUXDget_handle(NULL,"FORM");

   for (db = AUXDget_handle(hdl,"BACKEND"); db != NULL; db = AUXDnext_handle(db)) {
      nm = AUXDget_info(db,"NAME");
      if (nm == NULL) continue;
      fb = PALLOC(FORM_BACKEND_INFO);
      fb->name = nm;

      s = AUXDget_info(db,"INIT");
      if (s == NULL) continue;
      irtn = (Function_Ptr) DLlookup(s);
      if (irtn != NULL) (*irtn)(db);

      s = AUXDget_info(db,"EXEC_SCAN");
      if (s == NULL) continue;
      fb->exec_scan_rtn = (Function_Ptr) DLlookup(s);
      if (fb->exec_scan_rtn == NULL) continue;

      s = AUXDget_info(db,"SCAN");
      if (s == NULL) continue;
      fb->scan_rtn = (Function_Ptr) DLlookup(s);
      if (fb->scan_rtn == NULL) continue;

      s = AUXDget_info(db,"EXEC_BUILD");
      if (s == NULL) continue;
      fb->exec_build_rtn = (Function_Ptr) DLlookup(s);
      if (fb->exec_build_rtn == NULL) continue;

      fb->global_attrs = NULL;
      fb->item_attrs = NULL;
      fb->link_attrs = NULL;

      all_backends = APPEND(fb,all_backends);
    };

   s = AUXDget_info(hdl,"DEFAULT_BACKEND");
   default_backend = find_backend(s);
};






/************************************************************************/
/*									*/
/*	find_backend -- find a backend given a name			*/
/*									*/
/************************************************************************/


static FORM_BACKEND
find_backend(nm)
   String nm;
{
   Sequence l;
   FORM_BACKEND fb;

   if (nm == NULL || *nm == 0) return default_backend;

   forin (fb,FORM_BACKEND,l,all_backends) {
      if (STREQL(nm,fb->name)) return fb;
    };

   FORM_error("Can't locate backend %s",nm);

   return default_backend;
};





/************************************************************************/
/*									*/
/*	form_scan_reader -- read data from scan output			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
form_scan_reader(rmsk,wmsk,emsk)
   fd_set *rmsk;
   fd_set *wmsk;
   fd_set *emsk;
{
   Sequence l,la;
   FORM_SCAN fs;
   Character buf[BUF_SIZE+1];
   Integer ln,i;
   String s;

   forin (fs,FORM_SCAN,l,all_scans) {
      if (fs->pipe < 0 || !FD_ISSET(fs->pipe,rmsk))
	 continue;

      buf[0] = 0;

      ln = 0;
      while (fs->xtra[ln] != 0) {
	 buf[ln] = fs->xtra[ln];
	 ++ln;
       };
      i = read(fs->pipe,&buf[ln],BUF_SIZE-ln);
      if (i <= 0) {
	 finish_scan(fs);
	 break; 		/* l might not be valid anymore */
       }
      else {
	 buf[ln+i] = 0;
	 s = buf;

	 for ( ; ; ) {
	    ln = 0;
	    while (*s != 0 && *s != '\n') fs->xtra[ln++] = *s++;
	    fs->xtra[ln] = 0;
	    if (*s == 0) break;
	    else ++s;
	    (*fs->backend->scan_rtn)(fs,fs->xtra);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	finish_scan -- finish up scanning when backend exits		*/
/*									*/
/************************************************************************/


static void
finish_scan(fs)
   FORM_SCAN fs;
{
   all_scans = REMOB(fs,all_scans);

   CMPXclose(fs->pipe);
   close(fs->pipe);
   fs->pipe = -1;
   fs->pid = 0;
   (*fs->backend->scan_rtn)(fs,NULL);

   fs->done = TRUE;
};






/************************************************************************/
/*									*/
/*	form_exec_reader -- read data from exec output			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
form_exec_reader(rmsk,wmsk,emsk)
   fd_set *rmsk;
   fd_set *wmsk;
   fd_set *emsk;
{
   Sequence l,la;
   FORM_SCAN fs;
   Character buf[BUF_SIZE+1];
   Integer ln,i;
   String s;

   forin (fs,FORM_SCAN,l,all_scans) {
      if (fs->pipe < 0 || !FD_ISSET(fs->pipe,rmsk))
	 continue;

      buf[0] = 0;

      ln = 0;
      while (fs->xtra[ln] != 0) {
	 buf[ln] = fs->xtra[ln];
	 ++ln;
       };
      i = read(fs->pipe,&buf[ln],BUF_SIZE-ln);
      if (i <= 0) {
	 finish_exec(fs);
	 break; 		/* l might not be valid anymore */
       }
      else {
	 buf[ln+i] = 0;
	 s = buf;

	 for ( ; ; ) {
	    ln = 0;
	    while (*s != 0 && *s != '\n') fs->xtra[ln++] = *s++;
	    fs->xtra[ln] = 0;
	    if (*s == 0) break;
	    else ++s;
	    form_output_line(fs,fs->xtra);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	form_output_line -- handle one line of output from backend	*/
/*									*/
/************************************************************************/


static void
form_output_line(fs,text)
   FORM_SCAN fs;
   String text;
{
   Character buf[10240];

   sprintf(buf,"%s\n",text);
   fprintf(stderr,"%s",buf);

   FORM_win_output(fs->project,buf);	/* output to transcripts	*/

   FORM_msg_check_output(fs->data,buf);

   if (fs->outfile != NULL) {
      fprintf(fs->outfile,"%s",buf);
    };
};






/************************************************************************/
/*									*/
/*	finish_exec -- finish up executing				*/
/*									*/
/************************************************************************/


static void
finish_exec(fs)
   FORM_SCAN fs;
{
   int child, pid, status;
   String s,t;

   FORM_win_output(fs->project,END_LINE);

   all_scans = REMOB(fs,all_scans);

   pid = fs->pid;
   if (pid != 0) kill(pid,9);

   CMPXclose(fs->pipe);
   close(fs->pipe);
   fs->pipe = -1;
   fs->pid = 0;

   if (fs->outfile != NULL) fclose(fs->outfile);

   if (fs->rid >= 0) {
      MSGreply(fs->rid,fs->outname);
    };

   status = 0;
   if (pid != 0) {
      while ((child = wait3(&status,WNOHANG,0)) > 0) {
	 if (child == pid) break;
	 status = 0;
       };
      if (child <= 0) status = EDTexec_inq_status(pid);
    };

   s = (fs->outname == NULL ? "*" : fs->outname);
   if (fs->item != NULL) t = fs->item->name;
   else {
      t = FORM_item_attr_get(fs->project,"DEFAULT").string_value;
      if (t == NULL) t = "*";
    };

   MSGsenda("BUILD FINISH %d %s %s",status,s,t);

   if (fs->outname != NULL) SFREE(fs->outname);

   fs->project->locked = FALSE;

   free(fs);
};






/* end of formback.c */
