/************************************************************************/
/*									*/
/*		formattr.c						*/
/*									*/
/*	Attribute management routines for FORM package			*/
/*									*/
/************************************************************************/


#include "form_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define INIT_HASH_SIZE		32




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	HASH_TABLE		attr_hash;
static	Sequence		all_types;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	FORM_attr_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FORM_attr_init()
{
   attr_hash = HASHcreate(INIT_HASH_SIZE);
   all_types = NULL;
};





/************************************************************************/
/*									*/
/*	FORM_attr_define_type -- define new type			*/
/*									*/
/************************************************************************/


FORM_ATTR_TYPE
FORM_attr_define_type(name,cls,values,minv,maxv)
   String name;
   FORM_ATTR_CLASS cls;
   Sequence values;
   Float minv,maxv;
{
   FORM_ATTR_TYPE at;
   Sequence l,la;
   String s;
   Boolean fg;

   at = FORM_attr_find_type(name);

   if (at != NULL) {
      if (at->aclass != cls) fg = FALSE;
      else {
	 fg = TRUE;
	 switch (cls) {
	    case FORM_ATTR_CLASS_INT :
	    case FORM_ATTR_CLASS_FLOAT :
	       if (minv != at->min_value || maxv != at->max_value) fg = FALSE;
	       break;
	    case FORM_ATTR_CLASS_ENUM :
	       if (LENGTH(at->enum_values) != LENGTH(values)) fg = FALSE;
	       la = values;
	       forin (s,String,l,at->enum_values) {
		  if (STRNEQ(s,CAR(String,la))) fg = FALSE;
		  la = CDR(la);
		};
	       break;
	    default :
	       break;
	  };
       };
      if (!fg) {
	 FORM_error("Different attribute types named %s",name);
       };
    }
   else {
      at = PALLOC(FORM_ATTR_TYPE_INFO);
      at->name = SALLOC(name);
      at->aclass = cls;
      at->min_value = 0;
      at->max_value = 0;
      at->enum_values = NULL;
      all_types = APPEND(at,all_types);

      switch (cls) {
	 case FORM_ATTR_CLASS_INT :
	 case FORM_ATTR_CLASS_FLOAT :
	    at->min_value = minv;
	    at->max_value = maxv;
	    break;

	 case FORM_ATTR_CLASS_ENUM :
	    at->enum_values = values;
	    break;
       };
    };

   return at;
};





/************************************************************************/
/*									*/
/*	FORM_attr_find_type -- look up type by name			*/
/*									*/
/************************************************************************/


FORM_ATTR_TYPE
FORM_attr_find_type(name)
   String name;
{
   Sequence l;
   FORM_ATTR_TYPE at;

   forin (at,FORM_ATTR_TYPE,l,all_types) {
      if (STREQL(at->name,name)) break;
    };

   return at;
};





/************************************************************************/
/*									*/
/*	FORM_attr_define_id -- define an attribute class		*/
/*									*/
/************************************************************************/


FORM_ATTR_ID
FORM_attr_define_id(name,type,dflt,ro)
   String name;
   String type;
   FORM_ATTR_VALUE dflt;
   Boolean ro;
{
   ENTRY ent;
   ENTRY *fnd;
   FORM_ATTR_ID ai;
   FORM_ATTR_TYPE typ;

   typ = FORM_attr_find_type(type);
   if (typ == NULL) {
      FORM_error("Attribute %s has undefined type %s",name,type);
      return NULL;
    };

   ent.key = name;
   ent.data = NULL;

   fnd = HASHsearch(attr_hash,ent,ENTER);

   if (fnd->data != NULL) {
      ai = (FORM_ATTR_ID) fnd->data;
      if (ai->type != typ || ai->readonly != ro) {
	 FORM_error("Attempt to redefine attribute %s",name);
	 ai = NULL;
       };

    }
   else {
      ai = PALLOC(FORM_ATTR_ID_INFO);
      fnd->data = (char *) ai;
      ai->name = SALLOC(name);
      ai->type = typ;
      ai->dflt = dflt;
      ai->readonly = ro;
    };

   return ai;
};





/************************************************************************/
/*									*/
/*	FORM_attr_find_id -- find attribute id given name		*/
/*									*/
/************************************************************************/


FORM_ATTR_ID
FORM_attr_find_id(name)
   String name;
{
   ENTRY ent;
   ENTRY *fnd;

   ent.key = name;
   ent.data = NULL;

   fnd = HASHsearch(attr_hash,ent,FIND);

   if (fnd == NULL || fnd->data == NULL) {
      FORM_error("Attempt to use undefined attribute %s",name);
      return NULL;
    };

   return (FORM_ATTR_ID) fnd->data;
};





/************************************************************************/
/*									*/
/*	FORM_attr_define -- define an attribute 			*/
/*									*/
/************************************************************************/


FORM_ATTR
FORM_attr_define(id,value)
   FORM_ATTR_ID id;
   FORM_ATTR_VALUE value;
{
   FORM_ATTR at;

   at = PALLOC(FORM_ATTR_INFO);
   at->id = id;
   at->value = value;

   return at;
};




/************************************************************************/
/*									*/
/*	FORM_attr_free -- free storage for an attribute 		*/
/*									*/
/************************************************************************/


void
FORM_attr_free(at)
   FORM_ATTR at;
{
   free(at);
};





/************************************************************************/
/*									*/
/*	FORM_attr_value_boolean -- create boolean value 		*/
/*	FORM_attr_value_int -- create int value 			*/
/*	FORM_attr_value_string -- create string value			*/
/*	FORM_attr_value_enum -- create enum value			*/
/*	FORM_attr_value_float -- create float value			*/
/*									*/
/************************************************************************/


FORM_ATTR_VALUE
FORM_attr_value_boolean(b)
   Boolean b;
{
   FORM_ATTR_VALUE av;

   av.bool_value = b;

   return av;
};







FORM_ATTR_VALUE
FORM_attr_value_int(i)
   Integer i;
{
   FORM_ATTR_VALUE av;

   av.int_value = i;

   return av;
};







FORM_ATTR_VALUE
FORM_attr_value_string(s)
   String s;
{
   FORM_ATTR_VALUE av;

   av.string_value = SALLOC(s);

   return av;
};







FORM_ATTR_VALUE
FORM_attr_value_enum(ty,nm)
   FORM_ATTR_TYPE ty;
   String nm;
{
   FORM_ATTR_VALUE av;
   Sequence l;
   String s;
   Integer i;

   i = 1;
   forin (s,String,l,ty->enum_values) {
      if (STREQL(s,nm)) break;
      ++i;
    };

   if (s == NULL) i = 0;

   av.enum_value = i;

   return av;
};







FORM_ATTR_VALUE
FORM_attr_value_float(f)
   Float f;
{
   FORM_ATTR_VALUE av;

   av.float_value = f;

   return av;
};







FORM_ATTR_VALUE
FORM_attr_value_null()
{
   FORM_ATTR_VALUE av;

   bzero(&av,sizeof(FORM_ATTR_VALUE));

   return av;
};





/* end of formattr.c */
