/************************************************************************/
/*									*/
/*		form_local.h						*/
/*									*/
/*	Data structures for FORM front end for config management	*/
/*									*/
/************************************************************************/


#define PROT_NAME	FORM__sema

#include <datatypes.h>
#include <sequence.h>
#include <hashsearch.h>

#include <ash.h>
#include <rip.h>
#include <leaf.h>
#include <stem.h>
#include <wind.h>
#include <edt.h>

#include <auxd.h>
#include <gelo.h>
#include <pmat.h>
#include <msg.h>
#include <field.h>

#include <form.h>
#include <xref.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define END_LINE	"--------------------------------------------------\n"






/************************************************************************/
/*									*/
/*	Forward Data Type Definitions					*/
/*									*/
/************************************************************************/


typedef struct _FORM_ATTR *		FORM_ATTR;

typedef struct _FORM_ITEM *		FORM_ITEM;

typedef struct _FORM_LINK *		FORM_LINK;

typedef struct _FORM_BACKEND *		FORM_BACKEND;

typedef struct _FORM_SCAN *		FORM_SCAN;

typedef struct _FORM_WIN *		FORM_WIN;






/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Attributes			*/
/*									*/
/************************************************************************/


typedef struct _FORM_ATTR_TYPE *	FORM_ATTR_TYPE;
typedef struct _FORM_ATTR_ID *		FORM_ATTR_ID;




typedef enum _FORM_ATTR_CLASS {
   FORM_ATTR_CLASS_BOOLEAN,
   FORM_ATTR_CLASS_INT,
   FORM_ATTR_CLASS_STRING,
   FORM_ATTR_CLASS_ENUM,
   FORM_ATTR_CLASS_FLOAT,
} FORM_ATTR_CLASS;





typedef union _FORM_ATTR_VALUE {	/* Value holder for attribute	*/
   Boolean bool_value;			/*    FORM_ATTR_CLASS_BOOLEAN	*/
   Integer int_value;			/*    FORM_ATTR_CLASS_INT	*/
   String string_value; 		/*    FORM_ATTR_CLASS_STRING	*/
   Integer enum_value;			/*    FORM_ATTR_CLASS_ENUM	*/
   Float float_value;			/*    FORM_ATTR_CLASS_FLOAT	*/
} FORM_ATTR_VALUE;




typedef struct _FORM_ATTR_TYPE {	/* type information for attribute */
   String name; 			/*   type name (handle) 	*/
   FORM_ATTR_CLASS aclass;		/*   class of data		*/
   Float min_value;			/*   min for int/float data	*/
   Float max_value;			/*   max for int/float data	*/
   Sequence enum_values;		/*   list of enum names 	*/
} FORM_ATTR_TYPE_INFO;




typedef struct _FORM_ATTR_ID {		/* Attribute identifier 	*/
   String name; 			/*    attribute handle (name)	*/
   FORM_ATTR_TYPE type; 		/*    type of data		*/
   FORM_ATTR_VALUE dflt;		/*    default value		*/
   Boolean readonly;			/*    not user-setable		*/
} FORM_ATTR_ID_INFO;




typedef struct _FORM_ATTR {		/* Attribute instance		*/
   FORM_ATTR_ID id;			/*    Attribute name		*/
   FORM_ATTR_VALUE value;		/*    Actual value		*/
} FORM_ATTR_INFO;





/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Items (Objects/Projects)		*/
/*									*/
/************************************************************************/


typedef struct _FORM_ITEM {		/* Basic item description	*/
   FORM_ITEM project;			/*    Containing project	*/
   Sequence attrs;			/*    Attributes of this item	*/
   String name; 			/*    Name (handle)		*/
   Sequence in_links;			/*    Links to this item	*/
   Sequence out_links;			/*    Links from this item	*/
   Sequence items;			/*    Internal items (for proj) */
   Boolean is_project;			/*    Project/Object flag	*/
   Boolean locked;
} FORM_ITEM_INFO;





/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Links				*/
/*									*/
/************************************************************************/


typedef struct _FORM_LINK {		/* Link between items		*/
   FORM_ITEM project;			/*    Containing project	*/
   Sequence attrs;			/*    Attributes of link	*/
   FORM_ITEM from;			/*    From this item		*/
   FORM_ITEM to;			/*    To this item		*/
} FORM_LINK_INFO;





/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Back End Systems			*/
/*									*/
/************************************************************************/


typedef struct _FORM_BACKEND {		/* backend interface definition */
   String name; 			/*    name of backend		*/
   Function_Ptr exec_scan_rtn;		/*    Exec entry for scanning	*/
   Function_Ptr scan_rtn;		/*    Line entry for scanning	*/
   Function_Ptr exec_build_rtn; 	/*    Exec entry for building	*/
   Sequence global_attrs;		/*    Global attributes 	*/
   Sequence item_attrs; 		/*    Item attributes		*/
   Sequence link_attrs; 		/*    Link attributes		*/
} FORM_BACKEND_INFO;






/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Scanning Information		*/
/*									*/
/************************************************************************/


typedef struct _FORM_SCAN {		/* info for use while scanning	*/
   FORM_ITEM project;
   FORM_ITEM item;
   FORM_BACKEND backend;
   Integer pid;
   Integer pipe;
   Universal data;
   Integer rid;
   Boolean done;
   FILE * outfile;
   String outname;
   Character xtra[1024];
} FORM_SCAN_INFO;





/************************************************************************/
/*									*/
/*	Data Structure Definitions -- Windows				*/
/*									*/
/************************************************************************/


typedef Integer FORM_HILITE_STYLE;
#define FORM_HILITE_NONE	0
#define FORM_HILITE_SELECT	1
#define FORM_HILITE_OUTOFDATE	2
#define FORM_HILITE_CHANGED	3

#define FORM_HILITE_STYLE_COUNT 4



typedef struct _FORM_WIN {
   ASH_WINDOW window;
   ASH_WINDOW menu_win;
   ASH_WINDOW disp_win;
   ASH_WINDOW text_win;
   String project;
   FORM_ITEM root;
   FORM_ITEM selection;
   FORM_ITEM topdisp;
   FORM_ITEM dflt;
   Boolean display_all;
   Boolean display_force;
   Boolean display_fixcur;
   Boolean display_inlinks;
   Boolean display_outlinks;
   Integer display_levels;
   Integer display_zoom;
   GELO_OBJECT gid;
   GELO_METHOD method;
   GELO_CONN_METHOD connmethod;
   Boolean fixed;
   Boolean standard;
   Boolean centered;
   Integer whitespace;
   RIP_REGION region;
   EDT_ID edit_info;
   HASH_TABLE nodes;
   Sequence hilite[FORM_HILITE_STYLE_COUNT];
   Sequence ignore;
   Sequence dispproj;
   Sequence dispitem;
   Boolean useregex;
   String last_trans;
   Integer trans_max;
   Integer trans_cnt;
   String exec_btn;
} FORM_WIN_INFO;




/************************************************************************/
/*									*/
/*	Entry Points							*/
/*									*/
/************************************************************************/


extern	void			FORM_error();



extern	void			FORM_win_init();
extern	FORM_WIN		FORM_create();
extern	FORM_WIN		FORM_create_trans();
extern	FORM_WIN		FORM_find_window();
extern	FORM_WIN		FORM_find_project();
extern	void			FORM_win_update();
extern	void			FORM_win_output();



extern	void			FORM_menu_init();
extern	void			FORM_menu_setup();
extern	String			FORM_request_project();



extern	void			FORM_trans_init();
extern	void			FORM_trans_setup();
extern	void			FORM_trans_output();



extern	void			FORM_disp_init();
extern	void			FORM_disp_setup();
extern	void			FORM_disp_reset();
extern	void			FORM_disp_show();
extern	void			FORM_disp_free();
extern	void			FORM_disp_select();
extern	void			FORM_disp_ignore();
extern	void			FORM_disp_unignore();
extern	void			FORM_disp_regex();



extern	void			FORM_attr_init();
extern	FORM_ATTR_TYPE		FORM_attr_define_type();
extern	FORM_ATTR_TYPE		FORM_attr_find_type();
extern	FORM_ATTR_ID		FORM_attr_define_id();
extern	FORM_ATTR_ID		FORM_attr_find_id();
extern	FORM_ATTR		FORM_attr_define();
extern	void			FORM_attr_free();
extern	FORM_ATTR_VALUE 	FORM_attr_value_boolean();
extern	FORM_ATTR_VALUE 	FORM_attr_value_int();
extern	FORM_ATTR_VALUE 	FORM_attr_value_string();
extern	FORM_ATTR_VALUE 	FORM_attr_value_enum();
extern	FORM_ATTR_VALUE 	FORM_attr_value_float();
extern	FORM_ATTR_VALUE 	FORM_attr_value_null();




extern	void			FORM_item_init();
extern	FORM_ITEM		FORM_item_define();
extern	void			FORM_item_free();
extern	void			FORM_item_free_contents();
extern	void			FORM_item_free_attrs();
extern	void			FORM_item_attr_set();
extern	FORM_ATTR_VALUE 	FORM_item_attr_get();
extern	Boolean 		FORM_item_attr_check();
extern	void			FORM_item_wait_lock();
extern	FORM_LINK		FORM_link_define();
extern	void			FORM_link_free();
extern	void			FORM_link_attr_set();
extern	FORM_ATTR_VALUE 	FORM_link_attr_get();
extern	Boolean 		FORM_link_attr_check();



extern	void			FORM_proj_init();
extern	void			FORM_proj_setup();
extern	void			FORM_proj_update();
extern	FORM_ITEM		FORM_proj_find();
extern	FORM_ITEM		FORM_proj_find_file();
extern	Boolean 		FORM_proj_execute();
extern	void			FORM_proj_register();
extern	void			FORM_proj_remove();
extern	Integer 		FORM_proj_inq_names();



extern	void			FORM_setup_init();
extern	void			FORM_setup_load();



extern	void			FORM_back_init();
extern	void			FORM_back_load();
extern	FORM_ITEM		FORM_back_scan();
extern	Boolean 		FORM_back_exec_start();



extern	void			FORM_msg_init();
extern	void			FORM_msg_start();
extern	Universal		FORM_msg_start_output();
extern	void			FORM_msg_end_output();
extern	void			FORM_msg_check_output();
extern	void			FORM_msg_check_trans();





/* end of form_local.h */
