/************************************************************************/
/*									*/
/*		flownode.c						*/
/*									*/
/*	Routines for managing flow nodes for a system			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "flow_local.h"




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		node_filename();
static	void		count_arcs();
static	void		clear_conn();
static	Boolean 	check_conn();
static	FLOW_NODE	file_node();
static	FLOW_NODE	add_node();
static	void		merge_node();
static	void		add_connect();
static	FLOW_NODE	find_funct();
static	void		free_flow_node();
static	void		fix_disp_name();
static	void		enter_hash();




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Boolean 	ignore_case;





/************************************************************************/
/*									*/
/*	FLOW_node_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FLOW_node_init()
{
   ignore_case = FALSE;
   if (getenv("STDPASCAL")) ignore_case = TRUE;
};





/************************************************************************/
/*									*/
/*	FLOW_node_setup -- setup names for a new system 		*/
/*									*/
/************************************************************************/


void
FLOW_node_setup(fw)
   FLOW_WIN fw;
{
   String fcts,calls;
   String lin,fun,fil,frm,lno;
   FLOW_NODE fn;
   Integer ct;
   FILE * inf;
   Character lbuf[10240];

   FLOW_disp_free(fw);

   if (fw->root != NULL) {
      fn = fw->root;
      fw->root = NULL;
      free_flow_node(fn);
      if (fw->hashtable != NULL) HASHdestroy(fw->hashtable);
      fw->hashtable = NULL;
    };

   fw->root = NULL;
   fw->numconn = 0;
   fw->maxconn = 0;
   fw->conns = NULL;
   fw->main = NULL;
   fw->execute = NULL;
   fw->main = NULL;
   fw->maindisp = NULL;
   fw->callstack = NULL;
   fw->conn_valid = FALSE;
   fw->numnodes = 0;

   if (fw->selection != NULL) {
      FLOW_menu_set_selection(fw,NULL);
      fw->selection = NULL;
    };

   add_node(fw,NULL,"*ROOT*",FLOW_TYPE_ROOT);

   fcts = MSGcalla("XREF QUERY %s *F.name,F.file,F.line ",fw->system);
   inf = fopen(fcts,"r");
   if (inf == NULL) {
      fprintf(stderr,"FLOW: Problem with XREF\n");
      ASHremove(fw->window);
      return;
    };

   while (fgets(lbuf,10240,inf) != NULL) {
      fun = lbuf;
      fil = index(fun,'\177');
      if (fil == NULL) break;
      *fil++ = 0;
      lno = index(fil,'\177');
      if (lno == NULL) break;
      *lno++ = 0;
      lin = index(lno,'\177');
      if (lin == NULL) break;
      *lin++ = 0;
      if (*lin++ != '\n') break;

      fn = file_node(fw,fil,TRUE);
      fn = add_node(fw,fn,fun,FLOW_TYPE_FUNCTION);
      fn->line = atol(lno);
      if (STREQL(fun,"main") || STREQL(fun,"PROGRAM")) {
	 if (fw->main == NULL) fw->main = fn;
       };
    };

   fclose(inf);
   unlink(fcts);
   SFREE(fcts);

   merge_node(fw->root);

   fw->hashtable = HASHcreate(fw->numnodes*2);
   enter_hash(fw,fw->root);

   calls = MSGcalla("XREF QUERY %s *C.from,C.call,C.file,C.line ",fw->system);

   ct = 0;
   inf = fopen(calls,"r");
   while (fgets(lbuf,10240,inf) != NULL) ++ct;
   fclose(inf);

   fw->maxconn = ct+1;
   fw->conns = (FLOW_ARC) calloc(sizeof(FLOW_ARC_INFO),fw->maxconn);

   inf = fopen(calls,"r");
   while (fgets(lbuf,10240,inf) != NULL) {
      frm = lbuf;
      fun = index(frm,'\177');
      if (fun == NULL) break;
      *fun++ = 0;
      fil = index(fun,'\177');
      if (fil == NULL) break;
      *fil++ = 0;
      lno = index(fil,'\177');
      if (lno == NULL) break;
      *lno++ = 0;
      lin = index(lno,'\177');
      if (lin == NULL) break;
      *lin++ = 0;
      if (*lin++ != '\n') break;

      add_connect(fw,fil,frm,fun,atol(lno));
    };

   fclose(inf);
   unlink(calls);
   SFREE(calls);
};





/************************************************************************/
/*									*/
/*	FLOW_node_file -- get filename from flow node			*/
/*									*/
/************************************************************************/


void
FLOW_node_file(fn,buf)
   FLOW_NODE fn;
   String buf;
{
   buf[0] = 0;
   node_filename(fn,buf);
};





static void
node_filename(fn,buf)
   FLOW_NODE fn;
   String buf;
{
   if (fn->parent != NULL && fn->parent->type != FLOW_TYPE_ROOT) {
      node_filename(fn->parent,buf);
      if (fn->type != FLOW_TYPE_FUNCTION) strcat(buf,"/");
    };

   if (fn->type != FLOW_TYPE_FUNCTION) strcat(buf,fn->name);
};





/************************************************************************/
/*									*/
/*	FLOW_node_find_function -- find function given file and name	*/
/*									*/
/************************************************************************/


FLOW_NODE
FLOW_node_find_function(fw,file,fun)
   FLOW_WIN fw;
   String file;
   String fun;
{
   FLOW_NODE fn,fn0;

   if (file != NULL && STREQL(file,"*")) file = NULL;

   if (file != NULL) {
      fn0 = file_node(fw,file,FALSE);
      if (fn0 == NULL && file[0] == '/') return NULL;
    }
   else fn0 = NULL;

   if (fun == NULL) return fn0;

   if (fn0 == NULL) fn0 = fw->root;

   fn = find_funct(fw,fn0,fun,TRUE);

   return fn;
};





/************************************************************************/
/*									*/
/*	FLOW_node_count_arcs -- get count of in/out arcs		*/
/*									*/
/************************************************************************/


void
FLOW_node_count_arcs(fn,inp,outp)
   FLOW_NODE fn;
   Integer * inp;
   Integer * outp;
{
   *inp = 0;
   *outp = 0;

   count_arcs(fn,inp,outp);
};




static void
count_arcs(fn,inp,outp)
   FLOW_NODE fn;
   Integer * inp;
   Integer * outp;
{
   FLOW_NODE sf;

   *inp += fn->arcin;
   *outp += fn->arcout;

   for (sf = fn->son; sf != NULL; sf = sf->brother) count_arcs(sf,inp,outp);
};





/************************************************************************/
/*									*/
/*	FLOW_node_connected -- eliminate disconnected nodes		*/
/*									*/
/************************************************************************/


void
FLOW_node_connected(fw)
   FLOW_WIN fw;
{
   Boolean change;
   Integer i;

   clear_conn(fw->root);

   if (fw->main != NULL) fw->main->connected = TRUE;
   else return;

   for ( ; ; ) {
      change = FALSE;
      for (i = 0; i < fw->numconn; ++i) {
	 if (fw->conns[i].from->connected) {
	    if (!fw->conns[i].to->connected) {
	       fw->conns[i].to->connected = TRUE;
	       change = TRUE;
	     };
	  };
       };
      if (!change) break;
    };

   check_conn(fw->root);
};





static void
clear_conn(fn)
   FLOW_NODE fn;
{
   FLOW_NODE sf;

   fn->connected = FALSE;

   for (sf = fn->son; sf != NULL; sf = sf->brother) clear_conn(sf);
};





static Boolean
check_conn(fn)
   FLOW_NODE fn;
{
   FLOW_NODE sf;
   Boolean fg;

   if (fn->ignore) return FALSE;

   fg = fn->connected;

   for (sf = fn->son; sf != NULL; sf = sf->brother) fg |= check_conn(sf);

   fn->connected = fg;

   return fg;
};





/************************************************************************/
/*									*/
/*	file_node -- build/find a file node given name			*/
/*									*/
/************************************************************************/


static FLOW_NODE
file_node(fw,name,new)
   FLOW_WIN fw;
   String name;
   Boolean new;
{
   FLOW_NODE par,fn;
   String s;
   Integer i;

   par = fw->root;

   while (name != NULL) {
      for (fn = par->son; fn != NULL; fn = fn->brother) {
	 i = strlen(fn->name);
	 if (strncmp(name,fn->name,i) == 0 && (name[i] == 0 || name[i] == '/')) {
	    if (name[i] == '/') name = &name[i+1];
	    else name = NULL;
	    break;
	  };
       };
      if (fn != NULL) par = fn;
      else if (!new) return NULL;
      else {
	 s = name;
	 name = index(name,'/');
	 if (name != NULL) *name++ = 0;
	 par = add_node(fw,par,s,(name == NULL ? FLOW_TYPE_FILE : FLOW_TYPE_DIRECTORY));
       };
    };

   return par;
};





/************************************************************************/
/*									*/
/*	add_node -- build/find a node					*/
/*									*/
/************************************************************************/


static FLOW_NODE
add_node(fw,par,name,type)
   FLOW_WIN fw;
   FLOW_NODE par;
   String name;
   FLOW_NODE_TYPE type;
{
   FLOW_NODE fn,lfn;

   if (par == NULL && type != FLOW_TYPE_ROOT) par = fw->root;

   lfn = NULL;
   for (fn = (par == NULL ? fw->root : par->son); fn != NULL; fn = fn->brother) {
      lfn = fn;
      if (STREQL(name,fn->name)) break;
    };

   if (fn == NULL) {
      ++fw->numnodes;
      fn = PALLOC(FLOW_NODE_INFO);
      fn->type = type;
      fn->name = SALLOC(name);
      fn->matchname = NULL;
      fix_disp_name(fn);
      fn->parent = par;
      fn->son = NULL;
      fn->brother = NULL;
      fn->line = 0;
      fn->gid = NULL;
      fn->ignore = FALSE;
      fn->display = FALSE;
      fn->arcin = 0;
      fn->arcout = 0;
      fn->infoline = -1;
      fn->expand = (type == FLOW_TYPE_FUNCTION ? FALSE : TRUE);
      if (type == FLOW_TYPE_ROOT) {
	 fn->son = fw->root;
	 fw->root = fn;
       }
      else if (lfn == NULL) {
	 if (par == NULL) fw->root = fn;
	 else par->son = fn;
       }
      else lfn->brother = fn;
    };

   return fn;
};





/************************************************************************/
/*									*/
/*	merge_node -- get rid of extraneous levels			*/
/*									*/
/************************************************************************/


static void
merge_node(fn)
   FLOW_NODE fn;
{
   Character buf[1024];
   FLOW_NODE sf;

   for (sf = fn->son; sf != NULL; sf = sf->brother) {
      merge_node(sf);
    };

   sf = fn->son;
   if (sf != NULL && sf->brother == NULL && sf->type != FLOW_TYPE_FUNCTION &&
	  fn->type != FLOW_TYPE_ROOT) {
      sprintf(buf,"%s/%s",fn->name,sf->name);
      SFREE(fn->name);
      SFREE(sf->name);
      fn->name = SALLOC(buf);
      fn->dispname = fn->name;
      fn->son = sf->son;
      fn->type = sf->type;
      free(sf);
      for (sf = fn->son; sf != NULL; sf = sf->brother) sf->parent = fn;
    };
};






/************************************************************************/
/*									*/
/*	add_connect -- add a connection between functions		*/
/*									*/
/************************************************************************/


static void
add_connect(fw,fil,from,to,lin)
   FLOW_WIN fw;
   String fil;
   String from;
   String to;
   Integer lin;
{
   FLOW_NODE fn,ffn,tfn;

   fn = file_node(fw,fil,FALSE);
   if (fn == NULL) return;

   ffn = find_funct(fw,fn,from,TRUE);
   tfn = find_funct(fw,fn,to,FALSE);

   if (ffn == NULL || tfn == NULL) return;

   fw->conns[fw->numconn].from = ffn;
   fw->conns[fw->numconn].to = tfn;
   fw->conns[fw->numconn].line = lin;
   ++ffn->arcout;
   ++tfn->arcin;
   ++fw->numconn;
};






/************************************************************************/
/*									*/
/*	find_funct -- find function flow node given parent		*/
/*									*/
/************************************************************************/


static FLOW_NODE
find_funct(fw,fn0,name,mustfg)
   FLOW_WIN fw;
   FLOW_NODE fn0;
   String name;
   Boolean mustfg;
{
   FLOW_NODE sf,nfn;
   Character buf[1024];
   ENTRY item, *fnd;
   Boolean fg;
   String s,t;

   if (ignore_case) {
      fg = FALSE;
      t = buf;
      for (s = name; *s != 0; ++s) {
	 if (isupper(*s)) {
	    *t++ = tolower(*s);
	    fg = TRUE;
	  }
	 else *t++ = *s;
       };
      if (fg) name = buf;
    };

   item.key = name;
   item.data = NULL;

   fnd = HASHsearch(fw->hashtable,item,ENTER);

   for (sf = (FLOW_NODE) fnd->data; sf != NULL; sf = sf->alt) {
      for (nfn = sf->parent; nfn != NULL; nfn = nfn->parent) {
	 if (nfn == fn0) break;
       };
      if (nfn != NULL) break;
    };
   if (sf == NULL && !mustfg) {
      sf = (FLOW_NODE) fnd->data;
    };

   return sf;
};





/************************************************************************/
/*									*/
/*	free_flow_node -- free a node and its sons			*/
/*									*/
/************************************************************************/


static void
free_flow_node(fn)
   FLOW_NODE fn;
{
   FLOW_NODE fs;

   while (fn->son != NULL) free_flow_node(fn->son);

   if (fn->parent != NULL) {
      if (fn->parent->son == fn) fn->parent->son = fn->brother;
      else {
	 for (fs = fn->parent->son; fs != NULL; fs = fs->brother) {
	    if (fs->brother == fn) {
	       fs->brother = fn->brother;
	       break;
	     };
	  };
       };
    };

   if (fn->dispname != fn->name) SFREE(fn->dispname);
   SFREE(fn->name);

   free(fn);
};






/************************************************************************/
/*									*/
/*	fix_disp_name -- set name for display				*/
/*									*/
/************************************************************************/


static void
fix_disp_name(fn)
   FLOW_NODE fn;
{
   Character buf[10240];
   Boolean parfg;
   String s,p;

   fn->dispname = fn->name;

   parfg = FALSE;

   if (fn->type == FLOW_TYPE_FUNCTION) {
      p = buf;
      for (s = fn->name; *s != 0; ++s) {
	 if (!parfg && *s == '(') {
	    *p++ = '\n';
	    parfg = TRUE;
	  }
	 *p++ = *s;
	 if (*s == ':' && s[1] != ':' && s[1] != 0) *p++ = '\n';
       };

      *p = 0;
      if (STRNEQ(buf,fn->name)) {
	 fn->dispname = SALLOC(buf);
       };
    };
};




/************************************************************************/
/*									*/
/*	enter_hash -- enter all nodes into initial hash table		*/
/*									*/
/************************************************************************/


static void
enter_hash(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   ENTRY item;
   ENTRY * fnd;
   FLOW_NODE sf;
   Character buf[1024];
   String s,t;
   Boolean fg;

   if (fn->type == FLOW_TYPE_FUNCTION) {
      fn->matchname = fn->name;
      if (ignore_case) {
	 fg = FALSE;
	 t = buf;
	 for (s = fn->name; *s != 0; ++s) {
	    if (isupper(*s)) {
	       *t++ = tolower(*s);
	       fg = TRUE;
	     }
	    else *t++ = *s;
	  };
	 if (fg) fn->matchname = SALLOC(buf);
       };

      fn->alt = NULL;
      item.key = fn->matchname;
      item.data = (char *) fn;

      fnd = HASHsearch(fw->hashtable,item,ENTER);
      sf = (FLOW_NODE) fnd->data;
      if (sf != fn) {
	 fn->alt = sf->alt;
	 sf->alt = fn;
       };
    };

   for (sf = fn->son; sf != NULL; sf = sf->brother)
      enter_hash(fw,sf);
};





/* end of flownode.c */

