/************************************************************************/
/*									*/
/*		flowmenu.c						*/
/*									*/
/*	Menu and button routines for flow graphs			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "flow_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_INFO_ARC		32





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	int		flow_menu_flow();
static	int		flow_menu_node();
static	int		flow_menu_display();
static	int		flow_menu_info();
static	int		flow_menu_layout();
static	int		flow_menu_select();
static	int		flow_menu_regex();
static	int		flow_menu_infowin();
static	int		flow_btn_handler();
static	Boolean 	flow_arc_handler();
static	void		arc_info();
static	void		do_trace_display();
static	void		flow_reset_btns();
static	Integer 	get_selection_info();
static	int		node_compare();
static	void		display_node_info();
static	void		display_node();






/************************************************************************/
/*									*/
/*	Tables -- menu definitions					*/
/*									*/
/************************************************************************/


static STEM_PDM_DATA	flow_menus[] = {
   { STEM_PSTATE_MENU,	"Flow",         NULL },
      { STEM_PSTATE_BTN,   "Info Window",  flow_menu_infowin },
      { STEM_PSTATE_BTN,   "Restart",      flow_menu_flow },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX,   "Reset",        flow_menu_flow },
      { STEM_PSTATE_BTN,   "Update",       flow_menu_flow },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX,   "Trace",        flow_menu_flow },
      { STEM_PSTATE_BTN,   "Trace Display",flow_menu_flow },
      { STEM_PSTATE_BTN,   "Set System",   flow_menu_flow },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX,   "Quit",         flow_menu_flow },

   { STEM_PSTATE_MENU, "Selection",      NULL },
      { STEM_PSTATE_BTN,  "Set File",           flow_menu_select },
      { STEM_PSTATE_BTN,  "Set Function",       flow_menu_select },
      { STEM_PSTATE_BTN_DISABLE,  "Clear Selection",    flow_menu_select },
      { STEM_PSTATE_BTN,  "Set Selection",      flow_menu_select },
      { STEM_PSTATE_BTN,  "Select Pattern",     flow_menu_regex },

   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX,  "Display",      NULL },
      { STEM_PSTATE_BTN,   "Options",      flow_menu_display },
      { STEM_PSTATE_BTN,   "Show All",     flow_menu_display },
      { STEM_PSTATE_BTN,   "Connected",    flow_menu_display },
      { STEM_PSTATE_BTN,   "Show Callers", flow_menu_display },
      { STEM_PSTATE_BTN,   "Show Called",  flow_menu_display },
      { STEM_PSTATE_BTN,   "Expand All",   flow_menu_display },
      { STEM_PSTATE_BTN,   "Layout",       flow_menu_layout },

   { STEM_PSTATE_MENU_DISABLE|STEM_PSTATE_COMPLEX,  "Node",         NULL },
      { STEM_PSTATE_BTN,   "Show Item",    flow_menu_display },
      { STEM_PSTATE_BTN,   "Show Parent",  flow_menu_display },
      { STEM_PSTATE_BTN,   "Ignore",       flow_menu_node },
      { STEM_PSTATE_BTN,   "Expand",       flow_menu_node },
      { STEM_PSTATE_BTN,   "Compact",      flow_menu_node },
      { STEM_PSTATE_BTN,   "Show Info",    flow_menu_info },

   { STEM_PSTATE_END }
};






/************************************************************************/
/*									*/
/*	FLOW_menu_init -- module initialization 		       */
/*									*/
/************************************************************************/


void
FLOW_menu_init()
{
};





/************************************************************************/
/*									*/
/*	FLOW_menu_setup -- setup buttons				*/
/*									*/
/************************************************************************/


void
FLOW_menu_setup(fw)
   FLOW_WIN fw;
{
   if (fw->menu_win != NULL) {
      STEMpdm_define(fw->menu_win,fw,flow_menus);
    };

   fw->region = RIPdefine_region(fw->disp_win,0,0,0,0,RIP_ALL_CHARS,
				    RIP_BTN_ANY_EITHER,
				    flow_btn_handler,
				    ASH_SENSE_NO_CHANGE);
   RIPuse_local_window(fw->region);
   RIPset_data(fw->region,fw);

   fw->tracemode = FLOW_TRACE_NONE;

   FLOW_menu_setup_trace(fw);
   flow_reset_btns(fw);
};






/************************************************************************/
/*									*/
/*	FLOW_menu_set_selection -- set current selection		*/
/*									*/
/************************************************************************/


void
FLOW_menu_set_selection(fw,sel)
   FLOW_WIN fw;
   FLOW_NODE sel;
{
   if (fw->selection == NULL && sel != NULL) {
      STEMpdm_menu_enable(fw->menu_win,"Node",TRUE);
      STEMpdm_btn_enable(fw->menu_win,"Selection","Clear Selection",TRUE);
    }
   else if (fw->selection != NULL && sel == NULL) {
      STEMpdm_menu_enable(fw->menu_win,"Node",FALSE);
      STEMpdm_btn_enable(fw->menu_win,"Selection","Clear Selection",FALSE);
    };
};





/************************************************************************/
/*									*/
/*	FLOW_menu_setup_trace -- setup tracing for window		*/
/*									*/
/************************************************************************/


void
FLOW_menu_setup_trace(fw)
   FLOW_WIN fw;
{
   String s;

   STEMpdm_btn_select(fw->menu_win,"Flow","Trace Display",FALSE);
   STEMpdm_btn_select(fw->menu_win,"Flow","Trace",FALSE);
   MSGsenda("DDTR EVENT REMOVE %s * * 0 * * 0 CALL 0",fw->system);

   switch (fw->tracemode) {
      case FLOW_TRACE_ALL :
	 s = MSGcalla("DDTR EVENT ADD %s * * 0 * * 0 CALL 1",fw->system);
	 if (s != NULL) STEMpdm_btn_select(fw->menu_win,"Flow","Trace",TRUE);
	 else fw->tracemode = FLOW_TRACE_NONE;
	 break;

      case FLOW_TRACE_DISPLAY :
	 do_trace_display(fw,fw->root,FALSE);
	 STEMpdm_btn_select(fw->menu_win,"Flow","Trace Display",TRUE);
	 break;

      case FLOW_TRACE_NONE :
	 break;
    };
};






/************************************************************************/
/*									*/
/*	flow_menu_flow -- handle flow menu options			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_flow(fw,menu,btn)
   FLOW_WIN fw;
   String menu;
   String btn;
{
   Character cls[256];

   if (STREQL(btn,"Restart")) {
      FLOW_disp_reset(fw,TRUE);
    }
   else if (STREQL(btn,"Reset")) {
      FLOW_disp_reset(fw,FALSE);
    }
   else if (STREQL(btn,"Update")) {
      ASHinput_lock_makelong(fw->window);
      MSGcalla("XREF RELOAD %s",fw->system);
      FLOW_node_setup(fw);
      FLOW_disp_reset(fw,TRUE);
    }
   else if (STREQL(btn,"Reload")) {
      FLOW_node_setup(fw);
      FLOW_disp_reset(fw,TRUE);
    }
   else if (STREQL(btn,"Trace")) {
      if (fw->tracemode == FLOW_TRACE_ALL) fw->tracemode = FLOW_TRACE_NONE;
      else fw->tracemode = FLOW_TRACE_ALL;
      FLOW_menu_setup_trace(fw);
    }
   else if (STREQL(btn,"Trace Display")) {
      if (fw->tracemode == FLOW_TRACE_DISPLAY) fw->tracemode = FLOW_TRACE_NONE;
      else fw->tracemode = FLOW_TRACE_DISPLAY;
      FLOW_menu_setup_trace(fw);
    }
   else if (STREQL(btn,"Set System")) {
      if (fw->system == NULL) strcpy(cls,"");
      else strcpy(cls,fw->system);
      if (!FIELDsystem_request(fw->window,"Call Graph Viewer",cls)) return FALSE;
      if (cls[0] == 0 || STREQL(cls,"*")) return FALSE;
      fw->system = SALLOC(cls);
      ASHinput_lock_makelong(fw->window);
      sprintf(cls,"flowview: %s",cls);
      ASHset_window_name(fw->window,cls);
      FLOW_node_setup(fw);
      FLOW_disp_setup(fw);
    }
   else if (STREQL(btn,"Quit")) {
      ASHremove(fw->window);
      return TRUE;
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	flow_menu_node -- handle node menu options			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_node(fw,menu,btn)
   FLOW_WIN fw;
   String menu;
   String btn;
{
   if (fw->selection == NULL) return FALSE;

   if (STREQL(btn,"Ignore")) {
      FLOW_disp_ignore(fw,fw->selection);
    }
   else if (STREQL(btn,"Expand")) {
      FLOW_disp_expand(fw,fw->selection);
    }
   else if (STREQL(btn,"Compact")) {
      FLOW_disp_compact(fw,fw->selection);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flow_menu_display -- handle display commands			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_display(fw,mnm,bnm)
   FLOW_WIN fw;
   String mnm;
   String bnm;
{
   String menu[10240],buf[1024];
   Integer v0,v1,v2,v3,v4,v5,v6,v7;

   if (STREQL(bnm,"Show All")) {
      fw->display_all = !fw->display_all;
    }
   else if (STREQL(bnm,"Connected")) {
      fw->display_conn = !fw->display_conn;
    }
   else if (STREQL(bnm,"Show Callers")) {
      fw->display_callbys = !fw->display_callbys;
    }
   else if (STREQL(bnm,"Show Called")) {
      fw->display_callees = !fw->display_callees;
    }
   else if (STREQL(bnm,"Show Item")) {
      display_node(fw,fw->selection);
    }
   else if (STREQL(bnm,"Expand All")) {
      FLOW_disp_expand(fw,NULL);
      return TRUE;
    }
   else if (STREQL(bnm,"Show Parent")) {
      if (fw->selection != NULL) display_node(fw,fw->selection->parent);
    }
   else {
      strcpy(menu,"%CCall Graph Browser Display Options\n\n");

      strcat(menu,"%0.1o Show complete call graph\n");
      strcat(menu,"%0.0o Show graph around selection only\n\n");
      v0 = fw->display_all;

      strcat(menu,"%7o Only show reachable nodes\n\n");
      v7 = fw->display_conn;

      strcat(menu,"%1o Force redisplay on selection\n\n");
      v1 = fw->display_force;

      strcat(menu,"%2o Show calling routines of current\n");
      strcat(menu,"%3o Show routines called by current\n");
      v2 = fw->display_callbys;
      v3 = fw->display_callees;

      sprintf(buf,"      Levels: %%4.%uo All   %%4.0o None   %%4.1o Single   %%4.6d\n\n",-1);
      strcat(menu,buf);
      v4 = fw->display_levels;

      strcat(menu,"Show selection:\n");

      strcat(menu,"   %5o Expand size to show text\n\n");
      v5 = fw->display_fixcur;

      strcat(menu, "  Zoom selection:  %6.0o Normal   %6.2o Emphasize   Factor: %6.6d\n\n");
      v6 = fw->display_zoom;

      strcat(menu,"   %a%M   %c");

      if (!STEMdialog1(fw->window,menu,&v0,&v1,&v2,&v3,&v4,&v5,&v6,&v7))
	 return FALSE;

      if (v6 <= 0) v6 = 1;

      fw->display_all = v0;
      fw->display_force = v1;
      fw->display_callbys = v2;
      fw->display_callees = v3;
      fw->display_levels = v4;
      fw->display_fixcur = v5;
      fw->display_zoom = v6;
      fw->display_conn = v7;
    };

   FLOW_disp_show(fw);

   flow_reset_btns(fw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flow_menu_info -- show information about a node 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_info(fw,mnm,btn)
   FLOW_WIN fw;
   String mnm;
   String btn;
{
   if (fw->selection == NULL) return FALSE;

   display_node_info(fw,fw->selection);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	flow_menu_layout -- handle display layout menu			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_layout(fw,mnm,btn)
   FLOW_WIN fw;
   String mnm;
   String btn;
{
   Integer fix,std,cen,mthd,cmthd,white;

   fix = fw->fixed;
   std = fw->standard;
   cen = fw->centered;

   if (!GELOrequest_layout_methods(fw->window,&fw->method,&fw->connmethod,
				      &fix,&std,&cen,&fw->whitespace))
      return FALSE;

   fw->fixed = fix;
   fw->standard = std;
   fw->centered = cen;

   FLOW_disp_show(fw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flow_menu_select -- handle Select menu buttons			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_select(fw,mnm,bnm)
   FLOW_WIN fw;
   String mnm;
   String bnm;
{
   Character menu[2048],buf[1024],bufa[1024];
   FLOW_NODE fn;
   Boolean fg;

   fg = TRUE;

   if (STREQL(bnm,"Set File")) {
      strcpy(menu,"Set Selection to File or Directory\n\nPath Name: %0.64.256t\n\n   %a%M   %c");
      buf[0] = 0;
      if (!STEMdialog1(fw->window,menu,buf)) fg = FALSE;
      else {
	 fn = FLOW_node_find_function(fw,buf,NULL);
	 if (fn == NULL) fg = FALSE;
	 else FLOW_disp_select(fw,fn);
       };
    }
   else if (STREQL(bnm,"Set Function")) {
      strcpy(menu,"Set Selection to Function\n\n");
      strcat(menu,"Function Name:  %1.32.256t\n");
      strcat(menu,"File Name:      %0.32.256t\n\n   %a%M   %c");
      buf[0] = 0;
      bufa[0] = 0;
      if (!STEMdialog1(fw->window,menu,buf,bufa)) fg = FALSE;
      else {
	 fn = FLOW_node_find_function(fw,buf,bufa);
	 if (fn == NULL) fg = FALSE;
	 else FLOW_disp_select(fw,fn);
       }
    }
   else if (STREQL(bnm,"Clear Selection")) {
      FLOW_disp_select(fw,NULL);
    }
   else if (STREQL(bnm,"Set Selection")) {
      for (fn = fw->root;
	   fn->son != NULL && fn->son->son != NULL && fn->son->brother == NULL;
	   fn = fn->son);
      get_selection_info(fw,fn,-1);
    }
   else fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	flow_menu_regex -- regular expression selection 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_regex(fw,mnm,bnm)
   FLOW_WIN fw;
   String mnm;
   String bnm;
{
   Character menu[20480];
   Character ibuf[128],ebuf[128];
   Integer exfg,infg,apfg;

   strcpy(menu,"%CRegular Expression Display Selection\n\n");

   strcat(menu,"Include pattern: %0.48.128t\n");
   strcat(menu,"   %2o Exclude all others\n\n");

   strcat(menu,"Exclude pattern: %1.48.128t\n");
   strcat(menu,"   %3o Include all others\n\n");

   strcat(menu,"Apply to all:\n");
   strcat(menu,"   %4.1f Functions\n");
   strcat(menu,"   %4.2f Files\n");
   strcat(menu,"   %4.4f Directorys\n");

   strcat(menu,"   %a%M   %c");

   ebuf[0] = 0;
   ibuf[0] = 0;
   exfg = FALSE;
   infg = FALSE;
   apfg = 1;

   if (!STEMdialog1(fw->window,menu,ibuf,ebuf,&exfg,&infg,&apfg)) return FALSE;

   FLOW_disp_node_regex(fw,apfg,ibuf,exfg,ebuf,infg);

   FLOW_disp_show(fw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flow_menu_infowin -- handle requests for information window	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_menu_infowin(fw,mnm,bnm)
   FLOW_WIN fw;
   String mnm;
   String bnm;
{
   ASH_WINDOW w;

   if (fw->text_win == NULL) {
      FLOW_info_setup(fw);
    }
   else {
      for (w = fw->text_win; ASHinq_parent(w) != NULL; w = ASHinq_parent(w));
      ASHvisible(w,TRUE);
      ASHpop(w);
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	flow_btn_handler -- handle hits in flow window			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
flow_btn_handler(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   FLOW_WIN fw;
   FLOW_NODE fn;
   GELO_OBJECT go;
   Boolean fg;
   Character buf[1024];
   String s;
   static Integer xtrinfo;
   static GELO_OBJECT downobj = NULL;

   fw = (FLOW_WIN) RIPinq_data(rgn);
   if (fw == NULL || fw->window == NULL) return FALSE;

   if (btn & RIP_BTN_NONE) {
      xtrinfo = 0;
      downobj = NULL;
      switch (ch) {
	 case 'r' :
	    FLOW_disp_reset(fw,FALSE);
	    break;
	 case 'R' :
	    FLOW_disp_reset(fw,TRUE);
	    break;
	 case 'c' :
	 case 'C' :
	    FLOW_disp_item(fw,NULL);
	    break;
	 case 'a' :
	 case 'A' :
	    fw->display_all = !fw->display_all;
	    FLOW_disp_show(fw);
	    break;
	 default :
	    return FALSE;
       };

      return TRUE;
    };

   if (btn & RIP_BTN_TAP) {
      xtrinfo = 0;
      downobj = NULL;
    }
   else if (btn & RIP_BTN_DOWN) {
      downobj = GELOcorrelate(fw->disp_win,x,y);
      xtrinfo = btn & RIP_BTN_EXTRA;
      return FALSE;
    }
   else if (btn & RIP_BTN_UP) {
      btn |= xtrinfo;
    };

   go = GELOcorrelate(fw->disp_win,x,y);

   if (go == NULL || (downobj != NULL && downobj != go)) return FALSE;

   if (GELOinq_contents(go) != 0) {
      flow_arc_handler(fw,go,btn);
      return FALSE;
    };

   fn = (FLOW_NODE) GELOinq_user_structure(go);
   if (fn == NULL) return FALSE;

   fg = (fw->selection == fn);

   if (fn != NULL && fn->type == FLOW_TYPE_FUNCTION && fn->line != 0 &&
	  (btn & RIP_BTN_EXTRA) == 0) {
      FLOW_node_file(fn,buf);
      if (btn & RIP_BTN_LEFT) s = "LEFT";
      else if (btn & RIP_BTN_MID) s = "MID";
      else s = "RIGHT";
    }
   else s = NULL;

   if ((btn & RIP_BTN_EXTRA) != 0 && (btn & RIP_BTN_LEFT) != 0) {
      fn->ignore = TRUE;
      FLOW_disp_show(fw);
    }
   else if (fg && (btn & RIP_BTN_EXTRA) != 0 && (btn & RIP_BTN_RIGHT) != 0) {
      FLOW_disp_compact(fw,fn);
    }
   else {
      if (s != NULL) MSGsenda("FLOW SET %s %d %s",buf,fn->line,s);
      if ((btn & RIP_BTN_MID) != 0) {
	 display_node_info(fw,fn);
       }
      else if (fg) {
	 if (btn & RIP_BTN_LEFT) {
	    display_node(fw,fn);
	  }
	 else if (fn->son == NULL) {
	    FLOW_disp_compact(fw,fn);
	  }
	 else {
	    FLOW_disp_expand(fw,fn);
	  };
       }
      else {
	 FLOW_disp_select(fw,fn);
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flow_arc_handler -- handle hits on arc				*/
/*									*/
/************************************************************************/


static Boolean
flow_arc_handler(fw,go,btn)
   FLOW_WIN fw;
   GELO_OBJECT go;
   Integer btn;
{
   Integer i;
   String s;
   Character buf[1024];
   FLOW_NODE fn;

   for (i = 0; i < fw->numconn; ++i) {
      if (fw->conns[i].gid == go) break;
    };

   if (i >= fw->numconn) return FALSE;

   if (btn & RIP_BTN_LEFT) s = "LEFT";
   else if (btn & RIP_BTN_MID) s = "MID";
   else s = "RIGHT";

   if (fw->conns[i].line != 0 && (btn & RIP_BTN_EXTRA) == 0) {
      FLOW_node_file(fw->conns[i].from,buf);
      MSGsenda("FLOW SET %s %d %s",buf,fw->conns[i].line,s);
    };

   if (btn & RIP_BTN_MID) {
      arc_info(fw,go);
    }
   else {
      if ((btn & RIP_BTN_LEFT) != 0) fn = fw->conns[i].to;
      else fn = fw->conns[i].from;
      FLOW_disp_select(fw,fn);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	arc_info -- show information about an arc			*/
/*									*/
/************************************************************************/


static void
arc_info(fw,go)
   FLOW_WIN fw;
   GELO_OBJECT go;
{
   Character menu[10240];
   Character buf[1024],buf1[1024];
   FLOW_NODE fna,fnb;
   Integer i,j,k;

   for (i = 0; i < fw->numconn; ++i) {
      if (fw->conns[i].gid == go) break;
    };
   if (i >= fw->numconn) return;

   for (fna = fw->conns[i].from; fna != NULL; fna = fna->parent) {
      if (fna->display) break;
    };
   for (fnb = fw->conns[i].to; fnb != NULL; fnb = fnb->parent) {
      if (fnb->display) break;
    };
   if (fna == NULL || fnb == NULL) return;

   menu[0] = 0;
   strcat(menu,"%CArc Information\n\nFrom: ");

   switch (fna->type) {
      case FLOW_TYPE_DIRECTORY :
	 strcat(menu,"Directory ");
	 break;
      case FLOW_TYPE_FILE :
	 strcat(menu,"File ");
	 break;
      case FLOW_TYPE_FUNCTION :
	 strcat(menu,"Function ");
	 break;
    };
   strcat(menu,fna->name);
   strcat(menu,"\nTo:   ");
   switch (fnb->type) {
      case FLOW_TYPE_DIRECTORY :
	 strcat(menu,"Directory ");
	 break;
      case FLOW_TYPE_FILE :
	 strcat(menu,"File ");
	 break;
      case FLOW_TYPE_FUNCTION :
	 strcat(menu,"Function ");
	 break;
    };
   strcat(menu,fnb->name);
   strcat(menu,"\n\n");

   k = 0;
   for (j = i; j < fw->numconn; ++j) {
      if (fw->conns[j].gid == go) {
	 if (k >= MAX_INFO_ARC) {
	    strcat(menu,"\n...\n");
	    break;
	  };
	 if (fna != fw->conns[j].from || fnb != fw->conns[j].to) {
	    sprintf(buf,"\nFrom %s to %s\n",fw->conns[j].from->name,
		       fw->conns[j].to->name);
	    strcat(menu,buf);
	    fna = fw->conns[j].from;
	    fnb = fw->conns[j].to;

	  };
	 FLOW_node_file(fna->parent,buf);
	 if (buf[0] != 0) {
	    sprintf(buf1,"   At line %d in %s \n",fw->conns[j].line,buf);
	  }
	 else {
	    sprintf(buf1,"   At line %d \n",fw->conns[j].line);
	  }
	 strcat(menu,buf1);
	 ++k;
       };
    };

   strcat(menu,"\n%M   %a");

   STEMdialog1(fw->window,menu);
};






/************************************************************************/
/*									*/
/*	do_trace_display -- handle display tracing			*/
/*									*/
/************************************************************************/


static void
do_trace_display(fw,fn,all)
   FLOW_WIN fw;
   FLOW_NODE fn;
   Boolean all;
{
   FLOW_NODE sf;
   Character buf[1024],buf1[1024];
   String s;

   if (fn->ignore) return;
   if (fn->type == FLOW_TYPE_FUNCTION && (all || fn->display)) {
      FLOW_node_file(fn->parent,buf1);
      s = rindex(buf1,'/');
      if (buf1[0] == 0) strcpy(buf,"*");
      else if (s == NULL) strcpy(buf,buf1);
      else {
	 ++s;
	 strcpy(buf,s);
       };
      s = rindex(fw->system,'.');
      if (s != NULL &&
	     (STREQL(s,".p") || STREQL(s,".c") ||
		 STREQL(s,".C") || STREQL(s,".cc"))) {
	 s = "*";
       }
      else s = fw->system;
      if (fn != fw->main) {
	 MSGsenda("DDTR EVENT ADD %s %s %s 0 * * 0 CALL 1",s,buf,
		     fn->name);
       };
    }
   else if (fn->display) all = TRUE;

   for (sf = fn->son; sf != NULL; sf = sf->brother) {
      do_trace_display(fw,sf,all);
    };
};






/************************************************************************/
/*									*/
/*	flow_reset_btns -- handle setting conditions on menu btns	*/
/*									*/
/************************************************************************/


static void
flow_reset_btns(fw)
   FLOW_WIN fw;
{
   STEMpdm_btn_select(fw->menu_win,"Display","Show All",fw->display_all);
   STEMpdm_btn_select(fw->menu_win,"Display","Connected",fw->display_conn);
   STEMpdm_btn_select(fw->menu_win,"Display","Show Callers",fw->display_callbys);
   STEMpdm_btn_select(fw->menu_win,"Display","Show Called",fw->display_callees);
};





/************************************************************************/
/*									*/
/*	get_selection_info -- ask user about individual items		*/
/*									*/
/************************************************************************/


#define MAX_ITEMS	32


static Integer
get_selection_info(fw,fn,idx)
   FLOW_WIN fw;
   FLOW_NODE fn;
   Integer idx;
{
   Character menu[20480],buf[1024];
   Integer i,j,k,st,ct;
   FLOW_NODE sf;
   Integer ignfg[(MAX_ITEMS+16)/16],cur,show;
   FLOW_NODE * nds;

   ct = 0;
   for (sf = fn->son; sf != NULL; sf = sf->brother) ++ct;
   nds = (FLOW_NODE *) alloca(sizeof(FLOW_NODE)*(ct+2));
   ct = 0;
   for (sf = fn->son; sf != NULL; sf = sf->brother) nds[ct++] = sf;

   qsort(nds,ct,sizeof(FLOW_NODE),node_compare);

   st = idx;
   if (st < 0) st = 0;

   if (ct == 0) return 0;

   strcpy(menu,"%CView Nodes\n\nIgnore Items   Node%MIgnore Items   Node");

   for (i = 0; i < (MAX_ITEMS+16)/16; ++i) ignfg[i] = 0;
   cur = -1;

   for (i = st; i < ct && i-st < MAX_ITEMS; ++i) {
      sf = nds[i];
      j = (i-st) % 16;
      k = (i-st) / 16;
      if (i & 1) strcat(menu,"%M  ");
      else strcat(menu,"\n  ");
      if (sf->son != NULL) {
	 sprintf(buf,"%%%d.%df   %%1.%do   %%0.%d\"%s\"b  ",k+2,(1 << j),i,i,sf->name);
       }
      else {
	 sprintf(buf,"%%%d.%df         %%0.%d\"%s\"b  ",k+2,(1 << j),i,sf->name);
       };
      strcat(menu,buf);
      if (sf->ignore) ignfg[k] |= (1 << j);
      if (fw->selection == sf) cur = i;
    };

   strcat(menu,"\n\n");
   show = MAX_ITEMS;
   if (st != 0 || i < ct || (fn->parent != NULL && idx >= 0)) {
      if (st != 0) {
	 sprintf(buf,"   %%1.%d\"Previous Items\"a",MAX_ITEMS+1);
	 strcat(menu,buf);
       };
      if (i < ct) {
	 sprintf(buf,"%%M%%1.%d\"More Items\"a",MAX_ITEMS+2);
	 strcat(menu,buf);
       };
      if (fn->parent != NULL && idx >= 0) {
	 sprintf(buf,"\n   %%1.%d\"Parent Items\"a",MAX_ITEMS+3);
	 strcat(menu,buf);
       };
      strcat(menu,"\n\n");
    };

   strcat(menu,"      %a%M   %c");

   if (!STEMdialog1(fw->window,menu,&cur,&show,&ignfg[0],&ignfg[1],&ignfg[2],&ignfg[3]))
      return 0;

   for (i = st; i < ct && i-st < MAX_ITEMS; ++i) {
      sf = nds[i];
      j = (i-st) % 16;
      k = (i-st) / 16;
      if ((ignfg[k] & (1 << j)) != 0) sf->ignore = TRUE;
      else sf->ignore = FALSE;
    };
   if (cur >= 0) fw->selection = nds[cur];

   if (show != MAX_ITEMS) {
      sf = fn;
      if (show < MAX_ITEMS) {
	 sf = nds[show];
	 st = 0;
       }
      else if (show == MAX_ITEMS+1) {
	 st -= MAX_ITEMS;
	 if (st < 0) st = 0;
       }
      else if (show == MAX_ITEMS+2) {
	 st += MAX_ITEMS;
	 if (st >= ct) st = ct-2;
       }
      else return -1;

      i = get_selection_info(fw,sf,st);
      while (i < 0) {
	 i = get_selection_info(fw,fn,idx);
       };

      if (i > 0) return 1;
    };

   FLOW_disp_show(fw);

   return 1;
};






/************************************************************************/
/*									*/
/*	node_compare -- compare two nodes for sorting			*/
/*									*/
/************************************************************************/


static int
node_compare(fna,fnb)
   FLOW_NODE * fna, * fnb;
{
   return strcmp((*fna)->name,(*fnb)->name);
};





/************************************************************************/
/*									*/
/*	display_node_info -- show information about a node		*/
/*									*/
/************************************************************************/


static void
display_node_info(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   Character menu[10240];
   Character buf[1024],buf1[1024];
   FLOW_NODE sf;
   Integer ct,in,out;

   FLOW_info_node(fw,fn);

   menu[0] = 0;

   switch (fn->type) {
      case FLOW_TYPE_DIRECTORY :
	 strcat(menu,"Directory ");
	 break;
      case FLOW_TYPE_FILE :
	 strcat(menu,"File ");
	 break;
      case FLOW_TYPE_FUNCTION :
	 strcat(menu,"Function ");
	 break;
    };

   strcat(menu,fn->name);
   strcat(menu,"\n\n");

   if (fn->parent != NULL && fn->parent->type != FLOW_TYPE_ROOT) {
      FLOW_node_file(fn->parent,buf);
      if (fn->line > 0 && buf[0] != 0) {
	 sprintf(buf1,"At line %d in %s\n\n",fn->line,buf);
       }
      else if (fn->line > 0) {
	 sprintf(buf1,"At line %d\n\n",fn->line);
       }
      else if (buf[0] != 0) {
	 sprintf(buf1,"In %s\n\n",buf);
       }
      else buf1[0] = 0;

      strcat(menu,buf1);
    };

   if (fn->son != NULL) {
      ct = 0;
      for (sf = fn->son; sf != NULL; sf = sf->brother) ++ct;
      sprintf(buf,"%d son%s\n\n",ct,(ct == 1 ? "" : "s"));
      strcat(menu,buf);
    };

   FLOW_node_count_arcs(fn,&in,&out);

   sprintf(buf,"Called by %d routines\nCalls to  %d routines\n\n",in,out);
   strcat(menu,buf);

   strcat(menu,"%M   %a");

   STEMdialog1(fw->window,menu);
};







/************************************************************************/
/*									*/
/*	display_node -- handle request to display a particular item	*/
/*									*/
/************************************************************************/


static void
display_node(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   FLOW_info_node(fw,fn);

   if (fn == NULL || fn->type != FLOW_TYPE_FUNCTION)
      FLOW_disp_item(fw,fn);
};





/* end of flowmenu.c */
