/************************************************************************/
/*									*/
/*		flowdisp.c						*/
/*									*/
/*	Display interface for flow windows				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "flow_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/

#define DEFAULT_METHOD		GELO_METHOD_DAVG2
/* #define DEFAULT_METHOD	   GELO_METHOD_GRID  */
#define DEFAULT_CONNMETHOD	GELO_CONN_METHOD_DIRECT_RECT
#define DEFAULT_FIXED		FALSE
#define DEFAULT_STANDARD	FALSE
#define DEFAULT_CENTERED	TRUE
#define DIRECTORY_SHAPE 	GELO_SHAPE_HEXAGON
#define FILE_SHAPE		GELO_SHAPE_RECTANGLE
#define FUNCTION_SHAPE		GELO_SHAPE_ELLIPSE
#define DEFAULT_ARC_STYLE	ASH_STYLE_SOLID
#define DEFAULT_ARROW		GELO_ARROW_SINGLE_ALL
#define PERT_ARROW		GELO_ARROW_SINGLE
#define DEFAULT_WHITESPACE	0
#define SELECT_STYLE		1
#define EXECUTE_STYLE		5
#define STACK_STYLE		2

#define DEFAULT_DISPLAY_ALL	TRUE
#define DEFAULT_DISPLAY_FORCE	TRUE
#define DEFAULT_DISPLAY_CALLBYS TRUE
#define DEFAULT_DISPLAY_CALLEES TRUE
#define DEFAULT_DISPLAY_FIXCUR	FALSE
#define DEFAULT_DISPLAY_CONN	FALSE
#define DEFAULT_DISPLAY_LEVELS	(-1)
#define DEFAULT_DISPLAY_ZOOM	0

#define MAX_LEVEL		100000000




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	GELO_METHOD		default_method;
static	GELO_CONN_METHOD	default_connmethod;
static	Boolean 		default_fixed;
static	Boolean 		default_standard;
static	Boolean 		default_centered;
static	Integer 		default_whitespace;
static	GELO_SHAPE		directory_shape;
static	GELO_SHAPE		file_shape;
static	GELO_SHAPE		function_shape;
static	ASH_LINE_STYLE		arc_style;
static	GELO_ARC_ARROW		arrow_style;
static	GELO_ARC_ARROW		pert_arrow_style;
static	Integer 		select_style;
static	Integer 		execute_style;
static	Integer 		stack_style;
static	Boolean 		show_stack;
static	Boolean 		auto_update;

static	Boolean 		default_display_all;
static	Boolean 		default_display_force;
static	Boolean 		default_display_callbys;
static	Boolean 		default_display_callees;
static	Boolean 		default_display_fixcur;
static	Boolean 		default_display_conn;
static	Integer 		default_display_levels;
static	Integer 		default_display_zoom;

static	Sequence		default_name_inc;
static	Sequence		default_name_exc;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		apply_pattern();
static	void		clear_display();
static	Integer 	expand();
static	void		expand_all();
static	void		display_nodes();
static	void		display_conns();
static	FLOW_NODE	arc_node();
static	void		handle_exec_disp();
static	void		handle_start_disp();
static	void		handle_stop_disp();
static	void		handle_enter_disp();
static	void		handle_exit_disp();
static	void		handle_build_finish();
static	void		disp_execute();
static	void		disp_stack();
static	void		compute_display();
static	void		compute_levels();
static	void		clear_levels();
static	void		propogate_levels();
static	void		check_display();
static	Sequence	flow_auxd_list();






/************************************************************************/
/*									*/
/*	FLOW_disp_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FLOW_disp_init()
{
   AUXD hdl;
   String msg;

   hdl = AUXDget_handle(NULL,"FLOW");

   msg = AUXDget_info(hdl,"DEBUG_AT");
   if (msg != NULL) MSGregister(msg,handle_exec_disp,3,NULL);

   msg = AUXDget_info(hdl,"DEBUG_ENTER");
   if (msg != NULL) MSGregister(msg,handle_enter_disp,3,NULL);

   msg = AUXDget_info(hdl,"DEBUG_EXIT");
   if (msg != NULL) MSGregister(msg,handle_exit_disp,3,NULL);

   msg = AUXDget_info(hdl,"DEBUG_START");
   if (msg != NULL) MSGregister(msg,handle_start_disp,1,NULL);

   msg = AUXDget_info(hdl,"DEBUG_STOP");
   if (msg != NULL) MSGregister(msg,handle_stop_disp,1,NULL);

   msg = AUXDget_info(hdl,"BUILD_FINISH");
   if (msg != NULL) MSGregister(msg,handle_build_finish,2,NULL);

   if (!AUXDget_defined(hdl,"METHOD")) default_method = DEFAULT_METHOD;
   else default_method = (GELO_METHOD) AUXDget_info_int(hdl,"METHOD");

   if (!AUXDget_defined(hdl,"CONNMETHOD")) default_connmethod = DEFAULT_CONNMETHOD;
   else default_connmethod = (GELO_CONN_METHOD) AUXDget_info_int(hdl,"CONNMETHOD");

   if (!AUXDget_defined(hdl,"FIXED")) default_fixed = DEFAULT_FIXED;
   else default_fixed = AUXDget_info_int(hdl,"FIXED");

   if (!AUXDget_defined(hdl,"STANDARD")) default_standard = DEFAULT_STANDARD;
   else default_standard = AUXDget_info_int(hdl,"STANDARD");

   if (!AUXDget_defined(hdl,"CENTERED")) default_centered = DEFAULT_CENTERED;
   else default_centered = AUXDget_info_int(hdl,"CENTERED");

   if (!AUXDget_defined(hdl,"WHITE_SPACE")) default_whitespace = DEFAULT_WHITESPACE;
   else default_whitespace = AUXDget_info_int(hdl,"WHITE_SPACE");

   if (!AUXDget_defined(hdl,"DIRECTORY_SHAPE")) directory_shape = DIRECTORY_SHAPE;
   else directory_shape = (GELO_SHAPE) AUXDget_info_int(hdl,"DIRECTORY_SHAPE");

   if (!AUXDget_defined(hdl,"FILE_SHAPE")) file_shape = FILE_SHAPE;
   else file_shape = (GELO_SHAPE) AUXDget_info_int(hdl,"FILE_SHAPE");

   if (!AUXDget_defined(hdl,"FUNCTION_SHAPE")) function_shape = FUNCTION_SHAPE;
   else function_shape = (GELO_SHAPE) AUXDget_info_int(hdl,"FUNCTION_SHAPE");

   if (!AUXDget_defined(hdl,"ARC_STYLE")) arc_style = DEFAULT_ARC_STYLE;
   else arc_style = (ASH_LINE_STYLE) AUXDget_info_int(hdl,"ARC_STYLE");

   if (!AUXDget_defined(hdl,"ARROW_STYLE")) arrow_style = DEFAULT_ARROW;
   else arrow_style = (GELO_ARC_ARROW) AUXDget_info_int(hdl,"ARROW_STYLE");

   if (!AUXDget_defined(hdl,"PERT_ARROW_STYLE")) pert_arrow_style = PERT_ARROW;
   else pert_arrow_style = (GELO_ARC_ARROW) AUXDget_info_int(hdl,"PERT_ARROW_STYLE");

   show_stack = ! AUXDget_defined(hdl,"NO_STACK");
   auto_update = AUXDget_defined(hdl,"AUTO_UPDATE");

   if (!AUXDget_defined(hdl,"DISPLAY_ALL")) default_display_all = DEFAULT_DISPLAY_ALL;
   else default_display_all = AUXDget_info_int(hdl,"DISPLAY_ALL");

   if (!AUXDget_defined(hdl,"DISPLAY_FORCE")) default_display_force = DEFAULT_DISPLAY_FORCE;
   else default_display_force = AUXDget_info_int(hdl,"DISPLAY_FORCE");

   if (!AUXDget_defined(hdl,"DISPLAY_CALLBYS")) default_display_callbys = DEFAULT_DISPLAY_CALLBYS;
   else default_display_callbys = AUXDget_info_int(hdl,"DISPLAY_CALLBYS");

   if (!AUXDget_defined(hdl,"DISPLAY_CALLEES")) default_display_callees = DEFAULT_DISPLAY_CALLEES;
   else default_display_callees = AUXDget_info_int(hdl,"DISPLAY_CALLEES");

   if (!AUXDget_defined(hdl,"DISPLAY_FIXCUR")) default_display_fixcur = DEFAULT_DISPLAY_FIXCUR;
   else default_display_fixcur = AUXDget_info_int(hdl,"DISPLAY_FIXCUR");

   if (!AUXDget_defined(hdl,"DISPLAY_CONN")) default_display_conn = DEFAULT_DISPLAY_CONN;
   else default_display_conn = AUXDget_info_int(hdl,"DISPLAY_CONN");

   if (!AUXDget_defined(hdl,"DISPLAY_LEVELS")) default_display_levels = DEFAULT_DISPLAY_LEVELS;
   else default_display_levels = AUXDget_info_int(hdl,"DISPLAY_LEVELS");

   if (!AUXDget_defined(hdl,"DISPLAY_ZOOM")) default_display_zoom = DEFAULT_DISPLAY_ZOOM;
   else default_display_zoom = AUXDget_info_int(hdl,"DISPLAY_ZOOM");

   default_name_exc = flow_auxd_list(hdl,"NAME_EXCLUDE");
   default_name_inc = flow_auxd_list(hdl,"NAME_INCLUDE");

   select_style = SELECT_STYLE;
   execute_style = EXECUTE_STYLE;
   stack_style = STACK_STYLE;
};





/************************************************************************/
/*									*/
/*	FLOW_disp_setup -- setup display in window			*/
/*									*/
/************************************************************************/


void
FLOW_disp_setup(fw)
   FLOW_WIN fw;
{
   fw->gid = NULL;
   if (fw->disp_win == NULL) return;

   GELOwindow_open(fw->disp_win);

   fw->method = default_method;
   fw->connmethod = default_connmethod;
   fw->fixed = default_fixed;
   fw->standard = default_standard;
   fw->centered = default_centered;
   fw->whitespace = default_whitespace;

   fw->selection = NULL;
   fw->execute = NULL;
   fw->callstack = NULL;
   fw->level_valid = FALSE;
   fw->conn_valid = FALSE;

   fw->display_all = default_display_all;
   fw->display_force = default_display_force;
   fw->display_callbys = default_display_callbys;
   fw->display_callees = default_display_callees;
   fw->display_fixcur = default_display_fixcur;
   fw->display_conn = default_display_conn;
   fw->display_levels = default_display_levels;
   fw->display_zoom = default_display_zoom;

   FLOW_disp_reset(fw,TRUE);
};





/************************************************************************/
/*									*/
/*	FLOW_disp_reset -- set up initial nodes to display		*/
/*									*/
/************************************************************************/


void
FLOW_disp_reset(fw,clr)
   FLOW_WIN fw;
   Boolean clr;
{
   Integer ct,lct;
   String s;
   Sequence l;
   Boolean fg;

   fw->topdisp = fw->root;

   if (fw->root == NULL) return;

   clear_display(fw->root,clr);

   if (clr) {
      forin (s,String,l,default_name_exc) {
	 FLOW_disp_node_regex(fw,7,NULL,FALSE,s,FALSE);
       };
      fg = (default_name_exc == NULL);
      forin (s,String,l,default_name_inc) {
	 FLOW_disp_node_regex(fw,7,s,fg,NULL,FALSE);
	 fg = FALSE;
       };
    };

   ct = expand(fw->root,TRUE);

   while (ct < 3) {
      lct = ct;
      ct += expand(fw->root,TRUE);
      if (ct == lct) break;
    };

   if (clr && ct == 0) {
      STEMdialog1(fw->window," No flow graph to display \n\n%M   %a");
    };

   FLOW_disp_show(fw);

   if (clr) FLOW_menu_setup_trace(fw);
};





/************************************************************************/
/*									*/
/*	FLOW_disp_show -- handle displaying the tree			*/
/*									*/
/************************************************************************/


void
FLOW_disp_show(fw)
   FLOW_WIN fw;
{
   FLOW_NODE fn;
   Sequence l,la;

   ASHinput_lock_makelong(fw->window);

   if (fw->gid != NULL) {
      GELOwindow_free(fw->disp_win,fw->gid);
      fw->gid = NULL;
    };

   fw->gid = GELOdefine_layout();
   GELOset_owner(fw->gid,fw);
   GELOdefine_layout_method(fw->gid,fw->method);
   GELOdefine_layout_conn_method(fw->gid,fw->connmethod);
   GELOdefine_layout_fixed(fw->gid,fw->fixed);
   GELOdefine_layout_standard(fw->gid,fw->standard);
   GELOdefine_layout_centered(fw->gid,fw->centered);
   GELOdefine_layout_white_space(fw->gid,fw->whitespace);

   if (fw->topdisp == NULL) fw->topdisp = fw->root;

   compute_display(fw);

   fw->numnodes = 0;

   fw->maindisp = NULL;
   for (fn = fw->main; fn != NULL; fn = fn->parent) {
      if (fn->display) break;
    };
   if (fn != NULL) {
      display_nodes(fw,fn);
      fw->maindisp = fn;
    };

   display_nodes(fw,fw->root);

   display_conns(fw);

   GELOwindow_draw(fw->disp_win,fw->gid);

   if (fw->selection != NULL) {
      if (fw->selection->gid == NULL) {
	 FLOW_menu_set_selection(fw,NULL);
	 fw->selection = NULL;
       }
      else {
	 GELOwindow_select(fw->disp_win,fw->selection->gid,select_style,TRUE);
       };
    };

   if (fw->execute != NULL) {
      if (fw->execute->gid == NULL) fw->execute = NULL;
      else {
	 GELOwindow_select(fw->disp_win,fw->execute->gid,execute_style,TRUE);
       };
    };

   if (fw->callstack != NULL && show_stack) {
      la = fw->callstack;
      fw->callstack = NULL;
      forin (fn,FLOW_NODE,l,la) disp_stack(fw,fn,TRUE);
    };
};





/************************************************************************/
/*									*/
/*	FLOW_disp_free -- free current display				*/
/*									*/
/************************************************************************/


void
FLOW_disp_free(fw)
   FLOW_WIN fw;
{
   if (fw->gid != NULL) {
      GELOwindow_free(fw->disp_win,fw->gid);
      fw->gid = NULL;
    };
};





/************************************************************************/
/*									*/
/*	FLOW_disp_select -- set selection				*/
/*									*/
/************************************************************************/


void
FLOW_disp_select(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   Boolean redo;

   FLOW_info_node(fw,fn);

   if (fw->selection == fn) return;

   FLOW_menu_set_selection(fw,fn);

   fw->level_valid = FALSE;

   redo = fw->display_force;
   if (fw->display_all && !fw->display_fixcur && fw->display_zoom <= 1) redo = FALSE;

   if (redo) {
      fw->selection = fn;
      FLOW_disp_show(fw);
    }
   else {
      if (fw->selection != NULL) {
	 GELOwindow_select(fw->disp_win,fw->selection->gid,select_style,FALSE);
       };

      fw->selection = fn;

      if (fn != NULL && fn->gid != NULL) {
	 GELOwindow_select(fw->disp_win,fn->gid,select_style,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	FLOW_disp_expand -- expand node and redisplay			*/
/*	FLOW_disp_compact -- compact node and redisplay 		*/
/*	FLOW_disp_ignore -- ignore node and redisplay			*/
/*	FLOW_disp_item -- show only specified item			*/
/*									*/
/************************************************************************/


void
FLOW_disp_expand(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   Character buf[1024];

   if (fn == NULL) {
      expand_all(fw->root);
      FLOW_disp_show(fw);
    }
   else if (fn->type != FLOW_TYPE_FUNCTION) {
      expand(fn,TRUE);
      FLOW_disp_show(fw);
    }
   else if (fn->line != 0) {
      FLOW_node_file(fn,buf);
      MSGsenda("FLOW SET %s %d NONE",buf,fn->line);
    };
};





void
FLOW_disp_compact(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   FLOW_NODE sf;

   if (fn != NULL && fn->parent != NULL && fn->parent->type != FLOW_TYPE_ROOT) {
      fn->parent->expand = FALSE;
      for (sf = fw->topdisp; sf != NULL; sf = sf->parent) {
	 if (sf == fn->parent) fw->topdisp = NULL;
       };
      FLOW_disp_show(fw);
    };
};





void
FLOW_disp_ignore(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   if (fn == NULL) return;

   fn->ignore = TRUE;

   FLOW_disp_show(fw);
};





void
FLOW_disp_item(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   fw->topdisp = fn;

   if (fn != NULL && fn->son != NULL) fn->expand = TRUE;

   FLOW_disp_show(fw);
};






/************************************************************************/
/*									*/
/*	FLOW_disp_node_regex -- do pattern selection on nodes		*/
/*									*/
/************************************************************************/


void
FLOW_disp_node_regex(fw,what,inc,exfg,exc,infg)
   FLOW_WIN fw;
   Integer what;
   String inc;
   Boolean exfg;
   String exc;
   Boolean infg;
{
   if (what != 0) {
      if (inc != NULL && inc[0] != 0 && re_comp(inc) == NULL) {
	 apply_pattern(fw,fw->root,exfg,FALSE,what);
       };

      if (exc != NULL && exc[0] != 0 && re_comp(exc) == NULL) {
	 apply_pattern(fw,fw->root,infg,TRUE,what);
       };
    };
};





static void
apply_pattern(fw,fn,exfg,igfg,apfg)
   FLOW_WIN fw;
   FLOW_NODE fn;
   Boolean exfg;
   Boolean igfg;
   Integer apfg;
{
   Integer i;
   FLOW_NODE sf;

   switch (fn->type) {
      case FLOW_TYPE_DIRECTORY :
	 i = 4;
	 break;
      case FLOW_TYPE_FILE :
	 i = 2;
	 break;
      case FLOW_TYPE_FUNCTION :
	 i = 1;
	 break;
      default :
	 i = 0;
	 break;
    };

   if (i & apfg) {
      if (re_exec(fn->name)) {
	 fn->ignore = igfg;
       }
      else if (exfg) fn->ignore = !igfg;
    };

   if ((i & apfg) == 0 || !exfg) {
      for (sf = fn->son; sf != NULL; sf = sf->brother) {
	 apply_pattern(fw,sf,exfg,igfg,apfg);
       };
    };
};





/************************************************************************/
/*									*/
/*	clear_display -- clear displays for a tree			*/
/*									*/
/************************************************************************/


static void
clear_display(fn,clr)
   FLOW_NODE fn;
   Boolean clr;
{
   FLOW_NODE sf;

   fn->gid = NULL;
   fn->display = FALSE;
   fn->expand = FALSE;

   if (clr) fn->ignore = FALSE;

   for (sf = fn->son; sf != NULL; sf = sf->brother) clear_display(sf,clr);
};






/************************************************************************/
/*									*/
/*	expand -- expand a subtree one level, return count		*/
/*	expand_all -- expand all subtrees to bottom level		*/
/*									*/
/************************************************************************/


static int
expand(fn,dofg)
   FLOW_NODE fn;
   Boolean dofg;
{
   Integer ct;
   FLOW_NODE sf;

   if (fn->ignore) ct = 0;
   else if (dofg || fn->expand) {
      ct = 0;
      for (sf = fn->son; sf != NULL; sf = sf->brother)
	 ct += expand(sf,(dofg && fn->expand));
      if (dofg && ct > 0) fn->expand = TRUE;
    }
   else ct = 1;

   return ct;
};





static void
expand_all(fn)
   FLOW_NODE fn;
{
   FLOW_NODE sf;

   if (fn->son != NULL) fn->expand = TRUE;

   for (sf = fn->son; sf != NULL; sf = sf->brother)
      expand_all(sf);
};





/************************************************************************/
/*									*/
/*	display_nodes -- display nodes of a subtree			*/
/*									*/
/************************************************************************/


static void
display_nodes(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   FLOW_NODE sf;
   Float f;

   if (fn == fw->maindisp) return;

   fn->gid = NULL;

   if (!fn->display) {
      for (sf = fn->son; sf != NULL; sf = sf->brother) {
	 display_nodes(fw,sf);
       };
    }
   else if (fn->type == FLOW_TYPE_ROOT) ;
   else {
      fn->gid = GELOdefine_data();
      switch (fn->type) {
	 case FLOW_TYPE_DIRECTORY :
	    GELOdefine_data_shape(fn->gid,directory_shape);
	    break;
	 case FLOW_TYPE_FILE :
	    GELOdefine_data_shape(fn->gid,file_shape);
	    break;
	 case FLOW_TYPE_FUNCTION :
	    GELOdefine_data_shape(fn->gid,function_shape);
	    break;
       };
      GELOdefine_data_text(fn->gid,fn->dispname);
      GELOdefine_layout_component(fw->gid,fn->gid);
      GELOset_owner(fn->gid,fw);
      GELOset_contents(fn->gid,0);
      GELOset_user_structure(fn->gid,fn);

      if (fw->selection == fn) {
	 if (fw->display_fixcur) {
	    GELOuse_default_x(fn->gid,TRUE);
	    GELOuse_default_y(fn->gid,TRUE);
	  };
	 if (fw->display_zoom > 0) {
	    f = fw->display_zoom;
	    GELOset_priority_x(fn->gid,f);
	    GELOset_priority_y(fn->gid,f);
	  };
       };

      ++fw->numnodes;
    };
};






/************************************************************************/
/*									*/
/*	display_conns -- display connections in the graph		*/
/*									*/
/************************************************************************/


static void
display_conns(fw)
   FLOW_WIN fw;
{
   Integer i,j,ct;
   FLOW_ARC fa;
   FLOW_ARC saved;
   FLOW_NODE fnf,fnt;
   GELO_CONNECT gc;

   ct = MIN(fw->numnodes*fw->numnodes,fw->numconn);

   saved = (FLOW_ARC) alloca(ct*sizeof(FLOW_ARC_INFO));
   ct = 0;

   for (i = 0; i < fw->numconn; ++i) {
      fa = &fw->conns[i];
      fa->gid = NULL;

      fnf = arc_node(fa->from);
      fnt = arc_node(fa->to);

      if (fnf == NULL || fnt == NULL) continue;
      if (fnf == fnt && fnf->type != FLOW_TYPE_FUNCTION) continue;

      for (j = 0; j < ct; ++j) {
	 if (fnf == saved[j].from && fnt == saved[j].to) break;
       }
      if (j < ct) {
	 fa->gid = saved[j].gid;
	 j = GELOinq_contents(fa->gid);
	 GELOset_contents(fa->gid,j+1);
	 continue;
       };

      fa->gid = GELOdefine_arc();
      gc = GELOnew_connect(fnf->gid,GELO_PORT_ANY,fnt->gid,GELO_PORT_ANY);
      if (fw->method & (GELO_METHOD_PERTBITS|GELO_METHOD_BIT_GIOTTO))
	 GELOconnect_arc_style(gc,arc_style,pert_arrow_style);
      else
	 GELOconnect_arc_style(gc,arc_style,arrow_style);
      GELOdefine_arc_connect(fa->gid,gc);
      GELOset_contents(fa->gid,1);
      GELOset_owner(fa->gid,fw);
      GELOset_user_structure(fa->gid,fa);
      GELOdefine_layout_arc(fw->gid,fa->gid);

      saved[ct].from = fnf;
      saved[ct].to = fnt;
      saved[ct].gid = fa->gid;
      saved[ct].line = 0;
      ++ct;
    };

   fw->numarcs = ct;
};






/************************************************************************/
/*									*/
/*	arc_node -- get FLOW_NODE for arc display			*/
/*									*/
/************************************************************************/


static FLOW_NODE
arc_node(fn)
   FLOW_NODE fn;
{
   while (fn != NULL) {
      if (fn->ignore) return NULL;
      if (fn->display && fn->gid != NULL) break;
      fn = fn->parent;
    };

   return fn;
};





/************************************************************************/
/*									*/
/*	handle_exec_disp -- display execution status			*/
/*									*/
/************************************************************************/


static void
handle_exec_disp(sys,fil,fun,rid)
   String sys;
   String fil;
   String fun;
   Integer rid;
{
   FLOW_WIN fw;
   FLOW_NODE fn;

   if (rid >= 0) MSGreply(rid,NULL);

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;
   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   fn = FLOW_node_find_function(fw,fil,fun);

   while (fn != NULL && !fn->display) fn = fn->parent;

   disp_execute(fw,fn);

   ASHinput_unlock(fw->window);
};





/************************************************************************/
/*									*/
/*	handle_start_disp -- handle start run message			*/
/*									*/
/************************************************************************/


static void
handle_start_disp(sys,rid)
   String sys;
   Integer rid;
{
   FLOW_WIN fw;

   if (rid >= 0) MSGreply(rid,NULL);

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;
   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   while (fw->callstack != NULL) disp_stack(fw,NULL,FALSE);

   disp_execute(fw,fw->maindisp);

   ASHinput_unlock(fw->window);
};





/************************************************************************/
/*									*/
/*	handle_stop_disp -- handle stop run message			*/
/*									*/
/************************************************************************/


static void
handle_stop_disp(sys,rid)
   String sys;
   Integer rid;
{
   FLOW_WIN fw;

   if (rid >= 0) MSGreply(rid,NULL);

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;

   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   while (fw->callstack != NULL) disp_stack(fw,NULL,FALSE);

   disp_execute(fw,NULL);

   ASHinput_unlock(fw->window);
};





/************************************************************************/
/*									*/
/*	handle_enter_disp -- handle ENTER trace request 		*/
/*	handle_exit_disp -- handle EXIT trace request			*/
/*									*/
/************************************************************************/


static void
handle_enter_disp(sys,fil,fun,rid)
   String sys;
   String fil;
   String fun;
   Integer rid;
{
   FLOW_WIN fw;
   FLOW_NODE fn,ofn;

   if (rid >= 0) MSGreply(rid,NULL);

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;

   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   fn = FLOW_node_find_function(fw,fil,fun);

   while (fn != NULL && !fn->display) fn = fn->parent;

   ofn = fw->execute;

   disp_execute(fw,fn);
   if (show_stack) disp_stack(fw,ofn,TRUE);

   ASHinput_unlock(fw->window);
};





static void
handle_exit_disp(sys,fil,fun,rid)
   String sys;
   String fil;
   String fun;
   Integer rid;
{
   FLOW_WIN fw;
   FLOW_NODE fn,ofn;

   if (rid >= 0) MSGreply(rid,NULL);

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;

   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   fn = FLOW_node_find_function(fw,fil,fun);

   while (fn != NULL && !fn->display) fn = fn->parent;

   if (fw->callstack != NULL) ofn = CAR(FLOW_NODE,fw->callstack);
   else ofn = fw->maindisp;

   if (show_stack) disp_stack(fw,ofn,FALSE);
   disp_execute(fw,ofn);

   ASHinput_unlock(fw->window);
};





/************************************************************************/
/*									*/
/*	handle_build_finish -- handle BUILD command finished for system */
/*									*/
/************************************************************************/


static void
handle_build_finish(sys,sts,rid)
   String sys;
   Integer sts;
   Integer rid;
{
   FLOW_WIN fw;

   if (rid >= 0) MSGreply(rid,NULL);

   if (!auto_update || sts > 0) return;

   fw = FLOW_find_system(sys);
   if (fw == NULL) return;

   if (ASHinput_lock(fw->window,FALSE) != ASH_LOCK_GRANTED) return;

   FLOW_node_setup(fw);
   FLOW_disp_reset(fw,TRUE);

   ASHinput_unlock(fw->window);
};





/************************************************************************/
/*									*/
/*	disp_execute -- do execution-style hilighting			*/
/*	disp_stack -- do stack style hilighting 			*/
/*									*/
/************************************************************************/


static void
disp_execute(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   if (fw->execute == fn) return;

   if (fw->execute != NULL) {
      GELOwindow_select(fw->disp_win,fw->execute->gid,execute_style,FALSE);
      fw->execute = NULL;
    };

   if (fn != NULL && fn->gid != NULL) {
      GELOwindow_select(fw->disp_win,fn->gid,execute_style,TRUE);
      fw->execute = fn;
    };
};





static void
disp_stack(fw,fn,pushfg)
   FLOW_WIN fw;
   FLOW_NODE fn;
   Boolean pushfg;
{
   Boolean fg;

   if (!pushfg) {
      if (fw->callstack != NULL) {
	 fn = CAR(FLOW_NODE,fw->callstack);
	 fw->callstack = CDRF(fw->callstack);
       }
      else fn = fw->maindisp;
      if (fn != NULL && !MEMQ(fn,fw->callstack) && fn != fw->execute) {
	 GELOwindow_select(fw->disp_win,fn->gid,stack_style,FALSE);
       }
    }
   else {
      fg = MEMQ(fn,fw->callstack);
      fw->callstack = CONS(fn,fw->callstack);
      if (!fg && fn != fw->execute && fn != NULL) {
	 GELOwindow_select(fw->disp_win,fn->gid,stack_style,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	compute_display -- compute which nodes should be displayed	*/
/*									*/
/************************************************************************/


static void
compute_display(fw)
   FLOW_WIN fw;
{
   if (!fw->level_valid && !fw->display_all && fw->selection != NULL) {
      compute_levels(fw);
    };

   if (!fw->conn_valid && fw->display_conn) {
      FLOW_node_connected(fw);
    };

   check_display(fw,fw->root,FALSE);
};





/************************************************************************/
/*									*/
/*	compute_levels -- find levels from given node			*/
/*	clear_levels -- reset all levels to max 			*/
/*									*/
/************************************************************************/


static void
compute_levels(fw)
   FLOW_WIN fw;
{
   register FLOW_NODE fna,fnb;
   register Integer i,j,k;
   Integer max;
   Boolean chng;

   clear_levels(fw->root,MAX_LEVEL+1);

   if (fw->selection == NULL) return;

   clear_levels(fw->selection,0);

   max = (fw->display_levels < 0 ? fw->numconn : fw->display_levels);

   for (i = 0; i < max; ++i) {
      chng = FALSE;
      for (j = 0; j < fw->numconn; ++j) {
	 fna = fw->conns[j].from;
	 fnb = fw->conns[j].to;
	 k = fna->call_level+1;
	 if (k > 0) {
	    if (fnb->call_level > k) {
	       chng = TRUE;
	       fnb->call_level = k;
	     };
	  };
	 k = fnb->callby_level+1;
	 if (k > 0) {
	    if (fna->callby_level > k) {
	       chng = TRUE;
	       fna->callby_level = k;
	     };
	  };
       };
      if (!chng) break;
    };

   propogate_levels(fw->root);
};





static void
clear_levels(fn,val)
   FLOW_NODE fn;
   Integer val;
{
   FLOW_NODE sf;

   fn->callby_level = val;
   fn->call_level = val;

   for (sf = fn->son; sf != NULL; sf = sf->brother) clear_levels(sf,val);
};





static void
propogate_levels(fn)
   FLOW_NODE fn;
{
   FLOW_NODE sf;

   for (sf = fn->son; sf != NULL; sf = sf->brother) {
      propogate_levels(sf);
      if (sf->call_level < fn->call_level) fn->call_level = sf->call_level;
      if (sf->callby_level < fn->callby_level) fn->callby_level = sf->callby_level;
    };
};





/************************************************************************/
/*									*/
/*	check_display -- check if node should be displayed		*/
/*									*/
/************************************************************************/


static void
check_display(fw,fn,fg)
   FLOW_WIN fw;
   FLOW_NODE fn;
   Boolean fg;
{
   FLOW_NODE sf;

   fn->gid = NULL;

   if (fn->ignore) fg = FALSE;

   if (fn == fw->topdisp) fg = TRUE;

   if (fg && !fw->display_all && fw->selection != NULL && fn->call_level > 0) {
      fg = FALSE;
      if (fw->display_callbys) {
	 if (fw->display_levels < 0 && fn->callby_level < MAX_LEVEL) fg = TRUE;
	 else if (fn->callby_level <= fw->display_levels) fg = TRUE;
       }
      if (fw->display_callees) {
	 if (fw->display_levels < 0 && fn->call_level < MAX_LEVEL) fg = TRUE;
	 else if (fn->call_level <= fw->display_levels) fg = TRUE;
       };
    };

   if (fg && fw->display_conn && !fn->connected) fg = FALSE;

   if (fg && fn->expand) {
      fn->display = FALSE;
    }
   else {
      fn->display = fg;
      fg = FALSE;
    };

   for (sf = fn->son; sf != NULL; sf = sf->brother) check_display(fw,sf,fg);
};





/************************************************************************/
/*									*/
/*	flow_auxd_list -- get list of AUXD strings as Sequence		*/
/*									*/
/************************************************************************/


static Sequence
flow_auxd_list(hdl,id)
   AUXD hdl;
   String id;
{
   String txt[1024];
   Integer ct,i;
   Sequence l;

   if (!AUXDget_defined(hdl,id)) return NULL;

   ct = AUXDget_info_list(hdl,id,1024,txt);

   l = NULL;
   for (i = ct-1; i >= 0; --i) l = CONS(txt[i],l);

   return l;
};





/* end of flowdisp.c */

