/************************************************************************/
/*									*/
/*		dispwin.c						*/
/*									*/
/*	Window manipulation routines for data structure display 	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "disp_local.h"
#include <leaf.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	void		check_windows();
static	int		disp_control();
static	int		disp_sub_control();
static	void		disp_remove();
static	Boolean 	request_variable();
static	DISP_TYPE	request_type();
static	int		apple_set_btn();
static	int		pear_set_btn();
static	void		update_handler();
static	void		build_done_handler();






/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define PEAR_HDR	1
#define PEAR_WIN	2
#define APPLE_HDR	3
#define APPLE_WIN	4



static	LEAF_DATA	disp_pear_leaf[] = {
   LEAF_ROOT(NULL),
   { PEAR_HDR, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	pear_set_btn, NULL },
   { PEAR_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(PEAR_HDR) },
	NULL, NULL },
   LEAF_END
};





static	LEAF_DATA	disp_apple_leaf[] = {
   LEAF_ROOT(NULL),
   { APPLE_HDR, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	apple_set_btn, NULL },
   { APPLE_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(APPLE_HDR) },
	NULL, NULL },
   LEAF_END
};




static	LEAF_DATA	disp_both_leaf[] = {
   LEAF_ROOT(NULL),
   { PEAR_HDR, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	pear_set_btn, NULL },
   { PEAR_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_REL_LINE(50),
	     LEAF_COORD_RX, LEAF_COORD_NEXT(PEAR_HDR) },
	NULL, NULL },
   { APPLE_HDR, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(PEAR_WIN) },
	apple_set_btn, NULL },
   { APPLE_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(APPLE_HDR) },
	NULL, NULL },
   LEAF_END
};






/************************************************************************/
/*									*/
/*	DISP_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DISP_win_init()
{
   AUXD hdl;
   String msg;

   all_windows = NULL;

   hdl = AUXDget_handle(NULL,"DISP");

   msg = AUXDget_info(hdl,"UPDATE_MSG");
   if (msg != NULL) MSGregister(msg,update_handler,1,NULL);

   if (hdl != NULL && AUXDget_defined(hdl,"BUILD_CLEAR")) {
      msg = AUXDget_info(hdl,"RESET_MSG");
      if (msg != NULL) MSGregister(msg,build_done_handler,0,NULL);
    };
};





/************************************************************************/
/*									*/
/*	DISP_create -- create a new dispation window			*/
/*									*/
/************************************************************************/


DISP_WIN
DISP_create(w,var,typ,pfg,afg)
   ASH_WINDOW w;
   String var;
   String typ;
   Boolean pfg;
   Boolean afg;
{
   DISP_WIN dw;
   DISP_TYPE dty;
   String s;

   dw = PALLOC(DISP_WIN_INFO);
   dw->window = w;

   if (afg && pfg) dw->show_type = DISP_SHOW_BOTH;
   else if (afg) dw->show_type = DISP_SHOW_APPLE;
   else dw->show_type = DISP_SHOW_PEAR;

   dw->system = NULL;
   dw->variable = NULL;
   dw->file = NULL;
   dw->func = NULL;
   dw->all_types = NULL;
   dw->var_addr = 0;
   dw->var_type = NULL;
   dw->type = NULL;
   dw->apple_win = NULL;
   dw->pear_win = NULL;
   dw->apple_id = NULL;
   dw->have_pear = FALSE;
   dw->nested = FALSE;
   dw->usedb = TRUE;
   dw->update = FALSE;

   s = ASHinq_window_resource(w,"simple");
   if (s != NULL && STRNEQ(s,"off")) dw->simple = TRUE;
   else dw->simple = FALSE;

   PROTECT;
   all_windows = CONS(dw,all_windows);
   UNPROTECT;

   if (pfg) {
      if (var != NULL) DISP_var_info(dw,NULL,NULL,var);
      if (dw->variable == NULL) {
	 if (!request_variable(dw)) {
	    all_windows = REMOB(dw,all_windows);
	    return NULL;
	  };
       };
    };

   if (afg) {
      if (typ != NULL) {
	 dty = DISP_find_type(dw,NULL,NULL,typ);
	 if (dty->class == DISP_TYPE_UNKNOWN) dty = dw->var_type;
       }
      else if (dw->var_type != NULL) dty = dw->var_type;
      else dty = NULL;

      if (dty == NULL) {
	 dty = request_type(dw);
	 if (dty == NULL) {
	    all_windows = REMOB(dw,all_windows);
	    return NULL;
	  };
       };

      dw->type = dty;
    };

   if (!setup_window(dw)) return NULL;

   return dw;
};





/************************************************************************/
/*									*/
/*	DISP_find_window -- find disp window given ash window		*/
/*	DISP_find_window_by_addr -- find disp window for dummy addr	*/
/*									*/
/************************************************************************/


DISP_WIN
DISP_find_window(w)
   ASH_WINDOW w;
{
   register Sequence l;
   register DISP_WIN dw;

   forin (dw,DISP_WIN,l,all_windows) {
      if (dw->window == w) break;
      else if (dw->apple_win == w) break;
      else if (dw->pear_win == w) break;
    };

   return dw;
};





DISP_WIN
DISP_find_window_by_addr(a)
   Integer a;
{
   register Sequence l;
   register DISP_WIN dw;

   forin (dw,DISP_WIN,l,all_windows) {
      if (dw->var_addr == a) break;
    };

   return dw;
};





/************************************************************************/
/*									*/
/*	DISP_find_defined_type -- see if type already defined		*/
/*									*/
/************************************************************************/


DISP_TYPE
DISP_find_defined_type(dw,name)
   DISP_WIN dw;
   String name;
{
   Sequence l,la;
   DISP_TYPE dt;
   DISP_WIN ndw;

   if (dw != NULL) {
      forin (dt,DISP_TYPE,l,dw->all_types) {
	 if (STREQL(name,dt->name) && dt->user) return dt;
       };
    };

   if (dw != NULL && dw->show_type == DISP_SHOW_APPLE) {
      forin (ndw,DISP_WIN,la,all_windows) {
	 if (ndw != dw && ndw->show_type == DISP_SHOW_PEAR) {
	    forin (dt,DISP_TYPE,l,ndw->all_types) {
	       if (STREQL(name,dt->name) && !dt->user) {
		  dw->all_types = CONS(dt,dw->all_types);
		  dt->user = TRUE;
		  ++dt->usecount;
		  return dt;
		};
	     };
	  };
       };
    }
   else {
      if (dw == NULL || dw->show_type == DISP_SHOW_PEAR) {
	 forin (ndw,DISP_WIN,la,all_windows) {
	    if (ndw != dw && ndw->show_type == DISP_SHOW_APPLE) {
	       forin (dt,DISP_TYPE,l,ndw->all_types) {
		  if (STREQL(name,dt->name)) {
		     dw->all_types = CONS(dt,dw->all_types);
		     dt->user = TRUE;
		     ++dt->usecount;
		     return dt;
		   };
		};
	     };
	  };
       };

      if (dw != NULL) {
	 forin (dt,DISP_TYPE,l,dw->all_types) {
	    if (STREQL(name,dt->name)) return dt;
	  };
       };
    };

   return NULL;
};





/************************************************************************/
/*									*/
/*	DISP_clear_type -- clear definitions for changed type		*/
/*									*/
/************************************************************************/


void
DISP_clear_type(t)
   DISP_TYPE t;
{
   DISP_WIN dw;
   DISP_TYPE dt,st;
   Sequence l,la;

   t->drawinfo = NULL;
   APPLEsave_clear(t);

   forin (dw,DISP_WIN,la,all_windows) {
      forin (dt,DISP_TYPE,l,dw->all_types) {
	 if (dt->class == DISP_TYPE_SAME && dt->drawinfo == NULL) {
	    for (st = dt->base; ; st = st->base) {
	       if (st == t) {
		  dt->drawinfo = NULL;
		  APPLEsave_clear(dt);
		  break;
		}
	       else if (st->class != DISP_TYPE_SAME || st->drawinfo != NULL)
		  break;
	     };
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of an disp window		*/
/*									*/
/************************************************************************/


static Boolean
setup_window(dw)
   DISP_WIN dw;
{
   Character buf[1024],tbuf[1024];
   String id,nm;
   AUXD hdl;

   switch (dw->show_type) {
      case DISP_SHOW_BOTH :
	 id = "disptype";
	 nm = "BOTH_NAME";
	 LEAFsetup_window(dw->window,disp_both_leaf,dw);
	 break;
      case DISP_SHOW_APPLE :
	 id = "typeedit";
	 nm = "TYPE_NAME";
	 LEAFsetup_window(dw->window,disp_apple_leaf,dw);
	 break;
      case DISP_SHOW_PEAR :
	 id = "display";
	 nm = "DISPLAY_NAME";
	 LEAFsetup_window(dw->window,disp_pear_leaf,dw);
	 break;
    };

   hdl = AUXDget_handle(NULL,"DISP");
   if (hdl != NULL && AUXDget_defined(hdl,nm)) {
      id = AUXDget_info(hdl,nm);
    };

   LEAFset_control(dw->window,disp_control);

   LEAFredraw(dw->window);

   tbuf[0] = 0;

   if (dw->show_type != DISP_SHOW_APPLE) {
      if (dw->variable != NULL && dw->var_type != NULL) {
	 sprintf(buf,"%s: %s",dw->variable,dw->var_type->name);
	 sprintf(tbuf,"%s: %s",id,dw->variable);
       }
      else
	 sprintf(buf,"VARIABLE: ??? <click here to set>");
      LEAFset_data(SALLOC(buf),dw->window,PEAR_HDR,0);
      dw->pear_win = LEAFinq_window(dw->window,PEAR_WIN,0);
      if (dw->pear_win == NULL) {
	 fprintf(stderr,"Window too small to run data structure displayer\n");
	 ASHremove(dw->window);
	 return FALSE;
       };
      ASHset_window_id(dw->pear_win,"display");
      ASHset_window_defaults(dw->pear_win);
      ASHset_control(dw->pear_win,disp_sub_control);
    };

   if (dw->show_type != DISP_SHOW_PEAR) {
      if (dw->type != NULL) {
	 sprintf(buf,"TYPE: %s",dw->type->name);
	 if (tbuf[0] == 0) sprintf(tbuf,"%s: %s",id,dw->type->name);
       }
      else
	 sprintf(buf,"TYPE: ??? <click here to set>");
      LEAFset_data(SALLOC(buf),dw->window,APPLE_HDR,0);
      dw->apple_win = LEAFinq_window(dw->window,APPLE_WIN,0);
      if (dw->apple_win == NULL) {
	 fprintf(stderr,"Window too small to run type picture editor\n");
	 ASHremove(dw->window);
	 return FALSE;
       };
      ASHset_window_id(dw->apple_win,"typeedit");
      ASHset_window_defaults(dw->apple_win);
      ASHset_control(dw->apple_win,disp_sub_control);
    };

   if (tbuf[0] != 0) ASHset_window_name(dw->window,tbuf);

   LEAFredraw(dw->window);

   if (dw->apple_win != NULL) DISP_apple_setup(dw);
   if (dw->pear_win != NULL) DISP_pear_setup(dw);

   LEAFset_refresh(dw->window,check_windows);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   DISP_WIN dw;
   ASH_WINDOW aw,pw;

   dw = DISP_find_window(w);

   if (dw->apple_win == NULL) aw = NULL;
   else aw = LEAFinq_window(w,APPLE_WIN,0);

   if (dw->pear_win == NULL) pw = NULL;
   else pw = LEAFinq_window(w,PEAR_WIN,0);

   if (aw != dw->apple_win || pw != dw->pear_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };

   if (dw->update) DISP_pear_update(dw);
};





/************************************************************************/
/*									*/
/*	disp_control -- handle ASH and other control messages		*/
/*	disp_sub_control -- handle sub-window control messages		*/
/*									*/
/************************************************************************/


static int
disp_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register DISP_WIN dw;

   dw = DISP_find_window(w);

   if (dw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      dw->apple_win = NULL;
      dw->pear_win = NULL;
      dw->have_pear = FALSE;
      disp_remove(dw);
    };

   return ASH_CONTROL_REJECT;
};





static int
disp_sub_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   DISP_WIN dw;

   dw = DISP_find_window(w);

   if (dw == NULL || !ASHinq_valid_window(dw->window)) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      dw->apple_win = NULL;
      dw->pear_win = NULL;
      dw->have_pear = FALSE;
      ASHremove(dw->window);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	disp_remove -- remove given disp				*/
/*									*/
/************************************************************************/


static void
disp_remove(dw)
   DISP_WIN dw;
{
   PROTECT;

   all_windows = REMOB(dw,all_windows);

   UNPROTECT;

   dw->window = NULL;
};





/************************************************************************/
/*									*/
/*	request_variable -- get variable name from user 		*/
/*									*/
/************************************************************************/


static Boolean
request_variable(dw)
   DISP_WIN dw;
{
   register Integer ct;
   String menu[40];
   Character buf1[256],buf2[256],buf3[256],sbuf[256];
   Integer nest,db;
   String s;

   ct = 0;
   menu[ct++] = "%CVariable to Display\n";
   menu[ct++] = "Variable: %2.38t";
   menu[ct++] = "\n   Defined in ...";

   if (!dw->simple) {
      menu[ct++] = "      File:    %0.32t";
      menu[ct++] = "      Routine: %1.32t";
      menu[ct++] = "\nSystem:   %5.32t";
    }
   else {
      menu[ct++] = "      Routine: %1.32t";
    };

   if (!dw->simple) {
      menu[ct++] = "\n%3o Nested display";
      menu[ct++] = "%4o Use database";
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   if (dw->file != NULL && STRNEQ(dw->file,"*")) {
      s = rindex(dw->file,'/');
      if (s == NULL) s = dw->file;
      else ++s;
      strcpy(buf1,s);
    }
   else if (!dw->simple) FIELDinq_file(buf1);
   else buf1[0] = 0;

   if (dw->func != NULL && STRNEQ(dw->func,"*")) strcpy(buf2,dw->func);
   else if (!dw->simple) FIELDinq_function(buf2);
   else buf2[0] = 0;

   if (dw->variable != NULL) strcpy(buf3,dw->variable);
/* else FIELDinq_selection(buf3);	*/
   else buf3[0] = 0;

   nest = dw->nested;
   db = dw->usedb;
   if (dw->system == NULL) sbuf[0] = 0;
   else strcpy(sbuf,dw->system);

   if (!STEMdialog(dw->window,menu,buf1,buf2,buf3,&nest,&db,sbuf)) return FALSE;

   dw->nested = nest;
   dw->usedb = db;
   if (dw->system != NULL) SFREE(dw->system);
   if (sbuf[0] == 0 || STREQL(sbuf,"*")) dw->system = NULL;
   else dw->system = SALLOC(sbuf);

   if (!DISP_var_info(dw,buf1,buf2,buf3)) {
      dw->variable = NULL;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	request_type -- get type to edit/display			*/
/*									*/
/************************************************************************/


static DISP_TYPE
request_type(dw)
   DISP_WIN dw;
{
   register Integer ct;
   String menu[40];
   DISP_TYPE ty;
   Character buf1[64],buf2[64],buf3[64],sbuf[64];
   Integer nest,db;

   for ( ; ; ) {
      ct = 0;
      menu[ct++] = "%CType to Display\n";
      menu[ct++] = "Type:     %2.32t";
      if (!dw->simple) {
	 menu[ct++] = "File:     %0.32t";
	 menu[ct++] = "Function: %1.32t";
	 menu[ct++] = "\nSystem:   %5.32t";
       };
      menu[ct++] = "\n%3o Nested default display";
      menu[ct++] = "%4o Use database";

      menu[ct++] = "\n   %a%M   %c";
      menu[ct] = 0;

      if (dw->file != NULL && STRNEQ(dw->file,"*")) strcpy(buf1,dw->file);
      else buf1[0] = 0;
      if (dw->func != NULL && STRNEQ(dw->func,"*")) strcpy(buf2,dw->func);
      else buf2[0] = 0;
      if (dw->type != NULL) strcpy(buf3,dw->type->name);
      else buf3[0] = 0;
      if (dw->type != NULL) nest = dw->type->nested;
      else nest = dw->nested;
      if (dw->type != NULL) db = dw->type->usedb;
      else db = dw->usedb;
      if (dw->system == NULL) sbuf[0] = 0;
      else strcpy(sbuf,dw->system);

      if (!STEMdialog(dw->window,menu,buf1,buf2,buf3,&nest,&db,sbuf)) return NULL;

      ty = DISP_find_type(dw,buf1,buf2,buf3);
      if (ty->class == DISP_TYPE_UNKNOWN) ty = NULL;
      if (ty != NULL) ty->nested = nest;
      if (ty != NULL) ty->usedb = db;

      if (dw->system != NULL) SFREE(dw->system);
      if (sbuf[0] == 0 || STREQL(sbuf,"*")) dw->system = NULL;
      else dw->system = SALLOC(sbuf);

      if (ty != NULL) break;
    };

   dw->file = (buf1[0] != 0 ? SALLOC(buf1) : NULL);
   dw->func = (buf2[0] != 0 ? SALLOC(buf2) : NULL);

   return ty;
};





/************************************************************************/
/*									*/
/*	apple_set_btn -- handle user changing types			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
apple_set_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   DISP_WIN dw;
   DISP_TYPE dt;
   Character buf[128];

   w = RIPinq_window(rgn);
   dw = DISP_find_window(w);

   if (dw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   dt = request_type(dw);

   if (dt != NULL && dt != dw->type) {
      dw->type = dt;
      DISP_apple_setup(dw);
      if (dw->type != NULL)
	 sprintf(buf,"TYPE: %s",dw->type->name);
      else
	 sprintf(buf,"TYPE: ");
      LEAFset_data(SALLOC(buf),dw->window,APPLE_HDR,0);
      LEAFredraw(dw->window);
    };

   return TRUE;
};




/************************************************************************/
/*									*/
/*	pear_set_btn -- handle user changing structures 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
pear_set_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   DISP_WIN dw;
   Character buf[10240];

   w = RIPinq_window(rgn);
   dw = DISP_find_window(w);

   if (dw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   if (request_variable(dw)) {
      DISP_pear_setup(dw);
      if (dw->window != NULL) {
	 if (dw->variable != NULL)
	    sprintf(buf,"%s: %s",dw->variable,dw->var_type->name);
	 else
	    sprintf(buf,"VARIABLE: ??? <click here to set>");
	 LEAFset_data(SALLOC(buf),dw->window,PEAR_HDR,0);
	 LEAFredraw(dw->window);
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	update_handler -- handle update requests			*/
/*									*/
/************************************************************************/


static void
update_handler(sys,rid)
   String sys;
   Integer rid;
{
   DISP_WIN dw;
   Sequence l;
   Character buf[1024];

   strcpy(buf,sys);

   if (rid >= 0) {
      if (all_windows != NULL) {
	 MSGsenda("DDTR SET %s PROG 1",buf);
       };
      MSGreply(rid,NULL);
    };

   forin (dw,DISP_WIN,l,all_windows) {
      if (FIELDsystem_match(dw->system,buf) && dw->pear_win != NULL) {
	 dw->update = TRUE;
	 if (ASHinq_visible(dw->pear_win)) DISP_pear_update(dw);
       };
    };

   if (rid >= 0 && all_windows != NULL) {
      MSGsenda("DDTR SET %s ENDPROG 1",buf);
    };
};





/************************************************************************/
/*									*/
/*	build_done_handler -- handle debug reset events 		*/
/*									*/
/************************************************************************/


static void
build_done_handler(rid)
   Integer rid;
{
   Sequence l,la;
   DISP_WIN ndw;
   DISP_TYPE dt;

   forin (ndw,DISP_WIN,la,all_windows) {
      forin (dt,DISP_TYPE,l,ndw->all_types) {
	 dt->drawinfo = NULL;
       };
    };

   APPLEsave_clear_all();

   if (rid >= 0) MSGreply(rid,NULL);
};





/* end of dispwin.c */


