/************************************************************************/
/*									*/
/*		dispdb.c						*/
/*									*/
/*	Database routines for data structure package			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "disp_local.h"



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_DB	"./.display_defs"

#define SEPARATOR	'`'





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	get_type_file();
static	void		output_type_data();
static	void		output_gelo();
static	void		output_component();
static	void		output_tile_arc();
static	void		output_tile_constraint();
static	void		output_value();
static	String		outstring();
static	APPLE_GELO_OBJECT input_type_data();
static	APPLE_GELO	input_gelo();
static	APPLE_GELO_COMPONENT input_component();
static	APPLE_GELO_TARC input_tile_arc();
static	APPLE_GELO_TCONST input_tile_constraint();
static	Boolean 	input_value();
static	String		instring();






/************************************************************************/
/*									*/
/*	DISP_db_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DISP_db_init()
{
};





/************************************************************************/
/*									*/
/*	DISP_db_store -- store type in database 			*/
/*									*/
/************************************************************************/


void
DISP_db_store(t)
   DISP_TYPE t;
{
   Character buf[256];
   FILE * otf;

   if (t->name == NULL || t->drawinfo == NULL) return;

   if (!get_type_file(t,buf)) return;

   otf = fopen(buf,"w");
   if (otf == NULL) return;

   output_type_data(otf,t->drawinfo,FALSE);

   fclose(otf);
};





/************************************************************************/
/*									*/
/*	DISP_db_load -- load type from database 			*/
/*									*/
/************************************************************************/


void
DISP_db_load(t)
   DISP_TYPE t;
{
   Character buf[256];
   FILE * inf;

   if (!get_type_file(t,buf)) return;

   inf = fopen(buf,"r");
   if (inf == NULL) return;

   t->drawinfo = input_type_data(inf,t,NULL);

   fclose(inf);
};





/************************************************************************/
/*									*/
/*	get_type_file -- get filename for a type			*/
/*									*/
/************************************************************************/


static Boolean
get_type_file(t,buf)
   DISP_TYPE t;
   String buf;
{
   String s;

   s = getenv("DISPLAY_DB");
   if (s == NULL) s = DEFAULT_DB;

   if (access(s,0) < 0) {
      mkdir(s,0777);
      if (access(s,0) < 0) return FALSE;
    };

   sprintf(buf,"%s/%s",s,t->name);

   for (s = buf; *s != 0; ++s) {
      if (*s == ' ') *s = '-';
      else if (*s == '*') *s = '@';
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	output_type_data -- write out type data 			*/
/*									*/
/************************************************************************/


static void
output_type_data(otf,ago,fg)
   FILE * otf;
   APPLE_GELO_OBJECT ago;
   Boolean fg;
{
   Sequence l;
   APPLE_GELO ag;

   if (ago->alternative != NULL) output_type_data(otf,ago->alternative,TRUE);

   fprintf(otf,"AGO %c%s%c %d %d %d %d %d %d\n",
	      SEPARATOR,outstring(ago->cond_match),SEPARATOR,
	      ago->cond_null, ago->cond_layout, ago->cond_nested, ago->cond_array,
	      LENGTH(ago->use), fg);

   forin (ag,APPLE_GELO,l,ago->use) {
      output_gelo(otf,ag);
    };

   output_gelo(otf,ago->object);
};





/************************************************************************/
/*									*/
/*	output_gelo -- output gelo object				*/
/*									*/
/************************************************************************/


static void
output_gelo(otf,ag)
   FILE * otf;
   APPLE_GELO ag;
{
   APPLE_GELO_BOX box;
   APPLE_GELO_LAYOUT lay;
   APPLE_GELO_ARC arc;
   APPLE_GELO_FIELD fld;
   APPLE_GELO_TILED til;
   APPLE_GELO ag1;
   APPLE_GELO_COMPONENT agt;
   APPLE_GELO_TARC tarc;
   APPLE_GELO_TCONST tcon;
   Sequence l;

   if (ag == NULL) {
      fprintf(otf,"OBJ %d %d %d\n",APPLE_MODE_NONE,0,0);
      return;
    };

   fprintf(otf,"OBJ %d %d %d\n",ag->mode,ag->fix_x,ag->fix_y);

   switch (ag->mode) {
      default :
      case APPLE_MODE_IGNORE :
	 break;

      case APPLE_MODE_BOX :
	 box = (APPLE_GELO_BOX) ag;
	 fprintf(otf,"BOX %d %d %d %d %d\n",
		    box->fill,box->shape,box->invert,box->linestyle,box->font);
	 output_value(otf,&box->value);
	 break;

      case APPLE_MODE_LAYOUT :
	 lay = (APPLE_GELO_LAYOUT) ag;
	 fprintf(otf,"LAY %d %d %d %d %d %d %d %d %d %d %d\n",
		    lay->method,lay->connmethod,lay->boxed,lay->white_space,
		    lay->shrinkage,lay->unique,lay->fixed,lay->standard,
		    lay->centered,lay->simple,LENGTH(lay->components));
	 forin (ag1,APPLE_GELO,l,lay->components) {
	    output_gelo(otf,ag1);
	  };
	 break;

      case APPLE_MODE_ARC :
      case APPLE_MODE_POINTER :
	 arc = (APPLE_GELO_ARC) ag;
	 fprintf(otf,"ARC %d %d %d %d %d %d %d %c%s%c\n",
		    arc->fport,arc->tport,arc->style,arc->arrow,
		    arc->label_loc,arc->label_pos,arc->label_box,
		    SEPARATOR,outstring(arc->match),SEPARATOR);
	 output_value(otf,&arc->from);
	 output_value(otf,&arc->to);
	 output_value(otf,&arc->label);
	 break;

      case APPLE_MODE_FIELD :
	 fld = (APPLE_GELO_FIELD) ag;
	 fprintf(otf,"FLD %d %c%s%c\n",
		    fld->fport,SEPARATOR,outstring(fld->match),SEPARATOR);
	 output_value(otf,&fld->value);
	 break;

      case APPLE_MODE_TILED :
	 til = (APPLE_GELO_TILED) ag;
	 fprintf(otf,"TIL %d %d %d %d %d %d\n",
		    til->max_x,til->max_y,til->boxed,
		    LENGTH(til->components),LENGTH(til->arcs),
		    LENGTH(til->constraints));
	 forin (agt,APPLE_GELO_COMPONENT,l,til->components) {
	    output_component(otf,agt);
	  };
	 forin (tarc,APPLE_GELO_TARC,l,til->arcs) {
	    output_tile_arc(otf,tarc,til->components);
	  };
	 forin (tcon,APPLE_GELO_TCONST,l,til->constraints) {
	    output_tile_constraint(otf,tcon,til->components);
	  };
	 break;
    };
};





/************************************************************************/
/*									*/
/*	output_component -- component of tiling 			*/
/*	output_tile_arc -- arc of tiling				*/
/*	output_tile_constraint -- constraint of tiling			*/
/*	componentof -- index of component				*/
/*									*/
/************************************************************************/


static void
output_component(otf,c)
   FILE * otf;
   APPLE_GELO_COMPONENT c;
{
   fprintf(otf,"CMP %d %d %d %d %d %d %d %f %f %f %f\n",
	      c->lx,c->by,c->rx,c->ty,c->fix_x,c->fix_y,c->shrinkage,
	      c->expand_x,c->expand_y,c->priority_x,c->priority_y);
   output_gelo(otf,c->object);
};





static void
output_tile_arc(otf,a,cl)
   FILE * otf;
   APPLE_GELO_TARC a;
   Sequence cl;
{
   fprintf(otf,"TAR %d %d %d %d %d %d\n",
	      a->fport,a->tport,componentof(a->from,cl),componentof(a->to,cl),
	      a->style,a->arrow);
};





static void
output_tile_constraint(otf,c,cl)
   FILE * otf;
   APPLE_GELO_TCONST c;
   Sequence cl;
{
   fprintf(otf,"CON %d %d %d %d %f %f\n",
	      componentof(c->from,cl),componentof(c->to,cl),
	      c->from_y,c->to_y,c->multiplier,c->addend);
};






static Integer
componentof(c,cl)
   APPLE_GELO_COMPONENT c;
   Sequence cl;
{
   Integer i;
   APPLE_GELO_COMPONENT c1;
   Sequence l;

   i = 0;
   forin (c1,APPLE_GELO_COMPONENT,l,cl) {
      if (c1  == c) break;
      ++i;
    };

   return i;
};






/************************************************************************/
/*									*/
/*	output_value -- output an APPLE_VALUE item			*/
/*									*/
/************************************************************************/


static void
output_value(otf,v)
   FILE * otf;
   APPLE_VALUE v;
{
   DISP_FIELD df;

   if (v->name != NULL &&
	  (STREQL(v->name,APPLE_SELF_NAME) ||
	      STREQL(v->name,APPLE_DATA_NAME) ||
	      STREQL(v->name,APPLE_INDEX_NAME) ||
	      v->value == NULL)) {
      fprintf(otf,"VAL 0 %c%s%c\n",SEPARATOR,v->name,SEPARATOR);
    }
   else if (v->value == ((int) v->name)) {
      fprintf(otf,"VAL 1 %c%s%c\n",SEPARATOR,outstring(v->name),SEPARATOR);
    }
   else {
      df = (DISP_FIELD) v->value;
      fprintf(otf,"VAL 2 %s\n",df->name);
    };
};





/************************************************************************/
/*									*/
/*	outstring -- handle null strings				*/
/*									*/
/************************************************************************/


static String
outstring(s)
   String s;
{
   if (s == NULL) return "";

   return s;
};




/************************************************************************/
/*									*/
/*	input_type_data -- read type data from saved file		*/
/*									*/
/************************************************************************/


static APPLE_GELO_OBJECT
input_type_data(inf,t,alt)
   FILE * inf;
   DISP_TYPE t;
   APPLE_GELO_OBJECT alt;
{
   Sequence l;
   APPLE_GELO ag;
   APPLE_GELO_OBJECT ago;
   Integer i;
   Integer v1,v2,v3,v4,v5,v6;
   String s;

   ago = PALLOC(APPLE_GELO_OBJECT_B);
   ago->cond_fct = NULL;

   fscanf(inf,"AGO ");
   s = instring(inf);
   i = fscanf(inf," %d %d %d %d %d %d\n",&v1,&v2,&v3,&v4,&v5,&v6);
   if (i < 5) return NULL;

   if (i == 5) {				/* handle old files	*/
      v6 = v5;
      v5 = v4;
      v4 = FALSE;
    };

   ago->cond_match = s;
   ago->cond_null = v1;
   ago->cond_layout = v2;
   ago->cond_nested = v3;
   ago->cond_array = v4;

   l = NULL;
   for (i = 0; i < v5; ++i) {
      ag = input_gelo(inf,t);
      l = APPEND(ag,l);
    };
   ago->use = l;

   ago->object = input_gelo(inf,t);
   if (ago->object == NULL) return NULL;
   ago->alternative = alt;

   if (v6) ago = input_type_data(inf,t,ago);

   return ago;
};





/************************************************************************/
/*									*/
/*	input_gelo -- input APPLE_GELO object				*/
/*									*/
/************************************************************************/


static APPLE_GELO
input_gelo(inf,t)
   FILE * inf;
   DISP_TYPE t;
{
   APPLE_GELO ag;
   APPLE_GELO_BOX box;
   APPLE_GELO_LAYOUT lay;
   APPLE_GELO_ARC arc;
   APPLE_GELO_FIELD fld;
   APPLE_GELO_TILED til;
   APPLE_GELO_COMPONENT agt;
   APPLE_GELO_TARC tarc;
   APPLE_GELO_TCONST tcon;
   Integer v1,v2,v3,v4,v5,v6,v7,v8,v9,v10,v11,v12,v13,v14;
   Integer i;
   String s1;

   if (fscanf(inf,"OBJ %d %d %d\n",&v1,&v2,&v3) != 3) return NULL;

   switch ((APPLE_MODE) v1) {
      case APPLE_MODE_NONE :
	 return NULL;

      default :
      case APPLE_MODE_IGNORE :
	 ag = PALLOC(APPLE_GELO_B);
	 break;

      case APPLE_MODE_BOX :
	 box = PALLOC(APPLE_GELO_BOX_B);
	 ag = (APPLE_GELO) box;
	 if (fscanf(inf,"BOX %d %d %d %d %d\n", &v4,&v5,&v6,&v7,&v8) != 5) return NULL;
	 box->fill = v4;
	 box->shape = (GELO_SHAPE) v5;
	 box->invert = v6;
	 box->linestyle = v7;
	 box->font = v8;
	 if (!input_value(inf,&box->value,t)) return NULL;
	 break;

      case APPLE_MODE_LAYOUT :
	 lay = PALLOC(APPLE_GELO_LAYOUT_B);
	 ag = (APPLE_GELO) lay;
	 if (fscanf(inf,"LAY %d %d %d %d %d %d %d %d %d %d %d\n",
		   &v4,&v5,&v6,&v7,&v8,&v9,&v10,&v11,&v12,&v13,&v14) != 11) return NULL;
	 lay->method = (GELO_METHOD) v4;
	 lay->connmethod = (GELO_CONN_METHOD) v5;
	 lay->boxed = v6;
	 lay->white_space = v7;
	 lay->shrinkage = v8;
	 lay->unique = v9;
	 lay->fixed = v10;
	 lay->standard = v11;
	 lay->centered = v12;
	 lay->simple = v13;

	 lay->components = NULL;
	 for (i = 0; i < v14; ++i) {
	    lay->components = APPEND(input_gelo(inf,t),lay->components);
	  };
	 break;

      case APPLE_MODE_ARC :
      case APPLE_MODE_POINTER :
	 arc = PALLOC(APPLE_GELO_ARC_B);
	 ag = (APPLE_GELO) arc;
	 if (fscanf(inf,"ARC %d %d %d %d %d %d %d ",&v4,&v5,&v6,&v7,&v8,&v9,&v10) != 7)
	    return NULL;
	 s1 = instring(inf);
	 fscanf(inf,"\n");
	 arc->fport = (GELO_PORT) v4;
	 arc->tport = (GELO_PORT) v5;
	 arc->style = (ASH_LINE_STYLE) v6;
	 arc->arrow = (GELO_ARC_ARROW) v7;
	 arc->label_loc = (GELO_ARC_LABEL_LOC) v8;
	 arc->label_pos = (GELO_ARC_LABEL_POS) v9;
	 arc->label_box = v10;
	 arc->match = s1;
	 if (!input_value(inf,&arc->from,t)) return NULL;
	 if (!input_value(inf,&arc->to,t)) return NULL;
	 if (!input_value(inf,&arc->label,t)) return NULL;
	 break;

      case APPLE_MODE_FIELD :
	 fld = PALLOC(APPLE_GELO_FIELD_B);
	 ag = (APPLE_GELO) fld;
	 if (fscanf(inf,"FLD %d ",&v4) != 1) return NULL;
	 s1 = instring(inf);
	 fscanf(inf,"\n");
	 fld->fport = (GELO_PORT) v4;
	 fld->match = s1;
	 if (!input_value(inf,&fld->value,t)) return NULL;
	 break;

      case APPLE_MODE_TILED :
	 til = PALLOC(APPLE_GELO_TILED_B);
	 ag = (APPLE_GELO) til;
	 if (fscanf(inf,"TIL %d %d %d %d %d %d\n",&v4,&v5,&v6,&v7,&v8,&v9) != 6)
	    return NULL;
	 til->max_x = v4;
	 til->max_y = v5;
	 til->boxed = v6;

	 til->components = NULL;
	 for (i = 0; i < v7; ++i) {
	    agt = input_component(inf,t);
	    if (agt == NULL) return NULL;
	    til->components = APPEND(agt,til->components);
	  };

	 til->arcs = NULL;
	 for (i = 0; i < v8; ++i) {
	    tarc = input_tile_arc(inf,til->components);
	    if (tarc == NULL) return NULL;
	    til->arcs = APPEND(tarc,til->arcs);
	  };

	 til->constraints = NULL;
	 for (i = 0; i < v9; ++i) {
	    tcon = input_tile_constraint(inf,til->components);
	    if (tcon == NULL) return NULL;
	    til->constraints = APPEND(tcon,til->constraints);
	  };
	 break;
    };

   ag->mode = (APPLE_MODE) v1;
   ag->fix_x = v2;
   ag->fix_y = v3;

   return ag;
};





/************************************************************************/
/*									*/
/*	input_component -- component of tiling				*/
/*	input_tile_arc -- arc of tiling 				*/
/*	input_tile_constraint -- constraint of tiling			*/
/*	componentof -- index of component				*/
/*									*/
/************************************************************************/


static APPLE_GELO_COMPONENT
input_component(inf,t)
   FILE * inf;
   DISP_TYPE t;
{
   APPLE_GELO_COMPONENT c;
   Integer v1,v2,v3,v4,v5,v6,v7;
   float f1,f2,f3,f4;

   c = PALLOC(APPLE_GELO_COMPONENT_B);

   if (fscanf(inf,"CMP %d %d %d %d %d %d %d %f %f %f %f\n",
		 &v1,&v2,&v3,&v4,&v5,&v6,&v7,&f1,&f2,&f3,&f4) != 11) return NULL;

   c->lx = v1;
   c->by = v2;
   c->rx = v3;
   c->ty = v4;
   c->fix_x = v5;
   c->fix_y = v6;
   c->shrinkage = v7;
   c->expand_x = f1;
   c->expand_y = f2;
   c->priority_x = f3;
   c->priority_y = f4;

   c->object = input_gelo(inf,t);

   return c;
};





static APPLE_GELO_TARC
input_tile_arc(inf,cl)
   FILE * inf;
   Sequence cl;
{
   APPLE_GELO_TARC a;
   Integer v1,v2,v3,v4,v5,v6;

   a = PALLOC(APPLE_GELO_TARC_B);

   if (fscanf(inf,"TAR %d %d %d %d %d %d\n",&v1,&v2,&v3,&v4,&v5,&v6) != 6) return NULL;
   a->fport = (GELO_PORT) v1;
   a->tport = (GELO_PORT) v2;
   a->from = NTH(APPLE_GELO_COMPONENT,cl,v3+1);
   a->to = NTH(APPLE_GELO_COMPONENT,cl,v4+1);
   a->style = (ASH_LINE_STYLE) v5;
   a->arrow = (GELO_ARC_ARROW) v6;

   return a;
};





static APPLE_GELO_TCONST
input_tile_constraint(inf,cl)
   FILE * inf;
   Sequence cl;
{
   APPLE_GELO_TCONST c;
   Integer v1,v2,v3,v4;
   float f1,f2;

   c = PALLOC(APPLE_GELO_TCONST_B);

   if (fscanf(inf,"CON %d %d %d %d %f %f\n",&v1,&v2,&v3,&v4,&f1,&f2) != 6) return NULL;
   c->from = NTH(APPLE_GELO_COMPONENT,cl,v1+1);
   c->to = NTH(APPLE_GELO_COMPONENT,cl,v2+1);
   c->from_y = v3;
   c->to_y = v4;
   c->multiplier = f1;
   c->addend = f2;

   return c;
};






/************************************************************************/
/*									*/
/*	input_value -- input an APPLE_VALUE item			*/
/*									*/
/************************************************************************/


static Boolean
input_value(inf,v,t)
   FILE * inf;
   APPLE_VALUE v;
   DISP_TYPE t;
{
   Integer v1;
   Character sbuf[1024];
   DISP_FIELD df;
   Sequence l;

   if (fscanf(inf,"VAL %d ",&v1) != 1) return FALSE;

   switch (v1) {
      case 0 :
	 v->name = instring(inf);
	 fscanf(inf,"\n");
	 v->value = 0;
	 break;
      case 1 :
	 v->name = instring(inf);
	 fscanf(inf,"\n");
	 v->value = (int) v->name;
	 break;
      case 2 :
	 if (fscanf(inf,"%s\n",sbuf) != 1) return FALSE;
	 while (t != NULL && t->class != DISP_TYPE_RECORD) {
	    switch (t->class) {
	       case DISP_TYPE_RECORD :
		  break;
	       case DISP_TYPE_SAME :
		  t = t->base;
		  break;
	       case DISP_TYPE_POINTER :
		  t = t->base;
		  break;
	       default :
		  t = NULL;
	     };
	  };
	 if (t == NULL) return FALSE;
	 forin (df,DISP_FIELD,l,t->fields) {
	    if (STREQL(df->name,sbuf)) break;
	  };
	 v->name = SALLOC(sbuf);
	 if (df != NULL) {
	    v->value = (Integer) df;
	  }
	 else {
	    v->value = (int) v->name;
	  };
	 break;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	instring -- handle null strings 			       */
/*									*/
/************************************************************************/


static String
instring(inf)
   FILE * inf;
{
   Integer ch,i,c;
   Character buf[1024];
   String s;

   ch = getc(inf);
   i = 0;

   s = buf;
   while ((c = getc(inf)) >= 0 && c != ch) {
      *s++ = c;
      ++i;
    };

   if (i == 0) return NULL;

   *s = 0;
   return SALLOC(buf);
};




/* end of dispdb.c */
