/************************************************************************/
/*									*/
/*		ddtstate.c						*/
/*									*/
/*	Run time and debugger state facilities for ddt interface	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef enum _DEBUG_STATE {		/* state of system debugger	*/
   DEBUG_INIT,
   DEBUG_READY,
   DEBUG_RUN,
   DEBUG_CMD,
   DEBUG_RUN_CMD,			/* command that might run code	*/
   DEBUG_ABORT,
} DEBUG_STATE;




typedef enum _DDT_STATE {		/* our local state		*/
   DDT_INIT,
   DDT_READY,
   DDT_CMD,
   DDT_RUN,
   DDT_DEBUG,
   DDT_ABORT,
} DDT_STATE;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	DEBUG_STATE	debug_state;
static	DDT_STATE	ddt_state;
static	Boolean 	stop_flag;
static	Boolean 	prompt;
static	Boolean 	stop_expected;
static	Integer 	user_level;
static	STOP_ACTION	stop_action;
static	Integer 	stop_signal;

static	String		response;
static	Integer 	resp_size;
static	Integer 	resp_len;
static	Boolean 	resp_ok;

static	Integer 	resp_id;
static	Boolean 	resp_quiet;
static	Integer 	old_resp_id;
static	Boolean 	old_resp_quiet;
static	String		user_reply;
static	Integer 	user_size;
static	Integer 	user_len;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		debug_command();





/************************************************************************/
/*									*/
/*	DDT_state_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
DDT_state_init()
{
   ddt_state = DDT_INIT;
   debug_state = DEBUG_INIT;

   prompt = FALSE;
   stop_expected = FALSE;
   user_level = 0;
   stop_flag = FALSE;
   stop_action = STOP_UNDEFINED;
   stop_signal = 0;

   resp_size = 1024;
   response = (String) malloc(resp_size);
   resp_len = 0;
   response[0] = 0;
   resp_ok = FALSE;

   resp_id = -1;
   resp_quiet = FALSE;
   user_size = 1024;
   user_reply = (String) malloc(user_size);
   user_len = 0;
   user_reply[0] = 0;
};





/************************************************************************/
/*									*/
/*	DDTstate_loop -- main loop for ddt				*/
/*									*/
/************************************************************************/


void
DDT_state_loop()
{
   for ( ; ; ) {
      DDT_state_next();
    };
};





/************************************************************************/
/*									*/
/*	DDT_state_next -- process next event				*/
/*									*/
/************************************************************************/


void
DDT_state_next()
{
   if (prompt) {
      if (stop_expected && ddt_state != DDT_RUN) {
	 DDT_command_abort();
	 ddt_state = DDT_RUN;
       };
      prompt = FALSE;
      stop_expected = FALSE;
      resp_ok = TRUE;
      switch (ddt_state) {
	 case DDT_INIT :
	    ddt_state = DDT_CMD;
	    DDT_model_clear(TRUE);
	    DDT_x_startup();
	    DDT_lang_startup();
	    break;
	 case DDT_RUN :
	    ddt_state = DDT_CMD;
	    DDT_trace_process();
	    stop_flag = TRUE;
	    break;
	 default :
	    break;
       };
      ddt_state = DDT_READY;
      return;
    };

   if (stop_flag && ddt_state == DDT_READY) {
      if (DDT_event_process()) return;
      if (stop_action == STOP_STOP || stop_action == STOP_UNDEFINED) {
	 LOCATION floc,eloc;
	 Integer addr;

	 DDT_model_inq_location(&floc);
	 DDT_model_inq_exec_location(&eloc,&addr);
	 DDT_mprint("Stopped");
	 DDT_print_exec_position();
	 DDT_mprint("\n");
	 DDT_x_current_addr(FALSE);

	 if (((floc.file == NULL && eloc.file == NULL) ||
		 (floc.file != NULL && eloc.file != NULL &&
		     STREQL(floc.file,eloc.file))) &&
		floc.line == eloc.line &&
		floc.func == NULL && eloc.func != NULL)
	    floc.func = SALLOC(eloc.func);

	 if (((floc.file == NULL && eloc.file == NULL) ||
		 (floc.file != NULL && eloc.file != NULL && STREQL(floc.file,eloc.file))) &&
		((floc.func == NULL && eloc.func == NULL) ||
		 (floc.func != NULL && eloc.func != NULL && STREQL(floc.func,eloc.func))) &&
		floc.line == eloc.line) ;
	 else {
	    DDT_mprint("Focus is");
	    DDT_print_current_position();
	    DDT_mprint("\n");
	  };

	 DDT_x_current_line();
	 DDT_event_stop(&eloc);
	 DDT_state_cmd_end(resp_quiet,resp_id);
	 stop_flag = FALSE;
	 return;
       }
      else if (user_level != 0) {
	 if (DDT_command_process(FALSE)) return;
       }
      else {
	 if (DDT_command_process(FALSE)) return;
	 switch (stop_action) {
	    case STOP_STEP :
	       stop_action = STOP_UNDEFINED;
	       DDT_x_step(1,FALSE,FALSE,FALSE,FALSE);
	       break;
	    case STOP_NEXT :
	       stop_action = STOP_UNDEFINED;
	       DDT_x_step(1,TRUE,FALSE,FALSE,FALSE);
	       break;
	    case STOP_CONTINUE :
	       stop_action = STOP_UNDEFINED;
	       DDT_x_continue(1,stop_signal);
	       break;
	  };
       };
    }
   else if (ddt_state == DDT_READY) {
      if (DDT_command_process(FALSE)) return;
      if (user_level == 0 && DDT_command_process(TRUE)) return;
    };

   CMPXselect(0);
};





/************************************************************************/
/*									*/
/*	DDT_state_prompt -- check for prompt and handle if so		*/
/*									*/
/************************************************************************/


Boolean
DDT_state_prompt(text)
   String text;
{
   if (DDT_x_check_query(text)) return TRUE;

   if (!DDT_x_check_prompt(text)) return FALSE;

   prompt = TRUE;
   debug_state = DEBUG_READY;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_state_abort -- debugger aborted				*/
/*									*/
/************************************************************************/


void
DDT_state_abort()
{
   debug_state = DEBUG_ABORT;
   prompt = TRUE;
   resp_ok = TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_state_stop_expected -- indicate stop expected		*/
/*	DDT_state_user_level -- increase/decrease user level		*/
/*	DDT_state_stop_action -- set default stop action		*/
/*									*/
/************************************************************************/


void
DDT_state_stop_expected()
{
   stop_expected = TRUE;
};





void
DDT_state_user_level(fg)
   Boolean fg;
{
   if (fg) ++user_level;
   else if (user_level > 0) --user_level;
};




void
DDT_state_stop_action(act)
   STOP_ACTION act;
{
   if (((int) act) > ((int) stop_action)) stop_action = act;
   stop_signal = 0;
};





void
DDT_state_stop_signal(sig)
   Integer sig;
{
   DDT_state_stop_action(STOP_CONTINUE);

   if (stop_action == STOP_CONTINUE) {
      stop_signal = sig;
    };
};





/************************************************************************/
/*									*/
/*	DDT_state_cmd_begin -- start processing a DDT command		*/
/*	DDT_state_cmd_end -- finish processing a DDT command		*/
/*	DDT_state_cmd_abort -- finish processing DDT command w/ failure */
/*	DDT_state_cmd_event -- start/end event processing state 	*/
/*									*/
/************************************************************************/


void
DDT_state_cmd_begin(quiet,rid,qid,ctx)
   Boolean quiet;
   Integer rid;
   Integer qid;
   String ctx;
{
   old_resp_id = resp_id;
   old_resp_quiet = resp_quiet;

   ddt_state = DDT_CMD;
   resp_id = rid;
   resp_quiet = quiet;

   DDT_output_set(qid);
   DDT_x_set_context(qid,ctx);
};






/*ARGSUSED*/

void
DDT_state_cmd_end(quiet,rid)
   Boolean quiet;
   Integer rid;
{
   if (rid >= 0) {
      if (user_reply[0] == 0) MSGreply(rid,NULL);
      else MSGreply(rid,user_reply);
    };

   user_len = 0;
   user_reply[0] = 0;

   ddt_state = DDT_READY;

   resp_id = old_resp_id;
   resp_quiet = old_resp_quiet;

   DDT_output_set(0);
   DDT_x_set_context(0,NULL);
};





void
DDT_state_cmd_abort(quiet,rid)
   Boolean quiet;
   Integer rid;
{
   user_reply[0] = 0;

   DDT_state_cmd_end(quiet,rid);

   resp_id = old_resp_id;
   resp_quiet = old_resp_quiet;
};





void
DDT_state_cmd_event(fg)
   Boolean fg;
{
   ddt_state = (fg ? DDT_CMD : DDT_READY);
};





Boolean
DDT_state_cmd_quiet()
{
   return resp_quiet;
};





/************************************************************************/
/*									*/
/*	DDT_debug_command -- send command to the remote debugger	*/
/*	DDT_debug_call_command -- send call command to the remote dbgr	*/
/*	DDT_state_run_command -- send run command to the remote dbgr	*/
/*									*/
/************************************************************************/


/*VARARGS1*/

String
DDT_debug_command(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   String s;

   s = debug_command(DEBUG_CMD,pat,a1,a2,a3,a4,a5,a6,a7,a8);

   return s;
};





/*VARARGS1*/

String
DDT_debug_call_command(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   String s;

   s = debug_command(DEBUG_RUN_CMD,pat,a1,a2,a3,a4,a5,a6,a7,a8);

   return s;
};





static String
debug_command(st,pat,a1,a2,a3,a4,a5,a6,a7,a8)
   DEBUG_STATE st;
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   DDT_STATE old_state;
   STOP_ACTION old_stop;
   Integer old_stopsig;

   resp_len = 0;
   response[0] = 0;
   resp_ok = FALSE;

   debug_state = st;
   old_state = ddt_state;
   ddt_state = DDT_DEBUG;
   old_stop = stop_action;
   old_stopsig = stop_signal;
   stop_action = STOP_UNDEFINED;
   stop_signal = 0;

   DDT_printf(pat,a1,a2,a3,a4,a5,a6,a7,a8);

   while (!resp_ok) {
      DDT_state_next();
    };

   resp_ok = FALSE;

   if (debug_state != DEBUG_ABORT) ddt_state = old_state;
   stop_action = old_stop;
   stop_signal = old_stopsig;

   if (resp_len == 0) return NULL;
   else if (response[resp_len-1] == '\n') {
      response[resp_len-1] = 0;
    };

   return response;
};





/*VARARGS1*/

void
DDT_state_run_command(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   resp_len = 0;
   response[0] = 0;

   user_len = 0;
   user_reply[0] = 0;

   debug_state = DEBUG_RUN;
   ddt_state = DDT_RUN;
   stop_action = STOP_UNDEFINED;
   stop_signal = 0;

   DDT_output_set(0);
   DDT_x_set_context(0,NULL);

   DDT_printf(pat,a1,a2,a3,a4,a5,a6,a7,a8);
};





void
DDT_state_step_command(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   resp_len = 0;
   response[0] = 0;

   user_len = 0;
   user_reply[0] = 0;

   debug_state = DEBUG_RUN;
   ddt_state = DDT_RUN;
   stop_action = STOP_STOP;
   stop_signal = 0;

   DDT_output_set(0);
   DDT_x_set_context(0,NULL);

   DDT_printf(pat,a1,a2,a3,a4,a5,a6,a7,a8);
};





/************************************************************************/
/*									*/
/*	DDT_state_stop_run -- stop program from running 		*/
/*									*/
/************************************************************************/


void
DDT_state_stop_run()
{
   stop_action = STOP_STOP;

   if (debug_state == DEBUG_INIT) return;

   if (debug_state != DEBUG_READY) {
      DDT_send_interrupt();
    };
};





/************************************************************************/
/*									*/
/*	DDT_state_user_reply -- handle possible reply message		*/
/*									*/
/************************************************************************/


Boolean
DDT_state_user_reply(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   Character text[40960];
   register Integer ln;

   if (!resp_quiet) return FALSE;

   sprintf(text,pat,a1,a2,a3,a4,a5,a6,a7,a8);

   ln = strlen(text);
   if (user_len + ln >= user_size) {
      while (user_len + ln >= user_size) {
	 user_size *= 2;
       };
      user_reply = (String) realloc(user_reply,user_size);
    };
   strcat(user_reply,text);
   user_len += ln;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_state_inq_run -- check if we are running now		*/
/*									*/
/************************************************************************/


Boolean
DDT_state_inq_run()
{
   register Boolean fg;

   switch (ddt_state) {
      case DDT_INIT :
      case DDT_READY :
      case DDT_CMD :
      case DDT_DEBUG :
	 fg = FALSE;
	 break;
      case DDT_RUN :
	 fg = TRUE;
	 break;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_state_msg -- handle interpreting message from debugger	*/
/*									*/
/************************************************************************/


void
DDT_state_msg(text)
   String text;
{
   Integer ln;

   if ((debug_state == DEBUG_RUN || debug_state == DEBUG_RUN_CMD ||
	   debug_state == DEBUG_INIT) &&
	  DDT_x_run_monitor(text)) ;
   else if (debug_state == DEBUG_CMD || debug_state == DEBUG_RUN_CMD) {
      ln = strlen(text);
      if (resp_len + ln + 1 >= resp_size) {
	 while (resp_len + ln + 1 >= resp_size) {
	    resp_size *= 2;
	  };
	 response = (String) realloc(response,resp_size);
       };
      if (resp_len != 0) {
	 strcat(response,"\n");
	 resp_len += 1;
       };
      strcat(response,text);
      resp_len += ln;
    }
   else {
      DDT_msg_debug_out(text);
    };
};





/* end of ddtstate.c */

