/************************************************************************/
/*									*/
/*		ddtmap.c						*/
/*									*/
/*	Language-specific name mapping routines 			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"
#include "ddt_names.h"





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		fix_outexpr();
static	String		replace_id();





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	DDT_map_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_map_init()
{
   DDT_pascal_init();
   DDT_cplus12_init();
   DDT_cplus_init();
};






/************************************************************************/
/*									*/
/*	DDT_map_fix_function -- convert user function name to dbx name	*/
/*									*/
/************************************************************************/


String
DDT_map_fix_function(f,buf)
   String f;
   String buf;
{
   Character tbuf[1024];
   Integer i;
   Boolean fg;

   if (f == NULL) return NULL;

   fg = FALSE;

   if (f[0] == '\'') {
      strcpy(tbuf,&f[1]);
      i = strlen(tbuf);
      while (isspace(tbuf[i-1])) --i;
      if (tbuf[i-1] == '\'') tbuf[i-1] = 0;
      f = tbuf;
      fg = TRUE;
    };

   if (DDT__stdpascal) {
      f = DDT_pascal_fix_function(f,buf);
    }
   else if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) f = DDT_cplus12_fix_function(f,buf);
      else f = DDT_cplus_fix_function(f,buf);
    };

   if (fg && f != buf) {
      strcpy(buf,f);
      f = buf;
    };

   return f;
};





/************************************************************************/
/*									*/
/*	DDT_map_member_function -- find member function 		*/
/*									*/
/************************************************************************/


String
DDT_map_member_function(f,lhs,args,buf)
   String f;
   String lhs;
   String args;
   String buf;
{
   if (DDT__cplusplus == CPLUSPLUS20) {
      f = DDT_cplus_fix_member_function(f,lhs,args,buf);
    };

   return f;
};





/************************************************************************/
/*									*/
/*	DDT_map_unfix_function -- convert dbx name to source name	*/
/*									*/
/************************************************************************/


String
DDT_map_unfix_function(s)
   String s;
{
   if (s == NULL) return s;

   if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) s = DDT_cplus12_unfix_function(s);
      else s = DDT_cplus_unfix_function(s);
    };

   return s;
};





/************************************************************************/
/*									*/
/*	DDT_map_fix_variable -- map user var to dbx name		*/
/*									*/
/************************************************************************/


String
DDT_map_fix_variable(v,buf)
   String v;
   String buf;
{
   if (DDT__stdpascal) {
      v = DDT_pascal_fix_variable(v,buf);
    }
   else if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) v = DDT_cplus12_fix_variable(v,buf);
      else v = DDT_cplus_fix_variable(v,buf);
    };

   return v;
};





/************************************************************************/
/*									*/
/*	DDT_map_unfix_variable -- convert name from dbx to user 	*/
/*									*/
/************************************************************************/


String
DDT_map_unfix_variable(v)
   String v;
{
   if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) v = DDT_cplus12_unfix_variable(v);
      else v = DDT_cplus_unfix_variable(v);
    };

   return v;
};





/************************************************************************/
/*									*/
/*	DDT_map_fix_field -- map a field name from user to dbx		*/
/*									*/
/************************************************************************/


String
DDT_map_fix_field(fld,lhs,buf)
   String fld;
   String lhs;
   String buf;
{
   if (DDT__stdpascal) {
      fld = DDT_pascal_fix_field(fld,lhs,buf);
    }
   else if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) fld = DDT_cplus12_fix_field(fld,lhs,buf);
      else fld = DDT_cplus_fix_field(fld,lhs,buf);
    };

   return fld;
};





/************************************************************************/
/*									*/
/*	DDT_map_unfix_field -- convert dbx field name to user name	*/
/*									*/
/************************************************************************/


String
DDT_map_unfix_field(fld,lhs)
   String fld;
   String lhs;
{
   if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) fld = DDT_cplus12_unfix_field(fld,lhs);
      else fld = DDT_cplus_unfix_field(fld,lhs);
    };

   return fld;
};






/************************************************************************/
/*									*/
/*	DDT_map_fix_outexpr -- convert output from dbx to user		*/
/*									*/
/************************************************************************/


String
DDT_map_fix_outexpr(expr,calfg,ebuf)
   String expr;
   Boolean calfg;
   String ebuf;
{
   if (DDT__stdpascal) {
      expr = DDT_pascal_fix_outexpr(expr,calfg,ebuf);
    }
   else if (DDT__cplusplus != 0) {
      expr = fix_outexpr(expr,calfg,ebuf);
    };

   return expr;
};





/************************************************************************/
/*									*/
/*	DDT_map_find_fct_local -- find var as local in fct		*/
/*									*/
/************************************************************************/


String
DDT_map_find_fct_local(var,lcls,buf)
   String var;
   Sequence lcls;
   String buf;
{
   if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) var = DDT_cplus12_find_fct_local(var,lcls,buf);
      else var = DDT_cplus_find_fct_local(var,lcls,buf);
    };

   return var;
};







/************************************************************************/
/*									*/
/*	DDT_map_local_level -- return level number of local (or -1)	*/
/*									*/
/************************************************************************/


Boolean
DDT_map_test_save_local(var)
   String var;
{
   Boolean fg;

   if (DDT__cplusplus != 0) {
      if (DDT__cplusplus == CPLUSPLUS12) fg = DDT_cplus12_test_save_local(var);
      else fg = DDT_cplus_test_save_local(var);
    }
   else fg = FALSE;

   return fg;
};






/************************************************************************/
/*									*/
/*	DDT_map_field_var_cache -- allow caching of information 	*/
/*	DDT_map_field_var_info -- get info about variable		*/
/*									*/
/************************************************************************/


#define BUFCT		4
static	String		fldmap_lhs[BUFCT];
static	String		fldmap_rhs[BUFCT];
static	Integer 	fldmap_ct = 0;
static	Integer 	fldmap_lvl = 0;



void
DDT_map_field_var_cache(fg)
   Boolean fg;
{
   Integer i;

   if (fg) {
      if (fldmap_lvl++ == 0) fldmap_ct = 0;
    }
   else if (--fldmap_lvl <= 0) {
      fldmap_lvl = 0;
      for (i = 0; i < BUFCT; ++i) {
	 if (fldmap_lhs[i] != NULL) {
	    SFREE(fldmap_lhs[i]);
	    SFREE(fldmap_rhs[i]);
	    fldmap_lhs[i] = NULL;
	    fldmap_rhs[i] = NULL;
	  };
       };
    };
};





Boolean
DDT_map_field_var_info(lhs,buf,expfg)
   String lhs;
   String buf;
   Boolean expfg;
{
   Integer i;
   Character xbuf[1024];
   String s,t,u;

   buf[0] = 0;
   if (lhs == NULL || lhs[0] == 0) return FALSE;

   for (i = 0; i < BUFCT; ++i) {
      if (fldmap_lhs[i] != NULL && STREQL(lhs,fldmap_lhs[i])) {
	 strcpy(buf,fldmap_rhs[i]);
	 break;
       };
    };

   if (buf[0] == 0) {
      DDT_x_var_info(NULL,NULL,0,lhs,FALSE,FALSE,TRUE,buf);
      if (buf[0] != 0 && fldmap_lvl > 0) {
	 if (fldmap_lhs[fldmap_ct] != NULL) {
	    SFREE(fldmap_lhs[fldmap_ct]);
	    SFREE(fldmap_rhs[fldmap_ct]);
	  };
	 fldmap_lhs[fldmap_ct] = SALLOC(lhs);
	 fldmap_rhs[fldmap_ct] = SALLOC(buf);
	 fldmap_ct = (fldmap_ct+1) % BUFCT;
       };
    };

   s = buf;
   if (s[0] == 0) return FALSE;

   while (strncmp(s,"struct ",7) != 0 && strncmp(s,"union ",6) != 0) {
      s = index(s,' ');
      if (s == NULL) return FALSE;
      ++s;
      if (strncmp(s,"struct ",7) == 0 || strncmp(s,"union ",6) == 0) {
	 strcpy(xbuf,s);
	 s = index(xbuf,' ');
	 ++s;
	 s = index(s,' ');
	 if (s == NULL) return FALSE;
	 *s = 0;
	 if (!DDT_map_field_var_info(xbuf,buf,expfg)) return FALSE;
	 s = buf;
	 if (strncmp(s,"struct ",7) != 0 && strncmp(s,"union ",6) != 0) return FALSE;
       }
      else if (strncmp(buf,"typedef ",8) == 0) {
	 strcpy(xbuf,s);
	 s = index(xbuf,' ');
	 if (s == NULL) return FALSE;
	 *s = 0;
	 if (!DDT_map_field_var_info(xbuf,buf,expfg)) return FALSE;
	 s = buf;
	 if (strncmp(s,"struct ",7) != 0 && strncmp(s,"union ",6) != 0) return FALSE;
       }
      else {
	 strcpy(xbuf,buf);
	 s = index(xbuf,' ');
	 *s = 0;
	 if (!DDT_map_field_var_info(xbuf,buf,expfg)) return FALSE;
	 s = buf;
       };
    };

   if (expfg && index(s,'{') == NULL) {
      strcpy(xbuf,s);
      t = index(xbuf,' ');
      if (t == NULL) return FALSE;
      while (isspace(*t)) ++t;
      for (u = t; !isspace(*u) && *u != ';' && *u != 0; ++u);
      *u = 0;
      DDT_map_field_var_info(xbuf,buf,expfg);
      s = buf;
      if (strncmp(s,"struct ",7) != 0 && strncmp(s,"union ",6) != 0) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	fix_outexpr -- scan expression and convert what one can 	*/
/*									*/
/************************************************************************/


typedef struct _OUTINFO * OUTINFO;

typedef enum _OUTMODE {
   OUTMODE_NONE, OUTMODE_CALL, OUTMODE_FIELD, OUTMODE_ARRAY, OUTMODE_STRUCT,
} OUTMODE;

typedef struct _OUTINFO {
   OUTMODE mode;
   Character lhs[1024];
} OUTINFO_B;




static String
fix_outexpr(expr,calfg,ebuf)
   String expr;
   Boolean calfg;
   String ebuf;
{
   Integer quote,i,j;
   String s,t,p,id,bp;
   String buf,obuf;
   OUTINFO_B info[16];
   Integer infoct;

   DDT_map_field_var_cache(TRUE);

   strcpy(ebuf,expr);
   expr = ebuf;

   i = strlen(expr);
   buf = (String) alloca(2*i+32);
   obuf = (String) alloca(10*i+32);

   s = expr;
   t = obuf;
   infoct = 0;
   quote = 0;
   id = NULL;

   if (calfg) {
      info[0].mode = OUTMODE_CALL;
      if (strncmp(expr,"struct ",7) == 0 && index(expr,'=') == NULL) {
	 strcpy(buf,expr);
	 p = &buf[6];
	 while (isspace(*p)) ++p;
	 while (!isspace(*p)) ++p;
	 *p = 0;
	 strcpy(info[0].lhs,buf);
	 infoct = 1;
	 info[1].mode = OUTMODE_STRUCT;
       };
    }
   else {
      info[0].mode = OUTMODE_NONE;
      id = t;
      bp = buf;
    };
   info[0].lhs[0] = 0;

   while (*s != 0) {
      if (id != NULL) {
	 if (info[infoct].mode == OUTMODE_NONE && *s != '=') {
	    *bp++ = *s;
	  }
	 else if (isalnum(*s) || *s == '_' || *s == '$') {
	    *bp++ = *s;
	  }
	 else if (info[infoct].mode == OUTMODE_CALL) {
	    *bp = 0;
	    j = strlen(buf);
	    p = DDT_map_unfix_variable(buf);
	    t = replace_id(id,j,p,t);
	    id = NULL;
	  }
	 else if (info[infoct].mode == OUTMODE_STRUCT) {
	    *bp = 0;
	    j = strlen(buf);
	    p = DDT_map_unfix_field(buf,info[infoct-1].lhs);
	    t = replace_id(id,j,p,t);
	    id = NULL;
	  }
	 else if (*s != ' ' && *s != '\t' && *s != '=') {
	    id = NULL;
	  };
       };

      if (quote != 0) {
	 if (quote == *s) quote = 0;
       }
      else {
	 if (*s == '{') {
	    ++infoct;
	    info[infoct].mode = OUTMODE_FIELD;
	    info[infoct].lhs[0] = 0;
	  }
	 else if (*s == '(') {
	    if (id == NULL &&
		   (info[infoct].mode == OUTMODE_FIELD ||
		       info[infoct].mode == OUTMODE_ARRAY)) {
	       ++infoct;
	       info[infoct].mode = OUTMODE_ARRAY;
	       sprintf(info[infoct].lhs,"(%s)[0]",info[infoct-1].lhs);
	     };
	  }
	 else if ((*s == '}' || *s == ')') &&
		   (info[infoct].mode == OUTMODE_FIELD ||
		       info[infoct].mode == OUTMODE_ARRAY)) {
	    --infoct;
	    if (info[infoct].mode == OUTMODE_FIELD) info[infoct].lhs[0] = 0;
	  }
	 else if (*s == '"' || *s == '\'') {
	    quote = *s;
	  }
	 else if (id == NULL && (isalnum(*s) || *s == '_' || *s == '$')) {
	    id = t;
	    bp = buf;
	    *bp++ = *s;
	  }
	 else if (id != NULL && *s == '=') {
	    *bp = 0;
	    strcpy(info[infoct].lhs,buf);
	    p = buf;
	    j = strlen(buf);
	    switch (info[infoct].mode) {
	       case OUTMODE_CALL :
		  p = DDT_map_unfix_variable(buf);
		  break;
	       case OUTMODE_ARRAY :
	       case OUTMODE_FIELD :
		  if (info[infoct-1].lhs[0] != 0) {
		     sprintf(info[infoct].lhs,"(%s).%s",info[infoct-1].lhs,buf);
		     p = DDT_map_unfix_field(buf,info[infoct-1].lhs);
		   };
		  break;
	       default :
		  break;
	     };
	    t = replace_id(id,j,p,t);
	    id = NULL;
	  };
       };

      *t++ = *s++;
    };

   *t = 0;
   strcpy(ebuf,obuf);

   DDT_map_field_var_cache(FALSE);

   return ebuf;
};





static String
replace_id(src,ln,new,end)
   String src;
   Integer ln;
   String new;
   String end;
{
   String p,t;
   Character buf[10240];

   t = buf;
   p = new;
   while (*p != 0) *t++ = *p++;
   p = &src[ln];
   while (p != end) *t++ = *p++;
   *t = 0;

   p = buf;
   t = src;
   while (*p != 0) *t++ = *p++;

   return t;
};





/* end of ddtmap.c */
