/************************************************************************/
/*									*/
/*		ddtmangle.c						*/
/*									*/
/*	Name mangling and demangling routines for C++ 2.0		*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "ddt_local.h"
#include "ddt_names.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Type definitions -- mangled name representation 		*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/



static	void		demangle_tail();
static	String		demangle_args();
static	String		demangle_arg();
static	String		add_buf();
static	Integer 	getint();
static	void		setarg();






/************************************************************************/
/*									*/
/*	Tables -- operator encodings					*/
/*									*/
/************************************************************************/


struct name_pair {
   char *s1,*s2;
};

static struct name_pair nplist[] = {
{"lt","<"},    {"ls","<<"},      {"dv","/"},
{"gt",">"},    {"rs",">>"},      {"md","%"},
{"le","<="},   {"ml","*"},       {"pl","+"},
{"ge",">="},   {"ad","&"},       {"mi","-"},
{"ne","!="},   {"or","|"},       {"er","^"},
{"aa","&&"},   {"oo","||"},      {"as","="},
{"apl","+="},  {"ami","-="},     {"amu","*="},
{"adv","/="},  {"amd","%="},     {"aad","&="},
{"aor","|="},  {"aer","^="},     {"als","<<="},
{"ars",">>="}, {"pp","++"},      {"mm","--"},
{"vc","[]"},   {"cl","()"},      {"rf","->"},
{"eq","=="},   {"co","~"},       {"nt","!"},
{"nw","new"}, {"dl","delete"}, {"cm",","},
 {0,0}
};






/************************************************************************/
/*									*/
/*	DDT_mangle_init -- module initialization			*/
/*									*/
/************************************************************************/


void
DDT_mangle_init()
{
};





/************************************************************************/
/*									*/
/*	DDT_mangle_name -- build MANGLE structure for user name 	*/
/*									*/
/************************************************************************/


DDT_MANGLE
DDT_mangle_name(txt,typ)
   String txt;
   DDT_MANGLE_TYPE typ;
{
   String s,t;
   Integer i,j;
   DDT_MANGLE dm;
   Boolean haveargs;

   haveargs = FALSE;

   dm = PALLOC(DDT_MANGLE_INFO);

   s = dm->buf;
   for (i = 0; i < MAX_NAME_LEN; ++i) {
      if (*txt == 0) break;
      *s++ = *txt++;
    };
   *s++ = 0;
   dm->bufp = s;
   txt = dm->buf;

   dm->type = typ;
   dm->file = NULL;

   s = rindex(txt,'`');
   if (s == NULL) dm->prefix = NULL;
   else {
      dm->prefix = txt;
      *s++ = 0;
      txt = s;
    };

   s = index(txt,':');
   if (s != NULL && s[1] == ':') {              /* x::y found           */
      dm->class = txt;
      *s = 0;
      s += 2;
      txt = s;
    }
   else dm->class = NULL;

   for (s = txt; *s != 0 && index(":(,)",*s) == NULL; ++s);
   if (*s == '(') {
      dm->type = MANGLE_FUNCTION;
      haveargs = TRUE;
    };

   if (*s != 0) *s++ = 0;
   else s = NULL;

   dm->base = txt;
   if (strncmp(txt,"operator",8) == 0) {        /* normalize op names   */
      t = &txt[8];
      while (isspace(*t)) ++t;
      sprintf(dm->opbuf,"operator %s",t);
      dm->base = dm->opbuf;
      dm->type = MANGLE_FUNCTION;
    };

   if (s != NULL && (*s == 0 || *s == ')')) s = NULL;

   i = 0;
   for (txt = s; txt != NULL; txt = s) {
      j = 0;
      for (s = txt; *s != 0; ++s) {
	 if (*s == '(') ++j;
	 else if (j > 0 && *s == ')') --j;
	 else if (j == 0 && (*s == ',' || *s == ')')) break;
	 else if (j == 0 && s != txt && strncmp(s,"...",3) == 0) break;
       };
      j = *s;
      *s = 0;
      if (j == 0 || j == ')' || j == '.') s = NULL;
      else ++s;
      dm->args[i++] = txt;
      if (j == '.') dm->args[i++] = "...";
    };

   dm->args[i] = NULL;
   if (i > 0 || haveargs) dm->numargs = i;
   else dm->numargs = -1;

   return dm;
};








/************************************************************************/
/*									*/
/*	DDT_demangle_name -- build MANGLE structure for system name	*/
/*									*/
/************************************************************************/



DDT_MANGLE
DDT_demangle_name(txt)
   String txt;
{
   DDT_MANGLE dm;
   String s,t;
   Integer i,j;
   Character buf[1024];

   dm = PALLOC(DDT_MANGLE_INFO);
   dm->type = MANGLE_UNKNOWN;
   dm->file = NULL;
   dm->prefix = NULL;
   dm->base = NULL;
   dm->class = NULL;
   dm->numargs = -1;
   dm->args[0] = NULL;
   dm->bufp = dm->buf;
   dm->isconst = FALSE;

   i = 0;
   j = -1;
   for (s = txt; *s != 0; ++s) {
      if (*s == '`') j = i;
      ++i;
    };

   if (j >= 0) {
      if (j > 0) dm->prefix = add_buf(dm,txt,j);
      txt = &txt[j+1];
    };

   s = txt;
   if (s[0] == '_' && s[1] == '_') {            /* special cases start w/ __ */
      s += 2;

      if (isdigit(*s)) {			/* check for local	*/
	 while (isdigit(*s)) ++s;
	 dm->base = add_buf(dm,s,strlen(s));
	 dm->type = MANGLE_LOCAL;
	 goto done;
       };

      for (i = 0; s[i] != 0 && s[i] != '_'; ++i);       /* check for operator */
      for (j = 0; nplist[j].s1 != NULL; ++j) {
	 if (strncmp(nplist[j].s1,s,i) == 0) break;
       };
      if (nplist[j].s1 != NULL) {
	 sprintf(dm->opbuf,"operator %s",nplist[j].s2);
	 dm->base = dm->opbuf;
	 dm->type = MANGLE_FUNCTION;
	 s += i;
	 demangle_tail(dm,s);
	 goto done;
       };

      if (strncmp(s,"op",2) == 0) {             /* check for cast       */
	 t = demangle_arg(dm,&s[2],buf);
	 if (t != NULL) {
	    sprintf(dm->opbuf,"operator %s",buf);
	    dm->base = dm->opbuf;
	    dm->type = MANGLE_FUNCTION;
	    demangle_tail(dm,t);
	    goto done;
	  }
	 else {
	    dm->type = MANGLE_BAD;
	    goto done;
	  };
       };

      if (strncmp(s,"ct__",4) == 0) {           /* check for constructor */
	 s += 2;
	 t = &s[2];
	 i = getint(&t);
	 if (i == 0) {
	    dm->type = MANGLE_BAD;
	    goto done;
	  };
	 dm->base = add_buf(dm,t,i);
	 demangle_tail(dm,s);
	 goto done;
       };

      if (strncmp(s,"dt__",4) == 0) {           /* check for destructor  */
	 s += 2;
	 t = &s[2];
	 i = getint(&t);
	 if (i == 0) {
	    dm->type = MANGLE_BAD;
	    goto done;
	  };
	 buf[0] = '~';
	 strncpy(&buf[1],t,i);
	 buf[i+1] = 0;
	 dm->base = add_buf(dm,buf,i+1);
	 demangle_tail(dm,s);
	 goto done;
       };

      if (strncmp(s,"sti__",5) == 0 ||          /* check for static initializer */
	     strncmp(s,"std__",5) == 0) {       /* or for static destructor     */
	 if (s[2] == 'i') strcpy(buf,"#INIT__");
	 else strcpy(buf,"#FREE__");
	 strcat(buf,&s[5]);
	 dm->base = add_buf(dm,buf,0);
	 dm->type = MANGLE_FUNCTION;
	 goto done;
       };
    };

   i = 0;
   dm->type = MANGLE_BAD;
   for (s = txt; *s != 0; ++s) {
      if (s[0] == '_' && s[1] == '_' &&
	     (isdigit(s[2]) || s[2] == 'F' || (s[2] == 'C' && s[3] == 'F'))) {
	 dm->type = MANGLE_UNKNOWN;
	 dm->base = add_buf(dm,txt,i);
	 demangle_tail(dm,s);
	 if (dm->type != MANGLE_BAD) break;
       };
      ++i;
    };

done:
   if (dm->type == MANGLE_BAD || dm->base == NULL) {
      dm->type = MANGLE_UNKNOWN;
      dm->file = NULL;
      dm->base = add_buf(dm,txt,strlen(txt));
      dm->class = NULL;
      dm->numargs = -1;
    };

   if (dm->numargs >= 0) dm->args[dm->numargs] = NULL;
   else dm->args[0] = NULL;

   return dm;
};







/************************************************************************/
/*									*/
/*	DDT_mangle_match -- match two mangle structures 		*/
/*									*/
/************************************************************************/


Boolean
DDT_mangle_match(dma,dmb)
   DDT_MANGLE dma,dmb;
{
   Integer i,j,k,n;

   if (dma->base == NULL || dmb->base == NULL) return FALSE;
   if (STRNEQ(dma->base,dmb->base)) return FALSE;

   if (dma->type != MANGLE_UNKNOWN && dmb->type != MANGLE_UNKNOWN &&
	  dma->type != dmb->type)
      return FALSE;
   if (dma->file != NULL && dmb->file != NULL && STRNEQ(dma->file,dmb->file))
      return FALSE;
   if (dma->class != NULL && dmb->class != NULL && STRNEQ(dma->class,dmb->class))
      return FALSE;

   if (dma->numargs >= 0) {
      if (dma->numargs != dmb->numargs) return FALSE;
      n = dma->numargs;
      for (i = 0; i < n; ++i) {
	 if (dma->args[i] == NULL || dmb->args[i] == NULL) continue;
	 if (dma->args[i][0] == 0 || dmb->args[i][0] == 0) continue;
	 if (STRNEQ(dma->args[i],dmb->args[i])) return FALSE;
       };
    };

   if (dma->prefix != NULL && dmb->prefix != NULL) {
      i = strlen(dma->prefix);
      j = strlen(dmb->prefix);
      n = MIN(i,j);
      for (k = 0; k < n; ++k) {
	 if (dma->prefix[i-1-k] != dmb->prefix[j-1-k]) return FALSE;
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_mangle_output -- convert mangled name to ascii		*/
/*									*/
/************************************************************************/


void
DDT_mangle_output(dm,buf)
   DDT_MANGLE dm;
   String buf;
{
   Integer i;

   buf[0] = 0;

   if (dm->prefix != NULL) strcat(buf,dm->prefix);

   if (dm->class != NULL) {
      strcat(buf,dm->class);
      strcat(buf,"::");
    };

   strcat(buf,dm->base);

   if (dm->numargs >= 0 || dm->type == MANGLE_FUNCTION) {
      strcat(buf,"(");
      for (i = 0; i < dm->numargs; ++i) {
	 if (i != 0 && STRNEQ(dm->args[i],"...")) strcat(buf,",");
	 strcat(buf,dm->args[i]);
       };
      strcat(buf,")");
    };

   if (dm->isconst) strcat(buf," const");
};





/************************************************************************/
/*									*/
/*	DDT_mangle_add_name -- add name to mangle buffer		*/
/*									*/
/************************************************************************/


String
DDT_mangle_add_name(dm,txt)
   DDT_MANGLE dm;
   String txt;
{
   Integer i;
   String s;

   if (txt == NULL) return NULL;

   i = strlen(txt);

   s = add_buf(dm,txt,i);

   return s;
};





/************************************************************************/
/*									*/
/*	demangle_tail -- handle signature of a mangled name		*/
/*									*/
/************************************************************************/


static void
demangle_tail(dm,s)
   DDT_MANGLE dm;
   String s;
{
   Integer i;

   if (s[0] != '_' || s[1] != '_' || (!isdigit(s[2]) && s[2] != 'F')) {
      dm->type = MANGLE_BAD;
      return;
    };

   s = &s[2];
   if (isdigit(*s)) {
      i = getint(&s);
      if (i == 0 || strlen(s) < i) {
	 dm->type = MANGLE_BAD;
	 return;
       };
      dm->class = add_buf(dm,s,i);
      s += i;
    };

   if (*s == 'C') {
      dm->isconst = TRUE;
      ++s;
    };

   if (*s == 'F') {
      dm->type = MANGLE_FUNCTION;
      ++s;
      if (s[0] == 'v' && s[1] == 0) {
	 dm->numargs = 0;
	 dm->args[0] = NULL;
       }
      else if (s[0] == 0) dm->type = MANGLE_BAD;
      else {
	 demangle_args(dm,s);
       };
    };
};







/************************************************************************/
/*									*/
/*	demangle_args -- handle list of arguments			*/
/*									*/
/************************************************************************/


static String
demangle_args(dm,s)
   DDT_MANGLE dm;
   String s;
{
   String t;
   Character buf[1024];

   dm->numargs = 0;

   while (*s != 0 && *s != '_') {
      t = demangle_arg(dm,s,buf);
      if (t == NULL) {
	 dm->type = MANGLE_BAD;
	 return NULL;
       };
      dm->args[dm->numargs++] = add_buf(dm,buf,strlen(buf));
      s = t;
    };

   return s;
};






/************************************************************************/
/*									*/
/*	demangle_arg -- get one argument from an argument list		*/
/*									*/
/************************************************************************/


static String
demangle_arg(dm,s,buf)
   DDT_MANGLE dm;
   String s;
   String buf;
{
   Boolean sign,uns,cons,vol;
   Integer i,j;
   String sfx[128];
   Integer sfxct;
   DDT_MANGLE_INFO newdm;
   Character xbuf[1024];

   sign = FALSE;
   uns = FALSE;
   cons = FALSE;
   vol = FALSE;
   sfxct = 0;

   while (*s) {
      switch (*s) {
	 case 'T' :
	    ++s;
	    i = *s++ - '1';
	    if (i < 0 || i >= dm->numargs) return NULL;
	    strcpy(buf,dm->args[i]);
	    return s;

	 case 'N' :
	    ++s;
	    j = *s++ - '0';
	    i = *s++ - '1';
	    if (i < 0 || i >= dm->numargs) return NULL;
	    while (j > 1) {
	       dm->args[dm->numargs++] = dm->args[i];
	       --j;
	     };
	    strcpy(buf,dm->args[i]);
	    return s;

	 case 'S' :
	    sign = TRUE;
	    break;

	 case 'U' :
	    uns = TRUE;
	    break;

	 case 'C' :
	    cons = TRUE;
	    break;

	 case 'V' :
	    vol = TRUE;
	    break;

	 case 'P' :
	 case 'R' :
	 case 'M' :
	    if (cons) {
	       sfx[sfxct++] = " const";
	       cons = FALSE;
	     };
	    if (vol) {
	       sfx[sfxct++] = " volatile";
	       vol = FALSE;
	     };

	    if (*s == 'P') sfx[sfxct++] = "*";
	    else if (*s == 'R') sfx[sfxct++] = "&";
	    else {
	       sfx[sfxct++] = "::*";
	       ++s;
	       i = getint(&s);
	       sfx[sfxct++] = add_buf(dm,s,i);
	       s += i-1;
	       sfx[sfxct++] = " ";
	     };
	    break;

	 case 'i' :
	    setarg(buf,"int",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'c' :
	    setarg(buf,"char",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 's' :
	    setarg(buf,"short",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'l' :
	    setarg(buf,"long",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'f' :
	    setarg(buf,"float",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'd' :
	    setarg(buf,"double",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'r' :
	    setarg(buf,"long double",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case '1' : case '2' : case '3' :
	 case '4' : case '5' : case '6' :
	 case '7' : case '8' : case '9' :
	    i = getint(&s);
	    if (strlen(s) < i) return NULL;
	    setarg(buf,s,i,sign,uns,cons,vol,sfx,sfxct);
	    s += i;
	    return s;

	 case 'e' :
	    if (cons || vol || uns || sign || sfxct != 0) return NULL;
	    strcpy(buf,"...");
	    return ++s;

	 case 'v' :
	    if (cons || vol || uns || sign || sfxct == 0) return NULL;
	    setarg(buf,"void",0,sign,uns,cons,vol,sfx,sfxct);
	    return ++s;

	 case 'A' :
	    if (sfxct > 0) {
	       for (i = sfxct; i > 0; --i) sfx[i] = sfx[i-1];
	       sfx[0] = ")";
	       ++sfxct;
	       sfx[sfxct++] = "(";
	     };
	    for (i = sfxct-1; i >= 0; --i) sfx[i+3] = sfx[i];
	    sfx[0] = "]";
	    sfx[2] = "[";
	    ++s;
	    i = 0;
	    while (isdigit(s[i])) {
	       ++i;
	     };
	    if (s[i] != '_') return NULL;
	    sfx[1] = add_buf(dm,s,i);
	    s += i;
	    break;

	 case 'F' :
	    for (i = sfxct; i > 0; --i) sfx[i] = sfx[i-1];
	    sfx[0] = ")";
	    ++sfxct;
	    sfx[sfxct++] = "(";
	    newdm.numargs = 0;
	    ++s;
	    newdm.type = MANGLE_UNKNOWN;
	    s = demangle_args(dm,s);
	    if (s == NULL) return NULL;
	    strcpy(xbuf,"(");
	    for (i = 0; i < newdm.numargs; ++i) {
	       if (i != 0) {
		  if (STREQL(newdm.args[i],"...")) strcat(xbuf," ");
		  else strcat(xbuf,",");
		};
	       strcat(xbuf,newdm.args[i]);
	     };
	    strcat(xbuf,")");
	    sfx[sfxct++] = add_buf(dm,xbuf,strlen(xbuf));
	    break;

	 case '_' :
	    buf[0] = 0;
	    return s;

	 default :
	    return NULL;
       };
      ++s;
    };

   if (sfxct != 0 || uns || sign || cons || vol) return NULL;

   return s;
};







/************************************************************************/
/*									*/
/*	add_buf -- add string to buffer for mangle name 		*/
/*									*/
/************************************************************************/


static String
add_buf(dm,s,ln)
   DDT_MANGLE dm;
   String s;
   Integer ln;
{

   String p;

   p = dm->bufp;
   if (ln <= 0) ln = strlen(s);

   while (*s != 0 && ln > 0) {
      --ln;
      *dm->bufp++ = *s++;
    };
   *dm->bufp++ = 0;

   return p;
};






/************************************************************************/
/*									*/
/*	getint -- get integer off of string				*/
/*									*/
/************************************************************************/


static Integer
getint(sp)
   String * sp;
{
   Integer i;
   String s;

   i = 0;
   s = *sp;

   while (isdigit(*s)) i = i*10 + *s++ - '0';

   *sp = s;

   return i;
};






/************************************************************************/
/*									*/
/*	setarg -- constructing argument type string			*/
/*									*/
/************************************************************************/


static void
setarg(buf,base,baseln,sign,uns,cons,vol,sfx,sfxct)
   String buf;
   String base;
   Integer baseln;
   Boolean sign,uns,cons,vol;
   String sfx[];
   Integer sfxct;
{
   String s;
   Integer i;

   buf[0] = 0;
   if (cons) strcat(buf,"const ");
   if (vol) strcat(buf,"volative ");
   if (uns) strcat(buf,"unsigned ");
   if (sign) strcat(buf,"signed ");

   if (baseln <= 0) baseln = strlen(base);

   i = strlen(buf);
   s = &buf[i];
   while (baseln > 0) {
      *s++ = *base++;
      --baseln;
    };
   *s = 0;

   for (i = sfxct-1; i >= 0; --i) {
      strcat(buf,sfx[i]);
    };
};





/* end of ddtmangle.c */
