/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		ddtlangsyn.y						*/
/*									*/
/*	DDT syntax for textual command language 			*/
/*									*/
/************************************************************************/

#include "ddtlang_local.h"
#include <varargs.h>
#include <signal.h>
#include <sgtty.h>

#undef ECHO




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


#define IN_BUF_SIZE	500
#define MIN_READ	128


static	Boolean 	ready_to_read;

static	String		in_buf;
static	Integer 	in_count;
static	Integer 	in_size;
static	Integer 	in_length;
static	Boolean 	in_interrupt;
static	Integer 	in_prefix;

static	DDT_CMD 	error_cmd;

static	String		system_name;

static	Integer 	nested;

static	String		last_search;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/

static	void		show_context();
static	String		dprintf();
static	String		dconcat();
static	String		convert_to_hex();
static	DDT_CMD 	msgoutf();
static	DDT_CMD 	commandoutf();
static	String		fix_up_body();
static	String		fix_search_string();
static	int		ddt_reader();
static	void		stdin_read();
static	void		stdin_user();
static	void		interrupt_handler();




%}



%union {
   String	string;
   DDT_CMD	cmd;
}



%token <string> 	LX_RUN_ARG
%token <string> 	LX_NAME
%token <string> 	LX_ID
%token <string> 	LX_FILENAME
%token <string> 	LX_INT
%token <string> 	LX_STRING
%token <string> 	LX_FLT
%token <string> 	LX_CHAR
%left  <string> 	LX_OP '<' '>' '(' '[' '*' '/' '&' LX_DOTDOT
%token <string> 	LX_NESTLINE LX_ENDLINE LX_ELSELINE LX_SHELLLINE




%token		LX_ARGS LX_ASSIGN LX_BREAK
%token		LX_CALL LX_CATCH LX_CD LX_CONTEXT LX_CONTINUE
%token		LX_DEBUG LX_DEFINE LX_DELETE LX_DISPLAY LX_DOWN LX_DUMP
%token		LX_ELSE LX_END LX_EXEC
%token		LX_FILE LX_FUNC LX_HANDLE LX_HELP LX_IGNORE
%token		LX_INFO LX_KILL LX_LANGUAGE LX_LIST LX_LOOPEXIT LX_LOOPNEXT
%token		LX_MAKE LX_MONITOR LX_NEXT LX_NEXTI LX_NEXTQ
%token		LX_PRINT LX_PRINTENV LX_PWD LX_QUIT LX_QUOTE LX_RERUN
%token		LX_RETURN LX_RUN LX_SETENV LX_SHELL LX_STATUS
%token		LX_STEP LX_STEPI LX_STEPQ
%token		LX_STOP LX_STOPI LX_STOPEXIT LX_SYSTEM
%token		LX_TRACE LX_TRACEI LX_UNMONITOR LX_UNUSE LX_UP LX_UPDATE LX_UPDATEQ
%token		LX_USE LX_VSET LX_WATCH LX_WATCHI LX_WHATIS LX_WHEN
%token		LX_WHERE LX_WHEREIS LX_WHICH LX_WHILE LX_XSET

%token		LX_IN LX_AT LX_IF LX_ALL
%token		LX_EOL

%type	<string>	in_direct in_direct1  out_direct out_direct1
%type	<string>	condition signal procedure filename
%type	<string>	line_number expression variable args_expr op
%type	<string>	expression_list e_expression_list expr_elt
%type	<string>	address format
%type	<string>	opt_else cmd_body handle_key stop_id when_body

%type	<cmd>	command_seq command_seq_list
%type	<cmd>	active_command active_mcommand
%type	<cmd>	run_command rerun_command args_command trace_command
%type	<cmd>	update_command watch_command
%type	<cmd>	stop_command stepstep_command nextnext_command
%type	<cmd>	status_command delete_command handle_command catch_command
%type	<cmd>	ignore_command cont_command step_command next_command
%type	<cmd>	return_command call_command assign_command print_command
%type	<cmd>	up_command down_command file_command func_command
%type	<cmd>	where_command dump_command list_command use_command
%type	<cmd>	kill_command quit_command debug_command cd_command
%type	<cmd>	tracei_command stopi_command watchi_command monitor_command
%type	<cmd>	display_command whatis_command whereis_command which_command
%type	<cmd>	core_command core_command_body quote_command info_command
%type	<cmd>	while_command if_command when_command end_command else_command
%type	<cmd>	loopexit_command
%type	<cmd>	delete_event_list print_expr_list use_file_list
%type	<cmd>	system_command context_command make_command set_command
%type	<cmd>	env_command shell_command help_command search_command
%type	<cmd>	search_command_body


%start	input


%%


input	:	command
	|	input command
	;


command :	command_body LX_EOL
	|	multi_command
	|	error LX_EOL
			{ DDTLANG_eval_statement(error_cmd);
			  while (nest_level > 0) end_nest();
			  yyerrok;
			}
	;


command_body :	active_command
			{ DDTLANG_eval_statement($1);
			  DDTLANG_free_statement($1);
			}
	|	/* empty */
	;


multi_command : active_mcommand
			{ DDTLANG_eval_statement($1);
			  DDTLANG_free_statement($1);
			}
	;



command_seq :	command_seq_list LX_ENDLINE
			{ $$ = $1; }
	;


command_seq_list : /* empty */
			{ $$ = DDTLANG_statement(CMD_TYPE_SAVE); }
	|	command_seq_list active_command LX_EOL
			{ DDTLANG_add_args($1,$2);
			  $$ = $1;
			}
	|	command_seq_list active_mcommand LX_EOL
			{ DDTLANG_add_args($1,$2);
			  $$ = $1;
			}
	;



active_command : run_command
	|	rerun_command
	|	args_command
	|	trace_command
	|	update_command
	|	watch_command
	|	stop_command
	|	stepstep_command
	|	nextnext_command
	|	status_command
	|	delete_command
	|	handle_command
	|	catch_command
	|	ignore_command
	|	cont_command
	|	step_command
	|	next_command
	|	return_command
	|	call_command
	|	assign_command
	|	print_command
	|	up_command
	|	down_command
	|	file_command
	|	func_command
	|	where_command
	|	dump_command
	|	list_command
	|	use_command
	|	kill_command
	|	quit_command
	|	debug_command
	|	cd_command
	|	tracei_command
	|	stopi_command
	|	watchi_command
	|	monitor_command
	|	display_command
	|	whatis_command
	|	whereis_command
	|	which_command
	|	core_command
	|	quote_command
	|	info_command
	|	end_command
	|	else_command
	|	loopexit_command
	|	context_command
	|	system_command
	|	make_command
	|	set_command
	|	env_command
	|	shell_command
	|	help_command
	|	search_command
	;


active_mcommand : while_command
	|	if_command
	|	when_command
	;



run_command :	LX_RUN LX_RUN_ARG
			{ $$ = msgoutf(NULL,"RUN","%S * * 0",$2,0); }
	|	LX_RUN LX_RUN_ARG in_direct1 out_direct
			{ $$ = msgoutf(NULL,"RUN","%S %s %s 0",$2,$3,$4,0); }
	|	LX_RUN LX_RUN_ARG out_direct1 in_direct
			{ $$ = msgoutf(NULL,"RUN","%S %s %s 0",$2,$4,$3,0); }
	|	LX_RUN in_direct1 out_direct
			{ $$ = msgoutf(NULL,"RUN","* %s %s 0",$2,$3,0); }
	|	LX_RUN out_direct1 in_direct
			{ $$ = msgoutf(NULL,"RUN","* %s %s 0",$3,$2,0); }
	|	LX_RUN
			{ $$ = msgoutf(NULL,"RUN","* * * 0",0); }
	;


rerun_command : LX_RERUN LX_RUN_ARG
			{ $$ = msgoutf(NULL,"RUN","%S * * 1",$2,0); }
	|	LX_RERUN LX_RUN_ARG in_direct1 out_direct
			{ $$ = msgoutf(NULL,"RUN","%S %s %s 1",$2,$3,$4,0); }
	|	LX_RERUN LX_RUN_ARG out_direct1 in_direct
			{ $$ = msgoutf(NULL,"RUN","%S %s %s 1",$2,$4,$3,0); }
	|	LX_RERUN in_direct1 out_direct
			{ $$ = msgoutf(NULL,"RUN","* %s %s 1",$2,$3,0); }
	|	LX_RERUN out_direct1 in_direct
			{ $$ = msgoutf(NULL,"RUN","* %s %s 1",$3,$2,0); }
	|	LX_RERUN
			{ $$ = msgoutf(NULL,"RUN","* * * 1",0); }
	;


args_command :	LX_ARGS LX_RUN_ARG
			{ $$ = msgoutf(NULL,"SET","RUN_ARGS %S",$2,0);
			  $$ = msgoutf($$,"SET","INFILE *",0);
			  $$ = msgoutf($$,"SET","OUTFILE *",0);
			}
	|	LX_ARGS LX_RUN_ARG in_direct1 out_direct
			{ $$ = msgoutf(NULL,"SET","RUN_ARGS %S",$2,0);
			  $$ = msgoutf($$,"SET","INFILE %S",$3,0);
			  $$ = msgoutf($$,"SET","OUTFILE %S",$4,0);
			}
	|	LX_ARGS LX_RUN_ARG out_direct1 in_direct
			{ $$ = msgoutf(NULL,"SET","RUN_ARGS %S",$2,0);
			  $$ = msgoutf($$,"SET","INFILE %S",$3,0);
			  $$ = msgoutf($$,"SET","OUTFILE %S",$4,0);
			}
	|	LX_ARGS
			{ $$ = msgoutf(NULL,"SET","RUN_ARGS *",0);
			  $$ = msgoutf($$,"SET","INFILE *",0);
			  $$ = msgoutf($$,"SET","OUTFILE *",0);
			}
	;


in_direct :	/* empty */
			{ $$ = SALLOC("*"); }
	|	in_direct1
	;


out_direct :	/* empty */
			{ $$ = SALLOC("*"); }
	|	out_direct1
	;


in_direct1 :	'<' filename
			{ $$ = $2; }
	;


out_direct1 :	'>' filename
			{ $$ = $2; }
	;


trace_command : LX_TRACE LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 TRACE 3",$3,$4,0); }
	|	LX_TRACE LX_IN filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * %S 0 TRACE 3",$3,$5,$6,0); }
	|	LX_TRACE condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * %S 0 TRACE 3",$2,0); }
	|	LX_TRACE filename '#' line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * %s * %S 0 TRACE 3",$2,$4,$5,0); }
	|	LX_TRACE line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s * %S 0 TRACE 3",$2,$3,0); }
	|	LX_TRACE LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s * %S 0 TRACE 3",$3,$4,0); }
	|	LX_TRACE filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * %S 0 CALL 3",$2,$4,$5,0); }
	|	LX_TRACE procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 CALL 3",$2,$3,0); }
	|	LX_TRACE LX_IN LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 CALL 3",0); }
	|	LX_TRACE LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 TRACE 1",0); }
	|	LX_TRACE LX_IN LX_ALL expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 CALL 19",$4,$5,0); }
	;



update_command : LX_UPDATE LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %S * %S 0 UPDATE 11",$3,$4,0); }
	|	LX_UPDATE LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 UPDATE 11",$3,$4,0); }
	|	LX_UPDATE LX_IN filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * %S 0 UPDATE 11",$3,$5,$6,0); }
	|	LX_UPDATE LX_IN LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 DISPLAY 11",0); }
	|	LX_UPDATE
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 UPDATE 11",0); }
	|	LX_UPDATEQ LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s * %S 0 UPDATE 9",$3,$4,0); }
	|	LX_UPDATEQ LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 UPDATE 9",$3,$4,0); }
	|	LX_UPDATEQ LX_IN filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * %S 0 UPDATE 9",$3,$5,$6,0); }
	|	LX_UPDATEQ LX_IN LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 DISPLAY 9",0); }
	|	LX_UPDATEQ
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 UPDATE 9",0); }
	|	LX_UPDATE LX_IN LX_ALL expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 UPDATE 27",$4,$5,0); }
	|	LX_UPDATEQ LX_IN LX_ALL expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 UPDATE 25",$4,$5,0); }
	;


watch_command : LX_WATCH expression LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s %S %S 0 WATCH 3",$4,$2,$5,0); }
	|	LX_WATCH expression LX_AT filename '#' line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * %s %S %S 0 WATCH 3",$4,$6,$2,$7,0); }
	|	LX_WATCH expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S %S 0 WATCH 3",$2,$3,0); }
	|	LX_WATCH filename '#' variable condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * 0 %S %S 0 WATCH 3",$2,$4,$5,0); }
	|	LX_WATCH expression LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 %S %S 0 WATCH 3",$4,$2,$5,0); }
	|	LX_WATCH expression LX_IN filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 %S %S 0 WATCH 3",$4,$6,$2,$7,0); }
	;


condition :	/* empty */
			{ $$ = SALLOC("*"); }
	|	LX_IF expression
			{ $$ = $2; }
	;


stop_command :	stop_id LX_IF expression
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S * 0 %s 3",$3,$1,0); }
	|	stop_id LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s * %S 0 %s 3",$3,$4,$1,0); }
	|	stop_id LX_AT filename '#' line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * %s * %S 0 %s 3",$3,$5,$6,$1,0); }
	|	stop_id LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 %s 3",$3,$4,$1,0); }
	|	stop_id LX_IN filename '#' procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * %S 0 %s 3",$3,$5,$6,$1,0); }
	|	stop_id variable condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S %S 0 %s 3",$2,$3,$1,0); }
	|	stop_id filename '#' variable condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * 0 %S %S 0 %s 3",$2,$4,$5,$1,0); }
	|	LX_STOP
			{ $$ = msgoutf(NULL,"ACTION","STOP",0); }
	|	stop_id LX_IN LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 %s 19",$1,0); }
	|	stop_id LX_IN LX_ALL expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 %s 19",$4,$5,$1,0); }
	|	LX_STOPEXIT LX_IN procedure condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 STOPEXIT 3",$3,$4,0); }
	|	LX_STOPEXIT LX_IN LX_ALL
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * * 0 STOPEXIT 19",0); }
	|	LX_STOPEXIT LX_IN LX_ALL expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * %S 0 STOPEXIT 19",$4,$5,0); }
	;


stop_id :	LX_STOP
			{ $$ = SALLOC("BREAK"); }
	|	LX_BREAK
			{ $$ = SALLOC("TBREAK"); }
	;


stepstep_command : LX_STEP LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * * 0 STEPSTEP 3",$3,0); }
	|	LX_STEP LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * * 0 STEPSTEP 3",$3,$5,0); }
	|	LX_STEPQ LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * * 0 STEPSTEP 1",$3,0); }
	|	LX_STEPQ LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * * 0 STEPSTEP 1",$3,$5,0); }
	;


nextnext_command : LX_NEXT LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * * 0 NEXTNEXT 3",$3,0); }
	|	LX_NEXT LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * * 0 NEXTNEXT 3",$3,$5,0); }
	|	LX_NEXTQ LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 * * 0 NEXTNEXT 1",$3,0); }
	|	LX_NEXTQ LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 * * 0 NEXTNEXT 1",$3,$5,0); }
	;


status_command : LX_STATUS
			{ $$ = msgoutf(NULL,"EVENT SHOW","* * 0 * * 0 * 0",0); }
	;


delete_command : LX_DELETE delete_event_list
			{ $$ = $2; }
	|	LX_DELETE LX_ALL
			{ $$ = msgoutf(NULL,"EVENT REMOVE","* * 0 * * 0 * 0",0); }
	|	LX_DELETE LX_ALL LX_ID
			{ String s;
			  if (STREQL($3,"stop")) s = "BREAK";
			  else if (STREQL($3,"update")) s = "UPDATE";
			  else if (STREQL($3,"trace")) s = "TRACE";
			  else if (STREQL($3,"watch")) s = "WATCH";
			  else if (STREQL($3,"break")) s = "TBREAK";
			  else if (STREQL($3,"step")) s = "STEPSTEP";
			  else if (STREQL($3,"next")) s = "NEXTNEXT";
			  else if (STREQL($3,"monitor")) s = "MONITOR";
			  else if (STREQL($3,"display")) s = "DISPLAY";
			  else s = "NONE";
			  SFREE($3);
			  s = SALLOC(s);
			  $$ = msgoutf(NULL,"EVENT REMOVE","* * 0 * * 0 %s 0",s,0);
			}
	|	LX_DELETE LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT REMOVE","* %S 0 * * 0 * 0",$3,0); }
	|	LX_DELETE LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT REMOVE","%s %S 0 * * 0 * 0",$3,$5,0); }
	|	LX_DELETE LX_AT line_number
			{ $$ = msgoutf(NULL,"EVENT REMOVE","* * %s * * 0 * 0",$3,0); }
	;


delete_event_list : LX_INT
			{ $$ = msgoutf(NULL,"EVENT REMOVE","* * 0 * * 0 * %s",$1,0); }
	|	delete_event_list LX_INT
			{ $$ = msgoutf($1,"EVENT REMOVE","* * 0 * * 0 * %s",$2,0); }
	;


handle_command : LX_HANDLE
			{ $$ = msgoutf(NULL,"SHOW","SIGNAL",0); }
	|	LX_HANDLE signal handle_key
			{ $$ = msgoutf(NULL,"SET","%s %s",$3,$2,0); }
	;


handle_key :   LX_ID
			{ if (STREQL($1,"ignore")) $$ = "IGNORE";
			  else if (STREQL($1,"catch")) $$ = "CATCH";
			  else if (STREQL($1,"print")) $$ = "PRINTUSE";
			  else if (STREQL($1,"view")) $$ = "PRINTIGN";
			  else $1 = "BAD";
			  $$ = SALLOC($$);
			  SFREE($1);
			}
	;


catch_command : LX_CATCH signal
			{ $$ = msgoutf(NULL,"SET","PRINTUSE %s",$2,0); }
	|	LX_CATCH
			{ $$ = msgoutf(NULL,"SHOW","SIGNAL",0); }
	;


signal	:	LX_INT
	|	LX_ID
	;


ignore_command : LX_IGNORE signal
			{ $$ = msgoutf(NULL,"SET","IGNORE %s",$2,0); }
	|	LX_IGNORE
			{ $$ = msgoutf(NULL,"SHOW","SIGNAL",0); }
	;


cont_command :	LX_CONTINUE
			{ $$ = msgoutf(NULL,"STEP","1 * CONT",0); }
	|	LX_CONTINUE signal
			{ $$ = msgoutf(NULL,"STEP","1 %s CONT",$2,0); }
	;


step_command :	LX_STEP
			{ $$ = msgoutf(NULL,"STEP","1 0 STEP",0); }
	|	LX_STEP LX_INT
			{ $$ = msgoutf(NULL,"STEP","%s 0 STEP",$2,0); }
	|	LX_STEPI
			{ $$ = msgoutf(NULL,"STEP","1 0 STEPI",0); }
	|	LX_STEPI LX_INT
			{ $$ = msgoutf(NULL,"STEP","%s 0 STEPI",$2,0); }
	;


next_command :	LX_NEXT
			{ $$ = msgoutf(NULL,"STEP","1 0 NEXT",0); }
	|	LX_NEXT LX_INT
			{ $$ = msgoutf(NULL,"STEP","%s 0 NEXT",$2,0); }
	|	LX_NEXTI
			{ $$ = msgoutf(NULL,"STEP","1 0 NEXTI",0); }
	|	LX_NEXTI LX_INT
			{ $$ = msgoutf(NULL,"STEP","%s 0 NEXTI",$2,0); }
	;


return_command : LX_RETURN
			{ $$ = msgoutf(NULL,"STEP","1 0 RETURN",0); }
	|	LX_RETURN procedure
			{ $$ = msgoutf(NULL,"STEP","1 0 RETURN",0); }
	;


call_command :	LX_CALL procedure args_expr
			{ $$ = msgoutf(NULL,"CALL","%S %S",$2,$3,0); }
	|	LX_CALL procedure
			{ $$ = msgoutf(NULL,"CALL","%S *",$2,0); }
	;


assign_command : LX_ASSIGN expression '=' expression
			{ $$ = msgoutf(NULL,"ASSIGN","%S %S",$2,$4,0); }
	|	LX_ASSIGN expression ':' '=' expression
			{ $$ = msgoutf(NULL,"ASSIGN","%S %S",$2,$5,0); }
	;


print_command : LX_PRINT print_expr_list
			{ $$ = $2; }
	;


print_expr_list : expression
			{ $$ = msgoutf(NULL,"EVAL","%S",$1,0); }
	|	print_expr_list ',' expression
			{ $$ = msgoutf($1,"EVAL","%S",$3,0); }
	;


up_command :	LX_UP
			{ $$ = msgoutf(NULL,"VIEW","* * 0 0 1",0); }
	|	LX_UP LX_INT
			{ $$ = msgoutf(NULL,"VIEW","* * 0 0 %s",$2,0); }
	;


down_command :	LX_DOWN
			{ $$ = msgoutf(NULL,"VIEW","* * 0 0 -1",0); }
	|	LX_DOWN LX_INT
			{ $$ = msgoutf(NULL,"VIEW","* * 0 0 -%s",$2,0); }
	;


file_command :	LX_FILE filename
			{ $$ = msgoutf(NULL,"VIEW","%s * 0 0 0",$2,0); }
	|	LX_FILE
			{ $$ = msgoutf(NULL,"SHOW","LOCATION",0); }
	;


func_command :	LX_FUNC procedure
			{ $$ = msgoutf(NULL,"VIEW","* %S 0 0 0",$2,0); }
	|	LX_FUNC filename '#' procedure
			{ $$ = msgoutf(NULL,"VIEW","%s %S 0 0 0",$2,$4,0); }
	|	LX_FUNC
			{ $$ = msgoutf(NULL,"SHOW","FOCUS",0); }
	;


where_command : LX_WHERE
			{ $$ = msgoutf(NULL,"STACK","0 0 0",0); }
	|	LX_WHERE LX_INT
			{ $$ = msgoutf(NULL,"STACK","%s 0 0",$2,0); }
	|	LX_WHERE LX_INT LX_INT
			{ $$ = msgoutf(NULL,"STACK","%s %s 0",$2,$3,0); }
	;


dump_command :	LX_DUMP
			{ $$ = msgoutf(NULL,"STACK","0 0 1",0); }
	|	LX_DUMP LX_INT
			{ $$ = msgoutf(NULL,"STACK","%s 0 1",$2,0); }
	|	LX_DUMP LX_INT LX_INT
			{ $$ = msgoutf(NULL,"STACK","%s %s 1",$2,$3,0); }
	;


list_command :	LX_LIST line_number
			{ $$ = msgoutf(NULL,"VIEW","* * %s 1 0",$2,0); }
	|	LX_LIST filename '#' line_number
			{ $$ = msgoutf(NULL,"VIEW","%s * %s 1 0",$2,$4,0); }
	|	LX_LIST line_number ',' line_number
			{ Character buf[16];
			  Integer i;
			  i = atol($4) - atol($2)+1;
			  if (i <= 0) i = 1;
			  sprintf(buf,"%d",i);
			  $$ = msgoutf(NULL,"VIEW","* * %s %s 0",$2,SALLOC(buf),0);
			};
	|	LX_LIST filename '#' line_number ',' line_number
			{ Character buf[16];
			  Integer i;
			  i = atol($6) - atol($4)+1;
			  if (i <= 0) i = 1;
			  sprintf(buf,"%d",i);
			  $$ = msgoutf(NULL,"VIEW","%s * %s %s 0",$2,$4,SALLOC(buf),0);
			};
	|	LX_LIST procedure
			{ $$ = msgoutf(NULL,"VIEW","* %S 0 10 0",$2,0); }
	|	LX_LIST filename '#' procedure
			{ $$ = msgoutf(NULL,"VIEW","%s %S 0 10 0",$2,$4,0); }
	|	LX_LIST
			{ $$ = msgoutf(NULL,"VIEW","* * 0 10 0",0); }
	;


use_command :	LX_USE use_file_list
			{ $$ = $2; }
	|	LX_USE
			{ $$ = msgoutf(NULL,"SHOW","USE",0); }
	|	LX_UNUSE
			{ $$ = msgoutf(NULL,"SET","USE *",0); }
	;


use_file_list :     filename
			{ $$ = msgoutf(NULL,"SET","USE %s",$1,0); }
	|	use_file_list filename
			{ $$ = msgoutf($1,"SET","USE %s",$2,0); }
	;


kill_command :	LX_KILL
			{ $$ = msgoutf(NULL,"ACTION","KILL",0); }
	;


quit_command :	LX_QUIT
			{ if (!DDT__norun) {
			     $$ = msgoutf(NULL,"ACTION","QUIT",0);
			   }
			  else {
			     DDTLANG_set_output(FALSE);
			     exit(0);
			   };
			}
	;


debug_command : LX_DEBUG
			{ $$ = msgoutf(NULL,"ACTION","INIT",0); }
	|	LX_DEBUG '*'
			{ $$ = msgoutf(NULL,"ACTION","RESET",0); }
	|	LX_DEBUG filename
			{ $$ = msgoutf(NULL,"SET","NEWSYS %s",$2,0); }
	|	LX_DEBUG filename filename
			{ $2 = dprintf("%s %s",$2,$3);
			  $$ = msgoutf(NULL,"SET","NEWSYS %S",$2,0);
			}
	|	LX_DEBUG '*' filename
			{ $3 = dprintf("%s %s",DDT__system_name,$3);
			  $$ = msgoutf(NULL,"SET","NEWSYS %S",$3,0);
			}
	;


cd_command :	LX_CD filename
			{ MSGset_wd($2);
			  SFREE($2);
			  $$ = NULL;
			}
	;


tracei_command : LX_TRACEI expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * %S %S TRACE 3",$3,$2,0); }
	|	LX_TRACEI condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * %S 0 TRACE 3",$2,0); }
	;


stopi_command : LX_STOPI expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S %S * BREAK 3",$2,$3,0); }
	|	LX_STOPI condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * %S 0 BREAK 3",$2,0); }
	|	LX_STOPI LX_AT expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 * %S %S BREAK 3",$4,$3,0); }
	;


watchi_command : LX_WATCHI variable condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S %S 0 WATCH 3",$2,$3,0); }
	|	LX_WATCHI variable LX_AT expression condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S %S %S WATCH 3",$2,$5,$4,0); }
	;


monitor_command : LX_MONITOR expression LX_AT line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s %S %S 0 MONITOR 3",$4,$2,$5,0); }
	|	LX_MONITOR expression LX_AT filename '#' line_number condition
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * %s %S %S 0 MONITOR 3",$4,$6,$2,$7,0); }
	|	LX_MONITOR expression
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S * 0 MONITOR 3",$2,0); }
	|	LX_MONITOR expression LX_IN procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 %S * 0 MONITOR 3",$4,$2,0); }
	|	LX_MONITOR expression LX_IN filename '#' procedure
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 %S * 0 MONITOR 3",$4,$6,$2,0); }
	;


display_command : LX_DISPLAY expression
			{ $$ = msgoutf(NULL,"EVENT ADD","* * 0 %S * 0 DISPLAY 3",$2,0); }
	;


whatis_command : LX_WHATIS expression
			{ $$ = msgoutf(NULL,"SYMINFO","WHAT * * 0 %S",$2,0); }
	|	LX_WHATIS LX_IN expression
			{ $$ = msgoutf(NULL,"SYMINFO","VARINFO * * 0 %S",$3,0); }
	|	LX_WHATIS LX_AT LX_STRING
			{ String s;
			  $3[strlen($3)-1] = 0;
			  s = SALLOC(&$3[1]);
			  $$ = msgoutf(NULL,"SYMINFO","TYPEINFO * * 0 %S",s,0);
			  SFREE($3);
			}
	;


whereis_command : LX_WHEREIS variable
			{ $$ = msgoutf(NULL,"SYMINFO","WHERE * * 0 %S",$2,0); }
	;


which_command : LX_WHICH variable
			{ $$ = msgoutf(NULL,"SYMINFO","WHICH * * 0 %S",$2,0); }
	;


core_command :	/* null initial part */
			{ begin_core_cmd(); }
		    core_command_body
			{ $$ = $2; }
	;


core_command_body : address '/' format
			{ $$ = msgoutf(NULL,"DUMP","%s * 1 %s",$1,$3,0); }
	|	address ',' address '/' format
			{ $$ = msgoutf(NULL,"DUMP","%s %s 0 %s",$1,$3,$5,0); }
	|	address '/' LX_INT format
			{ $$ = msgoutf(NULL,"DUMP","%s * %s %s",$1,$3,$4,0); }
	;



quote_command : LX_QUOTE LX_SHELLLINE
			{ $$ = msgoutf(NULL,"SET","QUOTE %S",$2,0); }
	;


info_command :	LX_INFO LX_ID
			{ if (strncmp($2,"sources",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_FILES * * 0 *",0);
			  else if (strncmp($2,"functions",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_FCTS * * 0 *",0);
			  else if (strncmp($2,"variables",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_VARS * * 0 *",0);
			  else if (strncmp($2,"types",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_TYPS * * 0 *",0);
			  else if (strncmp($2,"files",2) == 0)
			     $$ = msgoutf(NULL,"SHOW","SYSTEM",0);
			  else if (strncmp($2,"run",2) == 0)
			     $$ = msgoutf(NULL,"SHOW","RUN",0);
			  else
			     $$ = msgoutf(NULL,"SYMINFO","BAD * * 0 *",0);
			}
	|	LX_INFO LX_ID expression
			{ if (strncmp($2,"sources",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_FILES * * 0 %S",$3,0);
			  else if (strncmp($2,"functions",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_FCTS * * 0 %S",$3,0);
			  else if (strncmp($2,"variables",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_VARS * * 0 %S",$3,0);
			  else if (strncmp($2,"types",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","LIST_TYPS * * 0 %S",$3,0);
			  else if (strncmp($2,"dynamic",2) == 0)
			     $$ = msgoutf(NULL,"SYMINFO","DYNAMIC_TYPE * * 0 %S",$3,0);
			  else
			     $$ = msgoutf(NULL,"SYMINFO","BAD * * 0 %S",$3,0);
			  SFREE($2);
			}
	;


while_command : LX_WHILE LX_NESTLINE LX_EOL cmd_body LX_ENDLINE
			{ $$ = DDTLANG_statement(CMD_TYPE_WHILE);
			  DDTLANG_add_args($$,$2);
			  DDTLANG_add_args($$,fix_up_body($4));
			}
	;


if_command :	LX_IF expression LX_EOL cmd_body opt_else LX_ENDLINE
			{ $$ = DDTLANG_statement(CMD_TYPE_IF);
			  DDTLANG_add_args($$,$2);
			  DDTLANG_add_args($$,fix_up_body($4));
			  DDTLANG_add_args($$,$5);
			}
	;


when_command :	LX_WHEN LX_AT line_number condition when_body
			{ $$ = msgoutf(NULL,"EVENT ADD","* * %s %s %S 0 WHEN 3",$3,$5,$4,0); }
	|	LX_WHEN LX_AT filename '#' line_number condition when_body
			{ $$ = msgoutf(NULL,"EVENT ADD","%s * %s %s %S 0 WHEN 3",$3,$5,$7,$6,0); }
	|	LX_WHEN LX_IN procedure condition when_body
			{ $$ = msgoutf(NULL,"EVENT ADD","* %S 0 %s %S 0 WHEN 3",$3,$5,$4,0); }
	|	LX_WHEN LX_IN filename '#' procedure condition when_body
			{ $$ = msgoutf(NULL,"EVENT ADD","%s %S 0 %s %S 0 WHEN 3",$3,$5,$7,$6,0); }
	;


when_body :	LX_EOL command_seq
			{ $$ = DDTLANG_setup_when($2); }
	;


opt_else :	/* empty */
			{ $$ = NULL; }
	|	LX_ELSELINE cmd_body
			{ $$ = fix_up_body($2); }
	;


cmd_body :	/* empty */
			{ $$ = SALLOC(""); }
	|	cmd_body LX_NESTLINE LX_EOL
			{ $$ = dconcat($1,SALLOC("\n"),$2,0); }
	|	cmd_body LX_EOL
	;


end_command :	LX_END
			{ $$ = DDTLANG_statement(CMD_TYPE_END); }
	;


else_command :	 LX_ELSE
			{ $$ = DDTLANG_statement(CMD_TYPE_ELSE); }
	;


loopexit_command : LX_LOOPEXIT
			{ $$ = DDTLANG_statement(CMD_TYPE_LOOPEXIT); }
	;


context_command : LX_CONTEXT LX_ID
			{ DDTLANG_set_context($2);
			  $$ = NULL;
			}
	|	LX_CONTEXT
			{ show_context();
			  $$ = NULL;
			}
	;


system_command : LX_SYSTEM LX_ID
			{ DDTLANG_set_system($2,0);
			  $$ = NULL;
			}
	|	LX_SYSTEM LX_ID LX_INT
			{ DDTLANG_set_system($2,atol($3));
			  $$ = NULL;
			}
	|	LX_SYSTEM '*'
			{ DDTLANG_set_system("*",0);
			  $$ = NULL;
			}
	;


make_command :	LX_MAKE
			{ $$ = commandoutf(NULL,CMD_TYPE_SEND,"BUILD MAKE %s",
					      DDTLANG_system_name());
			}
	|	LX_MAKE LX_ID
			{ $$ = commandoutf(NULL,CMD_TYPE_SEND,"BUILD COMMAND %s %s",
					      DDTLANG_system_name(),$2);
			}
	;


set_command :	LX_XSET LX_ID LX_INT
			{ $$ = msgoutf(NULL,"SET","%s %s",$2,$3,0); }
	;


env_command :	LX_PRINTENV
			{ $$ = msgoutf(NULL,"SHOW","ENV",0); }
	|	LX_PRINTENV LX_ID
			{ $$ = msgoutf(NULL,"SYMINFO","ENV * * 0 %S",$2,0); }
	|	LX_SETENV LX_SHELLLINE
			{ $$ = msgoutf(NULL,"SET","ENV %S",$2,0); }
	|	LX_PWD
			{ $$ = msgoutf(NULL,"SHOW","WD",0); }
	;


shell_command : LX_SHELL LX_SHELLLINE
			{ $$ = msgoutf(NULL,"SET","SHCMD %S",$2,0); }
	;


help_command :	LX_HELP
			{ DDTLANG_help(NULL);
			  $$ = NULL;
			}
	|	LX_HELP LX_ID
			{ DDTLANG_help($2);
			  SFREE($2);
			  $$ = NULL;
			}
	|	LX_HELP LX_IF
			{ DDTLANG_help("if");
			  $$ = NULL;
			}
	;


search_command :/* null initial part */
			{ begin_search_cmd(); }
		    search_command_body
			{ $$ = $2; }
	;


search_command_body: '/' LX_SHELLLINE
			{ $2 = fix_search_string('/',$2);
			  $$ = msgoutf(NULL,"SET","SEARCH %S",$2,0);
			}
	|	'?' LX_SHELLLINE
			{ $2 = fix_search_string('?',$2);
			  $$ = msgoutf(NULL,"SET","BSEARCH %S",$2,0);
			}
	;


procedure :	LX_ID
	|	procedure '.' LX_ID
			{ $$ = dprintf("%s.%s",$1,$3); }
	|	'~' LX_ID
			{ $$ = dprintf("~%s",$2); }
	;


line_number :	LX_INT
	|	procedure '.' LX_INT
			{ $$ = dprintf("%s.%s",$1,$3); }
	;


expression :	expr_elt
	|	expression expr_elt
			{ $$ = dprintf("%s %s",$1,$2); }
	;

expr_elt :	'(' e_expression_list ')'
			{ $$ = dprintf("(%s)",$2); }
	|	op
	|	LX_INT
	|	LX_FLT
	|	LX_STRING
	|	'[' e_expression_list ']'
			{ $$ = dprintf("[%s]",$2); }
	|	LX_ID
	|	LX_FILENAME
	|	LX_CHAR
	;


args_expr :	'(' e_expression_list ')'
			{ $$ = $2; }
	;


e_expression_list : /* empty */
			{ $$ = SALLOC(""); }
	|	expression_list
	;


expression_list : expression
	|	expression_list ',' expression
			{ $$ = dprintf("%s,%s",$1,$3); }
	;


op	:	LX_OP
	|	'~'
			{ $$ = SALLOC("~"); }
	|	'>'
			{ $$ = SALLOC(">"); }
	|	'<'
			{ $$ = SALLOC("<"); }
	|	'*'
			{ $$ = SALLOC("*"); }
	|	'/'
			{ $$ = SALLOC("/"); }
	|	'&'
			{ $$ = SALLOC("&"); }
	|	'.'
			{ $$ = SALLOC("."); }
	|	LX_DOTDOT
			{ $$ = SALLOC(".."); }
	;


variable :	LX_ID
	|	LX_FILENAME
	;


filename :	LX_STRING
	|	LX_ID
	|	LX_FILENAME
	|	'.'
			{ $$ = SALLOC("."); }
	|	LX_DOTDOT
			{ $$ = SALLOC(".."); }
	;


address :	LX_INT
			{ $$ = convert_to_hex($1); }
	|	'&' LX_ID
			{ $$ = dprintf("&%s",$2); }
	;


format	:	/* empty */
			{ $$ = SALLOC("*"); }
	|	LX_ID
	;


%%


#include "ddtlanglex.c"




/************************************************************************/
/*									*/
/*	DDT_lang_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_lang_init()
{
   Integer msk[2];

   system_name = NULL;

   DDTLANG_stmt_init();

   DDT_input_set_function(ddt_reader);

   ready_to_read = FALSE;

   in_count = 0;
   in_size = 0;
   in_length = IN_BUF_SIZE;
   in_buf = (String) malloc(in_length);
   in_interrupt = FALSE;

   error_cmd = DDTLANG_statement(CMD_TYPE_ERROR);

   nested = 0;

   msk[0] = 1 << 0;
   CMPXregister(1,msk,NULL,NULL,stdin_read);

   signal(SIGINT,interrupt_handler);
};





/************************************************************************/
/*									*/
/*	DDT_lang_loop -- main loop reading input			*/
/*									*/
/************************************************************************/


void
DDT_lang_loop()
{
   Character buf[64];
   String s;

   while (!ready_to_read) DDT_state_next();

   in_prefix = 0;

   sprintf(buf,".ddtinit",getenv("HOME"));
   if (!DDT__norun && access(buf,4) >= 0) {
      DDT_input_source(buf);
      ++in_prefix;
    };

   sprintf(buf,"%s/.ddtinit",getenv("HOME"));
   if (!DDT__norun && access(buf,4) >= 0) {
      DDT_input_source(buf);
      ++in_prefix;
    };

   s = (String) getenv("DDT_INIT");
   if (s != NULL && !DDT__norun && access(s,4) >= 0) {
      DDT_input_source(s);
      ++in_prefix;
    };

   DDTLANG_eval();
};





/************************************************************************/
/*									*/
/*	DDT_lang_startup -- startup completed				*/
/*									*/
/************************************************************************/


void
DDT_lang_startup()
{
   ready_to_read = TRUE;
};





/************************************************************************/
/*									*/
/*	DDTLANG_eval -- evaluate commands				*/
/*									*/
/************************************************************************/


void
DDTLANG_eval()
{
   for ( ; ; ) {
      yyparse();
    };
};





/************************************************************************/
/*									*/
/*	DDTLANG_nest -- add nested text 				*/
/*									*/
/************************************************************************/


void
DDTLANG_nest(text)
   String text;
{
   DDT_input_enqueue(text);
};





/************************************************************************/
/*									*/
/*	DDTLANG_system_name -- return system name string		*/
/*	DDTLANG_set_system -- set system name				*/
/*	DDTLANG_set_context -- set context				*/
/*	show_context -- display context 				*/
/*									*/
/************************************************************************/


String
DDTLANG_system_name()
{
   if (system_name == NULL) return DDT__system_name;

   return system_name;
};




void
DDTLANG_set_system(name,pid)
   String name;
   Integer pid;
{
   Character buf[1024];

   if (system_name != NULL) SFREE(system_name);

   if (DDT__system_name == NULL) DDT__system_name = SALLOC(name);

   sprintf(buf,"%s^%d^%d^*",name,pid,getpid());

   system_name = SALLOC(buf);
};




void
DDTLANG_set_context(ctx)
   String ctx;
{
   Character buf[1024];
   String s;

   if (system_name == NULL) DDTLANG_set_system(DDT__system_name,0);

   s = rindex(system_name,'^');
   *s = 0;
   if (ctx == NULL || *ctx == 0) ctx = "*";

   sprintf(buf,"%s^%s",system_name,ctx);

   SFREE(system_name);
   system_name = SALLOC(buf);
};





static void
show_context()
{
   String s;

   if (system_name == NULL) s = NULL;
   else {
      s = rindex(system_name,'^');
      ++s;
      if (STREQL(s,"*") || *s == 0) s = NULL;
    };

   if (s == NULL) printf("Current context is *ANY*\n");
   else printf("Current context is %s\n",s);
};





/************************************************************************/
/*									*/
/*	DDTLANG_set_output -- set output tty				*/
/*									*/
/************************************************************************/


void
DDTLANG_set_output(fg)
   Boolean fg;
{
   String s;
   extern String ttyname();

   if (fg) {
      s = ttyname(1);
      if (s == NULL) s = ttyname(2);
      if (s == NULL) s = ttyname(0);
      if (s == NULL) return;
    }
   else s = "*";

   MSGsenda("DDT SET %s DDTOUT %s",system_name,s);
};





/************************************************************************/
/*									*/
/*	DDTLANG_set_nested -- set for nested commands			*/
/*									*/
/************************************************************************/


void
DDTLANG_set_nested(fg)
   Boolean fg;
{
   if (fg) ++nested;
   else if (nested <= 0) nested = 0;
   else --nested;
};





/************************************************************************/
/*									*/
/*	yyerror -- handle parsing errors				*/
/*	yywrap -- finish routine					*/
/*									*/
/************************************************************************/


yyerror(msg)
   String msg;
{
   printf("DDT: %s\n",msg);
};




yywrap()
{
	return(1);
}





/************************************************************************/
/*									*/
/*	dprintf -- use printf to put together set of strings		*/
/*									*/
/************************************************************************/


static String
dprintf(va_alist)
   va_dcl
{
   va_list ap;
   Character buf[1024];
   String s,t,p,q;

   va_start(ap);

   s = va_arg(ap,String);
   t = buf;

   while (*s != 0) {
      if (*s != '%') *t++ = *s++;
      else {
	 ++s;
	 switch (*s++) {
	    default :
	       *t++ = *s;
	       break;
	    case 's' :
	       p = va_arg(ap,String);
	       for (q = p; *q != 0; ++q) {
		  *t++ = *q;
		};
	       if (p != NULL && p != DDTLANG_system_name()) SFREE(p);
	       break;
	    case 'S' :
	       *t++ = LIT_STRING;
	       p = va_arg(ap,String);
	       for (q = p; *q != 0; ++q) {
		  *t++ = *q;
		};
	       SFREE(p);
	       *t++ = LIT_STRING;
	       break;
	  };
       };
    };

   *t = 0;

   return SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	dconcat -- concatenate strings					*/
/*									*/
/************************************************************************/


static String
dconcat(va_alist)
   va_dcl
{
   va_list ap;
   register String s;
   Character buf[1024];

   va_start(ap);

   buf[0] = 0;
   for ( ; ; ) {
      s = va_arg(ap,String);
      if (s == NULL) break;
      strcat(buf,s);
      SFREE(s);
    };

   return SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	convert_to_hex -- convert number to hex 			*/
/*									*/
/************************************************************************/


static String
convert_to_hex(s)
   String s;
{
   Integer i;
   Character buf[32];

   i = atol(s);
   sprintf(buf,"0x%x",i);

   SFREE(s);
   s = SALLOC(buf);

   return s;
};





/************************************************************************/
/*									*/
/*	msgoutf -- construct a message for DDT				*/
/*									*/
/************************************************************************/


static DDT_CMD
msgoutf(va_alist)
   va_dcl
{
   va_list ap;
   String pat,typ,rhs;
   String s,sys;
   String args[10];
   register Integer i;
   Character buf[1024];
   DDT_CMD c;

   va_start(ap);

   c = va_arg(ap,DDT_CMD);
   typ = va_arg(ap,String);
   pat = va_arg(ap,String);

   for (i = 0; i < 10; ++i) {
      args[i] = va_arg(ap,String);
      if (args[i] == NULL) break;
    };

   rhs = dprintf(pat,args[0],args[1],args[2],args[3],args[4],args[5],
		    args[6],args[7],args[8],args[9]);

   s = (nested > 0 ? "DDTR" : "DDT");
   sys = DDTLANG_system_name();
   if (sys == NULL) sys = "*";

   sprintf(buf,"%s %s %s %s",s,typ,sys,rhs);
   SFREE(rhs);

   if (c == NULL) c = DDTLANG_statement(CMD_TYPE_EVAL);
   DDTLANG_add_args(c,SALLOC(buf));

   return c;
};





/************************************************************************/
/*									*/
/*	commandoutf -- construct a message for someone else		*/
/*									*/
/************************************************************************/


static DDT_CMD
commandoutf(va_alist)
   va_dcl
{
   va_list ap;
   DDT_CMD_TYPE ctyp;
   String pat,rhs;
   String args[10];
   register Integer i;
   DDT_CMD c;

   va_start(ap);

   c = va_arg(ap,DDT_CMD);
   ctyp = va_arg(ap,DDT_CMD_TYPE);
   pat = va_arg(ap,String);

   for (i = 0; i < 10; ++i) {
      args[i] = va_arg(ap,String);
      if (args[i] == NULL) break;
    };

   rhs = dprintf(pat,args[0],args[1],args[2],args[3],args[4],args[5],
		    args[6],args[7],args[8],args[9]);

   if (c == NULL) c = DDTLANG_statement(ctyp);
   DDTLANG_add_args(c,rhs);

   return c;
};





/************************************************************************/
/*									*/
/*	fix_up_body -- handle body fixup				*/
/*									*/
/************************************************************************/


static String
fix_up_body(body)
   String body;
{
   if (body != NULL) {
      body = dconcat(body,SALLOC("\nend\n"),0);
    };

   return body;
};





/************************************************************************/
/*									*/
/*	fix_search_string -- handle search string fixup 		*/
/*									*/
/************************************************************************/


static String
fix_search_string(trm,str)
   Character trm;
   String str;
{
   String s;
   Integer i;

   i = strlen(str);
   s = &str[i-1];

   while (i > 0 && isspace(*s)) {
      *s-- = 0;
      --i;
    };

   if (*s == trm && (i == 1 || s[-1] != '\\')) {
      *s-- = 0;
      --i;
    };

   if (str[0] == 0) {
      SFREE(str);
      str = SALLOC(last_search);
    }
   else {
      last_search = SALLOC(str);
    };

   return str;
};





/************************************************************************/
/*									*/
/*	ddt_reader -- our substitute for fgetc				*/
/*	stdin_read -- read from std input				*/
/*	stdin_user -- read from std input and send to ddt directly	*/
/*									*/
/************************************************************************/


static int
ddt_reader(fil)
   FILE * fil;
{
   Integer ch;

   if (fil != stdin) {
      if (in_interrupt) {
	 in_interrupt = FALSE;
	 return -1;
       };
      ch = getc(fil);
      if (ch < 0 && in_prefix > 0) {
	 --in_prefix;
	 if (in_prefix == 0) {
	    DDT_input_flush();
	  };
       }
      return ch;
    };

   while (in_count >= in_size && !in_interrupt) {
      in_size = 0;
      in_count = 0;
      DDT_state_next();
    };

   if (in_interrupt) {
      in_interrupt = FALSE;
      return '\n';
    };

   return in_buf[in_count++];
};





/*ARGSUSED*/

static void
stdin_read(rfg,wfg,xfg)
   Integer * rfg, * wfg, *xfg;
{
   Integer i;

   if (DDT_state_inq_run()) {
      if (DDT_model_show_in_file() == NULL) {
	 stdin_user();
	 return;
       };
    };

   if (in_length - in_size < MIN_READ) {
      in_length *= 2;
      in_buf = (String) realloc(in_buf,in_length);
    };

   i = read(fileno(stdin),&in_buf[in_size],in_length-in_size-2);
   if (i > 0) {
      in_size += i;
      in_buf[in_size] = 0;
    };
};





static void
stdin_user()
{
   Character buf[1024];
   Integer i;

   i = read(fileno(stdin),buf,1024-2);

   if (i < 0) return;

   if (i == 0) DDT_user_eof();
   else {
      buf[i] = 0;
      DDT_user_input(buf);
    };
};





/************************************************************************/
/*									*/
/*	interrupt_handler -- handle incoming interrupt			*/
/*									*/
/************************************************************************/


static void
interrupt_handler()
{
   Integer arg;

   DDT_input_flush();

   arg = 0;
   ioctl(fileno(stdin),TIOCFLUSH,&arg);
   in_size = 0;
   in_interrupt = TRUE;
   nested = 0;

   printf("\n");

   MSGsenda("DDTR STOP %s",DDTLANG_system_name());
};





/* end of ddtlangsyn.y */

