/************************************************************************/
/*									*/
/*		ddtlangstmt.c						*/
/*									*/
/*	Statement processing for debugger interface			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "ddtlang_local.h"
#include <setjmp.h>




/************************************************************************/
/*									*/
/*	Local variables 						*/
/*									*/
/************************************************************************/


typedef struct _LOOPDATA * LOOPDATA;
typedef struct _WHENDATA * WHENDATA;


typedef struct _LOOPDATA {
   String eval;
   LOOPDATA next;
} LOOPDATA_INFO;



typedef struct _WHENDATA {
   DDT_CMD cmds;
   String id;
   WHENDATA next;
} WHENDATA_INFO;



static	Sequence	all_vars;
static	LOOPDATA	loops;
static	Integer 	when_ct;
static	WHENDATA	whens;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		evaluate_if();
static	void		evaluate_while();
static	void		while_result();
static	void		when_handler();
static	void		pick_handler();





/************************************************************************/
/*									*/
/*	Module initialization						*/
/*									*/
/************************************************************************/


void
DDTLANG_stmt_init()
{
   all_vars = NULL;
   loops = NULL;
   when_ct = 0;
   whens = NULL;

   MSGregister("WHEN %1s %2s %3d %4s",when_handler,4,NULL);
   MSGregister("DEBUG PICK %1s %2s 1 %3d %4r",pick_handler,4,NULL);
};





/************************************************************************/
/*									*/
/*	DDTLANG_statement -- build new statement			*/
/*									*/
/************************************************************************/


DDT_CMD
DDTLANG_statement(type)
   DDT_CMD_TYPE type;
{
   DDT_CMD c;

   c = PALLOC(DDT_CMD_INFO);

   c->type = type;
   c->args = NULL;

   return c;
};





/************************************************************************/
/*									*/
/*	DDTLANG_add_args -- add to arguments of statement		*/
/*									*/
/************************************************************************/


void
DDTLANG_add_args(c,arg)
   DDT_CMD c;
   String arg;
{
   c->args = APPEND(arg,c->args);
};





/************************************************************************/
/*									*/
/*	DDTLANG_free_statement -- free storage for statement		*/
/*									*/
/************************************************************************/


void
DDTLANG_free_statement(c)
   DDT_CMD c;
{
   Sequence l;
   String s;

   if (c == NULL) return;

   if (c->args != NULL) {
      forin (s,String,l,c->args) SFREE(s);
      LFREE(c->args);
    };

   free(c);
};





/************************************************************************/
/*									*/
/*	DDTLANG_eval_statement -- evaluate a statement			*/
/*									*/
/************************************************************************/


DDT_CMD_RESULT
DDTLANG_eval_statement(c)
   DDT_CMD c;
{
   String s,r;
   Sequence l;
   DDT_CMD_RESULT rslt;

   rslt = CMD_RESULT_OK;

   if (c == NULL) return rslt;

   switch (c->type) {
      case CMD_TYPE_EVAL :
	 forin (s,String,l,c->args) {
	    r = MSGcall(s);
	    if (r != NULL) SFREE(r);
	  };
	 break;

      case CMD_TYPE_SEND :
	 forin (s,String,l,c->args) {
	    MSGsend(s);
	  };
	 break;

      case CMD_TYPE_IF :
	 evaluate_if(CAR(String,c->args),
			CADR(String,c->args),
			CADDR(String,c->args));
	 break;

      case CMD_TYPE_WHILE :
	 evaluate_while(CAR(String,c->args),CADR(String,c->args));
	 break;

      case CMD_TYPE_END :
	 while_result(CMD_RESULT_OK);
	 break;

      case CMD_TYPE_ERROR :
	 while_result(CMD_RESULT_ERROR);
	 break;

      case CMD_TYPE_LOOPEXIT :
	 while_result(CMD_RESULT_LOOPEXIT);
	 break;

      default :
	 break;
    };

   return rslt;
};





/************************************************************************/
/*									*/
/*	DDTLANG_setup_when -- setup when statement			*/
/*									*/
/************************************************************************/


String
DDTLANG_setup_when(cmds)
   DDT_CMD cmds;
{
   Character buf[256];
   WHENDATA wd;

   ++when_ct;

   sprintf(buf,"DDT_%s_%d",DDTLANG_system_name(),when_ct);

   wd = PALLOC(WHENDATA_INFO);
   wd->cmds = cmds;
   wd->id = SALLOC(buf);
   wd->next = whens;
   whens = wd;

   return SALLOC(buf);
};




/************************************************************************/
/*									*/
/*	evaluate_if -- handle if statement				*/
/*									*/
/************************************************************************/


static void
evaluate_if(expr,tr,fa)
   String expr;
   String tr;
   String fa;
{
   String s;

   s = MSGcalla("DDTR EVAL %s %s",DDTLANG_system_name(),expr);

   if (s == NULL) return;

   if (STREQL(s,"0")) tr = fa;

   SFREE(s);

   DDTLANG_nest(tr);
};





/************************************************************************/
/*									*/
/*	evaluate_while -- handle while statement			*/
/*									*/
/************************************************************************/


static void
evaluate_while(expr,body)
   String expr;
   String body;
{
   Character buf[10240];
   String s;
   Integer i;
   LOOPDATA ld;

   i = strlen(expr);
   if (expr[i-1] == '\n') expr[i-1] = 0;
   while (isspace(*expr)) ++expr;

   ++body;
   s = rindex(body,'\n');
   *s = 0;
   s = rindex(body,'\n');
   *s = 0;

   sprintf(buf,"if %s\n%s\nelse\nloopexit\nend\n",expr,body);

   ld = PALLOC(LOOPDATA_INFO);
   ld->eval = SALLOC(buf);
   ld->next = loops;
   loops = ld;

   while_result(CMD_RESULT_OK);
};




static void
while_result(sts)
   DDT_CMD_RESULT sts;
{
   LOOPDATA ld;

   if (loops == NULL) return;

   switch (sts) {
      case CMD_RESULT_OK :
	 DDTLANG_nest(loops->eval);
	 break;

      default :
	 ld = loops;
	 loops = ld->next;
	 SFREE(ld->eval);
	 free(ld);
	 break;
    };

   if (sts == CMD_RESULT_ERROR && loops != NULL) while_result(sts);
};




/************************************************************************/
/*									*/
/*	when_handler -- handle when message requests			*/
/*									*/
/************************************************************************/


static void
when_handler(sys,file,line,id,rid)
   String sys;
   String file;
   Integer line;
   String id;
   Integer rid;
{
   WHENDATA wd;
   Character buf[1024];


   for (wd = whens; wd != NULL; wd = wd->next) {
      if (STREQL(id,wd->id)) break;
    };

   if (wd == NULL) {
      if (rid >= 0) MSGreply(rid,NULL);
      return;
    };

   strcpy(buf,sys);
   if (rid >= 0) {
      MSGsenda("DDTR SET %s PROG 1",buf);
      MSGreply(rid,NULL);
    };

   DDTLANG_set_nested(TRUE);
   DDTLANG_nest(wd->cmds);
   DDTLANG_set_nested(FALSE);

   if (rid >= 0) {
      MSGsenda("DDTR SET %s ENDPROG 1",buf);
    };
};





/************************************************************************/
/*									*/
/*	pick_handler -- handle queries for name disambiguation		*/
/*									*/
/************************************************************************/


static void
pick_handler(sys,orig,ct,alts,rid)
   String sys;
   String orig;
   Integer ct;
   String alts;
   Integer rid;
{
   String s,p;
   Character buf[1024];
   Integer i,j;

   if (STRNEQ(sys,DDTLANG_system_name())) {
      if (rid >= 0) MSGreply(rid,NULL);
      return;
    };

   printf("\nChoose proper alternative for %s:\n",orig);

   s = alts;
   for (i = 0; i < ct; ++i) {
      p = buf;

      while (*s != 0 && *s != ';') *p++ = *s++;
      if (*s == ';') ++s;
      *p = 0;

      printf("\t(%d) %s\n",i+1,buf);
    };

   printf("Enter number of proper selection: ");
   j = fscanf(stdin,"%d",&i);

   if (j == 1 && i > 0 && i <= ct) {
      sprintf(buf,"%d",i-1);
      if (rid >= 0) MSGreply(rid,buf);
    }
   else if (rid >= 0) MSGreply(rid,NULL);
};





/* end of ddtlangstmt.c */
