/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		ddtlanglex.l						*/
/*									*/
/*	Lexical definitions for ddt parser				*/
/*									*/
/************************************************************************/


#include <sys/time.h>

#define INPUT_LEN	10240
#define MAX_NEST	8

#undef input
#undef output
#undef unput
#undef YYLMAX

#define input() 	read_input()
#define output(x)	(x)
#define unput(x)	read_unput(x)
#define YYLMAX		4096

extern	double		atof();
static	String		get_hex();
static	Integer 	test_id();
static	void		start_nest();
static	void		end_nest();
static	void		begin_core_cmd();
static	void		begin_search_cmd();
static	void		begin_reset();

static	Character	line[INPUT_LEN];
static	Integer 	line_idx = -1;
static	Integer 	line_len = -1;
static	Integer 	nest_level = 0;
static	Boolean 	nest_lit[MAX_NEST];



%}

%a 60480
%p 10240
%o 40960
%n 2048


%START RUN_MODE CMD_MODE FIRST NEST CORE SHELL_MODE INFO_MODE STMT


WHITE	([ \t\r])+
WHITE0	([ \t\r])*

INT	([0-9]+)
HEXINT	(0x[0-9a-f]+)

STRING	(\"([^\n"\\]|(\\.))*\")

ID	([a-zA-Z_@$`][a-zA-Z_$0-9@?`]*)
CPID	({ID}(("::"|":"){ID})+)
CPID1	({ID}(("::~"|":~"){ID})+)
FID	(([0-9a-zA-Z_@$?][a-zA-Z_$0-9@?`]*)|"..")
STRUCT	("struct"{WHITE}{ID})
UNION	("union"{WHITE}{ID})
CLASS	("class"{WHITE}{ID})
FILEPFX (("/")|("../")|("./"))
FILE	(({FILEPFX}*){FID}([./]{FID})*)

CHAR0	(\'.\')
CHAR1	(\'\\.\')
CHAR2	(\'\\([0-7]{1,3})\')
CHAR	({CHAR0}|{CHAR1}|{CHAR2})

QID	(\'([^\n'\\]|(\\.))*\')

FLT0	([0-9]+\.[0-9]*)
EXP	([Ee][-+]?[0-9]+)
FLT1	({INT}{EXP})
FLT2	({FLT0}{EXP})
FLT	({FLT0}|{FLT1}|{FLT2})

OPCHAR	([-+%|^])
OPSTR	("!="|"=="|"<="|">="|"&&"|"||"|"++"|"--"|"->"|"^.")


SINGLE	([=:{}()#<>.,;~/?])

CMMT1	("--".*)
CMMT	({CMMT1})

RUN_ARGS ([^\n<>]+)
end_LINE ({WHITE0}end{WHITE0}\n)
else_LINE ({WHITE0}else{WHITE0}\n)



%%


<RUN_MODE>{RUN_ARGS}		{ BEGIN CMD_MODE;
				  yylval.string = SALLOC(yytext);
				  return LX_RUN_ARG;
				}
<RUN_MODE>[><]			{ BEGIN CMD_MODE;
				  return yytext[0];
				}
<SHELL_MODE>.*			{ yylval.string = SALLOC(yytext);
				  return LX_SHELLLINE;
				}
<STMT>[\n]			{ start_nest(FALSE); begin_reset(); return LX_EOL; }
<FIRST>[\n]			{ BEGIN NEST; start_nest(TRUE); return LX_EOL; }
<NEST>{end_LINE}		{ begin_reset(); end_nest(); return LX_ENDLINE; }
<NEST>{else_LINE}		{ return LX_ELSELINE; }
<NEST>.*			{ yylval.string = SALLOC(yytext);
				  return LX_NESTLINE;
				}
<NEST>[\n]			{ return LX_EOL; }
<INFO_MODE>[^ \t\n]+		{ yylval.string = SALLOC(yytext);
				  return LX_ID;
				};

"in"                            { return LX_IN; }
"at"                            { return LX_AT; }
"all"                           { return LX_ALL; }

<CMD_MODE,FIRST,STMT>"if"       { return LX_IF; }
<CMD_MODE,FIRST,CORE,STMT>{ID}	{ yylval.string = SALLOC(yytext);
				  return LX_ID;
				}


{OPSTR} 			{ yylval.string = SALLOC(yytext);
				  return LX_OP;
				}
".."                            { return LX_DOTDOT; }
{OPCHAR}			{ yylval.string = SALLOC(yytext);
				  return LX_OP;
				}
{SINGLE}			{ return yytext[0]; }

{INT}				{ yylval.string = SALLOC(yytext);
				  return LX_INT;
				}
{HEXINT}			{ yylval.string = get_hex(yytext);
				  return LX_INT;
				}
{CHAR}				{ yylval.string = SALLOC(yytext);
				  return LX_INT;
				}
{STRING}			{ yylval.string = SALLOC(yytext);
				  return LX_STRING;
				}
{FLT}				{ yylval.string = SALLOC(yytext);
				  return LX_FLT;
				}

{STRUCT}			{ yylval.string = SALLOC(yytext);
				  return LX_ID;
				}
{UNION} 			{ yylval.string = SALLOC(yytext);
				  return LX_ID;
				}
{CLASS} 			{ yylval.string = SALLOC(yytext);
				  return LX_ID;
				}
{ID}				{ return test_id(yytext); }
{CPID}				{ return test_id(yytext); }
{CPID1} 			{ return test_id(yytext); }
{QID}				{ return test_id(yytext); }

<CMD_MODE>{FILE}		{ yylval.string = SALLOC(yytext);
				  return LX_FILENAME;
				}

{CMMT}				;
{WHITE} 			;

[\n]				{ begin_reset(); return LX_EOL; }

.				{ return yytext[0]; }




%%


/************************************************************************/
/*									*/
/*	Tables -- keywords						*/
/*									*/
/************************************************************************/


typedef struct _KEY {
   String text;
   Integer mode;
   Integer rslt;
} KEY_ID;

static	KEY_ID	keywords[] = {
   { "args",            RUN_MODE,       LX_ARGS },
   { "assign",          CMD_MODE,       LX_ASSIGN },
   { "break",           CMD_MODE,       LX_BREAK },
   { "call",            CMD_MODE,       LX_CALL },
   { "catch",           CMD_MODE,       LX_CATCH },
   { "cd",              CMD_MODE,       LX_CD },
   { "cont",            CMD_MODE,       LX_CONTINUE },
   { "context",         CMD_MODE,       LX_CONTEXT },
   { "debug",           CMD_MODE,       LX_DEBUG },
   { "define",          NEST,           LX_DEFINE },
   { "delete",          CMD_MODE,       LX_DELETE },
   { "display",         CMD_MODE,       LX_DISPLAY },
   { "down",            CMD_MODE,       LX_DOWN },
   { "dump",            CMD_MODE,       LX_DUMP },
   { "else",            CMD_MODE,       LX_ELSE },
   { "end",             CMD_MODE,       LX_END },
   { "exec",            CMD_MODE,       LX_EXEC },
   { "file",            CMD_MODE,       LX_FILE },
   { "func",            CMD_MODE,       LX_FUNC },
   { "handle",          CMD_MODE,       LX_HANDLE },
   { "help",            CMD_MODE,       LX_HELP },
   { "if",              FIRST,          LX_IF },
   { "ignore",          CMD_MODE,       LX_IGNORE },
   { "info",            INFO_MODE,      LX_INFO },
   { "kill",            CMD_MODE,       LX_KILL },
   { "language",        CMD_MODE,       LX_LANGUAGE },
   { "list",            CMD_MODE,       LX_LIST },
   { "loopexit",        CMD_MODE,       LX_LOOPEXIT },
   { "loopnext",        CMD_MODE,       LX_LOOPNEXT },
   { "make",            CMD_MODE,       LX_MAKE },
   { "monitor",         CMD_MODE,       LX_MONITOR },
   { "next",            CMD_MODE,       LX_NEXT },
   { "nexti",           CMD_MODE,       LX_NEXTI },
   { "nextq",           CMD_MODE,       LX_NEXTQ },
   { "print",           CMD_MODE,       LX_PRINT },
   { "printenv",        CMD_MODE,       LX_PRINTENV },
   { "pwd",             CMD_MODE,       LX_PWD },
   { "quit",            CMD_MODE,       LX_QUIT },
   { "quote",           SHELL_MODE,     LX_QUOTE },
   { "rerun",           RUN_MODE,       LX_RERUN },
   { "return",          CMD_MODE,       LX_RETURN },
   { "run",             RUN_MODE,       LX_RUN },
   { "setenv",          SHELL_MODE,     LX_SETENV },
   { "sh",              SHELL_MODE,     LX_SHELL },
   { "status",          CMD_MODE,       LX_STATUS },
   { "step",            CMD_MODE,       LX_STEP },
   { "stepi",           CMD_MODE,       LX_STEPI },
   { "stepq",           CMD_MODE,       LX_STEPQ },
   { "stop",            CMD_MODE,       LX_STOP },
   { "stopexit",        CMD_MODE,       LX_STOPEXIT },
   { "stopi",           CMD_MODE,       LX_STOPI },
   { "system",          CMD_MODE,       LX_SYSTEM },
   { "trace",           CMD_MODE,       LX_TRACE },
   { "tracei",          CMD_MODE,       LX_TRACEI },
   { "unmonitor",       CMD_MODE,       LX_UNMONITOR },
   { "unuse",           CMD_MODE,       LX_UNUSE },
   { "up",              CMD_MODE,       LX_UP },
   { "update",          CMD_MODE,       LX_UPDATE },
   { "updateq",         CMD_MODE,       LX_UPDATEQ },
   { "use",             CMD_MODE,       LX_USE },
   { "vset",            CMD_MODE,       LX_VSET },
   { "watch",           CMD_MODE,       LX_WATCH },
   { "watchi",          CMD_MODE,       LX_WATCHI },
   { "whatis",          CMD_MODE,       LX_WHATIS },
   { "when",            STMT,           LX_WHEN },
   { "where",           CMD_MODE,       LX_WHERE },
   { "whereis",         CMD_MODE,       LX_WHEREIS },
   { "which",           CMD_MODE,       LX_WHICH },
   { "while",           NEST,           LX_WHILE },
   { "xset",            CMD_MODE,       LX_XSET },
   { 0, 0, LX_ID }
};



/************************************************************************/
/*									*/
/*	Lexical routines						*/
/*									*/
/************************************************************************/


static String
get_hex(s)
   String s;
{
   register Integer i;
   Character buf[32];

   i = 0;
   ++s;
   ++s;

   while (*s != 0) {
      if (isdigit(*s)) i = i*16 + (*s) - '0';
      else if (isupper(*s)) i = i*16 + (*s) - 'A' + 10;
      else i = i*16 + (*s) - 'a' + 10;
      ++s;
     };

   sprintf(buf,"%u",i);

   return SALLOC(buf);
};




static Integer
test_id(txt)
   String txt;
{
   Integer i,rslt;

   for (i = 0; keywords[i].text != NULL; ++i) {
      if (STREQL(keywords[i].text,txt)) {
	 BEGIN keywords[i].mode;
	 rslt = keywords[i].rslt;
	 if (keywords[i].mode == NEST) start_nest(TRUE);
	 else if (keywords[i].mode == SHELL_MODE) DDT_input_set_delimiter(0);
	 else if (keywords[i].rslt == LX_END && nest_level > 0) {
	    end_nest();
	    rslt = LX_ENDLINE;
	  };
	 return rslt;
       }
    };

   yylval.string = SALLOC(txt);

   return LX_ID;
};





/************************************************************************/
/*									*/
/*	read_input -- read input for lex				*/
/*	read_unput -- put back input for lex				*/
/*									*/
/************************************************************************/


static Integer
read_input()
{
   if (line_len < 0) {			/* first time			*/
      if (!DDT__standalone) DDT_input_set_prompt("<ddt !>\1 ");
      else DDT_input_set_prompt("<ddt !> ");
      line_len = line_idx = 0;
    };

   while (line_idx >= line_len) {
      if (DDT_input_line(INPUT_LEN,line) == NULL) continue;
      line_len = strlen(line);
      line[line_len++] = '\n';
      line[line_len] = 0;
      line_idx = 0;
    };

   return line[line_idx++];
};





static Integer
read_unput(ch)
{
   if (line_idx == 0) return ch;
   line[--line_idx] = ch;

   return ch;
};





/************************************************************************/
/*									*/
/*	start_nest -- begin parsing nested lines			*/
/*	end_nest -- end parsing nested lines				*/
/*									*/
/************************************************************************/


static void
start_nest(lit)
   Boolean lit;
{
   ++nest_level;

   nest_lit[nest_level] = lit;
   if (nest_level == 1) DDT_input_set_prompt(">> ");
   else DDT_input_set_prompt(">>> ");

   if (lit) DDT_input_set_literal(TRUE);
};




static void
end_nest()
{
   if (nest_level == 0) return;

   --nest_level;
   if (nest_level == 0 || !nest_lit[nest_level]) DDT_input_set_literal(FALSE);
   else DDT_input_set_literal(TRUE);

   if (nest_level == 0) DDT_input_set_prompt("<ddt !> ");
   else if (nest_level == 1) DDT_input_set_prompt(">> ");
   else DDT_input_set_prompt(">>> ");
};





/************************************************************************/
/*									*/
/*	begin_core_cmd -- start dump command				*/
/*	begin_serach_cmd -- start search command			*/
/*	begin_reset -- reset to start of command			*/
/*									*/
/************************************************************************/


static void
begin_core_cmd()
{
   BEGIN CORE;
};





static void
begin_search_cmd()
{
   BEGIN SHELL_MODE;
   DDT_input_set_delimiter(0);
};





static void
begin_reset()
{
   BEGIN 0;
   DDT_input_set_delimiter(';');
};





/* end of ddtlex.l */





