/************************************************************************/
/*									*/
/*		ddtinput.c						*/
/*									*/
/*	Input and history routines for DDT				*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAXARGS 	1024
#define MAX_LINE_SIZE	10240




/************************************************************************/
/*									*/
/*	Type Defionitions						*/
/*									*/
/************************************************************************/



typedef struct _ALIAS * 	ALIAS;
typedef struct _HISTORY *	HISTORY;
typedef struct _FSTACK *	FSTACK;
typedef struct _VAR *		VAR;
typedef struct _LINE *		LINE;



typedef struct _ALIAS {
   String name;
   Integer length;
   String value;
   ALIAS next;
} ALIAS_INFO;




typedef struct _HISTORY {
   String value;
   Integer number;
   HISTORY last;
   HISTORY next;
} HISTORY_INFO;




typedef struct _FSTACK {
   FILE * file;
   FSTACK next;
   LINE queue;
} FSTACK_INFO;




typedef struct _VAR {
   String name;
   String value;
   VAR next;
} VAR_INFO;





typedef struct _LINE {
   String text;
   LINE next;
} LINE_INFO;




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	FSTACK		all_files;
static	HISTORY 	last_hist;
static	HISTORY 	first_hist;
static	ALIAS		all_alias;
static	VAR		all_vars;
static	LINE		all_lines;

static	Function_Ptr	input_fct;
static	Integer 	end_of_line;
static	Boolean 	quote_it;
static	String		prompt;
static	Integer 	history_cnt;
static	Integer 	cur_history;

static	FILE *		curfile;

extern	int		fgetc();




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	get_next_line();
static	void		output_prompt();
static	Boolean 	do_history();
static	String		get_history();
static	String		match_history();
static	String		subst_history();
static	String		history_words();
static	void		save_history();
static	void		pop_history();
static	void		do_alias();
static	void		do_variable();
static	Boolean 	check_builtin();
static	void		handle_alias();
static	void		handle_unalias();
static	void		handle_set();
static	void		handle_unset();
static	void		handle_history();
static	void		handle_source();
static	void		split_command();





/************************************************************************/
/*									*/
/*	DDT_input_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
DDT_input_init()
{
   input_fct = fgetc;
   end_of_line = ';';
   all_files = NULL;
   last_hist = NULL;
   first_hist = NULL;
   all_vars = NULL;
   all_lines = NULL;

   quote_it = FALSE;
   curfile = stdin;

   prompt = SALLOC("<ddt !> ");
   history_cnt = 50;
   cur_history = 0;
};





/************************************************************************/
/*									*/
/*	DDT_input_set_function -- set character reading function	*/
/*									*/
/************************************************************************/


void
DDT_input_set_function(fct)
   Function_Ptr fct;
{
   input_fct = fct;
};





/************************************************************************/
/*									*/
/*	DDT_input_set_delimiter -- set end of line delimiter		*/
/*									*/
/************************************************************************/


void
DDT_input_set_delimiter(chr)
   Integer chr;
{
   end_of_line = chr;
};





/************************************************************************/
/*									*/
/*	DDT_input_set_literal -- turn on/off literal mode		*/
/*									*/
/************************************************************************/


void
DDT_input_set_literal(fg)
   Boolean fg;
{
   quote_it = fg;
};





/************************************************************************/
/*									*/
/*	DDT_input_set_prompt -- set prompt string			*/
/*									*/
/************************************************************************/


void
DDT_input_set_prompt(str)
   String str;
{
   if (prompt != NULL) SFREE(prompt);

   if (str == NULL) prompt = NULL;
   else prompt = SALLOC(str);
};





/************************************************************************/
/*									*/
/*	DDT_input_set_alias -- define a new alias			*/
/*									*/
/************************************************************************/


void
DDT_input_set_alias(als,value)
   String als;
   String value;
{
   ALIAS a,la;

   la = NULL;
   for (a = all_alias; a != NULL; a = a->next) {
      if (STREQL(a->name,als)) break;
      la = a;
    };

   if (a == NULL && value == NULL) return;

   if (a == NULL) {
      a = PALLOC(ALIAS_INFO);
      a->name = SALLOC(als);
      a->length = strlen(a->name);
      a->value = SALLOC(value);
      a->next = all_alias;
      all_alias = a;
    };

   if (value != NULL && STRNEQ(a->value,value)) {
      SFREE(a->value);
      a->value = SALLOC(value);
    }
   else if (value == NULL) {
      if (la == NULL) all_alias = a->next;
      else la->next = a->next;
      SFREE(a->name);
      SFREE(a->value);
      free(a);
    };
};





/************************************************************************/
/*									*/
/*	DDT_input_set_var -- define a new variable			*/
/*									*/
/************************************************************************/


void
DDT_input_set_var(var,value)
   String var;
   String value;
{
   VAR v,lv;

   lv = NULL;
   for (v = all_vars; v != NULL; v = v->next) {
      if (STREQL(v->name,var)) break;
      lv = v;
    };

   if (v == NULL && value == NULL) return;

   if (v == NULL) {
      v = PALLOC(VAR_INFO);
      v->name = SALLOC(var);
      v->value = SALLOC(value);
      v->next = all_vars;
      all_vars = v;
    };

   if (value != NULL && STRNEQ(v->value,value)) {
      SFREE(v->value);
      v->value = SALLOC(value);
    }
   else if (value == NULL) {
      if (lv == NULL) all_vars = v->next;
      else lv->next = v->next;
      SFREE(v->name);
      SFREE(v->value);
      free(v);
    };

   if (STREQL(v->name,"prompt")) {
      DDT_input_set_prompt(value);
    }
   else if (STREQL(v->name,"history") && value != NULL) {
      history_cnt = atol(value);
    };
};





/************************************************************************/
/*									*/
/*	DDT_input_source -- source in a new file			*/
/*									*/
/************************************************************************/


void
DDT_input_source(name)
   String name;
{
   FILE * inf;
   FSTACK fs;

   inf = fopen(name,"r");
   if (inf == NULL) return;

   fs = PALLOC(FSTACK_INFO);
   fs->file = curfile;
   fs->next = all_files;
   all_files = fs;
   fs->queue = all_lines;
   all_lines = NULL;

   curfile = inf;
};





/************************************************************************/
/*									*/
/*	DDT_input_enqueue -- add lines to start of input		*/
/*									*/
/************************************************************************/


void
DDT_input_enqueue(text)
   String text;
{
   LINE ln;
   String buf;

   if (text == NULL || text[0] == 0) return;

   buf = SALLOC(text);

   sprintf(buf,"%s\n",text);

   ln = PALLOC(LINE_INFO);
   ln->text = buf;
   ln->next = all_lines;
   all_lines = ln;
};





/************************************************************************/
/*									*/
/*	DDT_input_flush -- clean up queued files and text		*/
/*									*/
/************************************************************************/


void
DDT_input_flush()
{
   LINE ln;
   FSTACK fs;

   while (all_lines != NULL) {
      ln = all_lines;
      SFREE(ln->text);
      all_lines = ln->next;
      free(ln);
    };

   while (all_files != NULL) {
      fclose(curfile);
      fs = all_files;
      all_files = fs->next;
      curfile = fs->file;
      while (fs->queue != NULL) {
	 ln = fs->queue;
	 SFREE(ln->text);
	 fs->queue = ln->next;
	 free(ln);
       };
      free(fs);
    };
};





/************************************************************************/
/*									*/
/*	DDT_input_line -- get next line of input			*/
/*									*/
/************************************************************************/


String
DDT_input_line(max,out)
   Integer max;
   String out;
{
   Character buf[MAX_LINE_SIZE],rslt[MAX_LINE_SIZE];

   for ( ; ; ) {
      if (!get_next_line(rslt)) return NULL;

      if (quote_it) {
	 buf[0] = 0;
       }
      else {
	 do_alias(rslt);
	 split_command(rslt,buf);
	 DDT_input_enqueue(buf);
	 do_variable(rslt);
	 if (check_builtin(rslt)) continue;
       };

      break;
    };

   if (max > 0) strncpy(out,rslt,max);
   else strcpy(out,max);

   return out;
};






/************************************************************************/
/*									*/
/*	get_next_line -- read next line into buffer			*/
/*									*/
/************************************************************************/


static Boolean
get_next_line(buf)
   String buf;
{
   Integer len,c;
   String s;
   LINE ln;
   FSTACK fs;

   if (all_lines != NULL) {
      ln = all_lines;
      all_lines = ln->next;
      strcpy(buf,ln->text);
      SFREE(ln->text);
      return TRUE;
    };

   if (all_files == NULL) {
      output_prompt();
    };

   len = 0;
   s = buf;

   for ( ; ; ) {
      c = (*input_fct)(curfile);
      if (c == EOF) {
	 if (len == 0) {
	    if (all_files == NULL) return FALSE;
	    else {
	       fclose(curfile);
	       fs = all_files;
	       curfile = fs->file;
	       all_files = fs->next;
	       all_lines = fs->queue;
	       free(fs);
	       if (all_files == NULL) {
		  output_prompt();
		};
	       continue;
	     };
	  };
	 break;
       }
      else if (c == '\n') {
	 break;
       }
      else if (++len < MAX_LINE_SIZE-1) *s++ = c;
    };

   *s = 0;

   do_history(buf,TRUE);
   if (buf[0] != 0 && all_files == NULL) save_history(buf);

   return TRUE;
};




/************************************************************************/
/*									*/
/*	output_prompt -- handle prompting				*/
/*									*/
/************************************************************************/


static void
output_prompt()
{
   String s;
   Character buf[32];

   s = prompt;
   while (*s != 0) {
      if (*s == '!') {
	 sprintf(buf,"%d",cur_history+1);
	 fputs(buf,stdout);
       }
      else putchar(*s);
      ++s;
    };

   fflush(stdout);
};





/************************************************************************/
/*									*/
/*	do_history -- do history substitution in the command		*/
/*									*/
/************************************************************************/


static Boolean
do_history(text,pfg)
   String text;
   Boolean pfg;
{
   Character rslt[MAX_LINE_SIZE],mbuf[1024],rbuf[1024],hbuf[MAX_LINE_SIZE];
   String s,t,h;
   String p,q;
   Boolean nocol,hadhist;
   Integer n,fr,to;

   hadhist = FALSE;
   s = text;
   while (isspace(*s)) ++s;

   t = rslt;

   while (*s != 0) {
      if (*s == '\\') {
	 *t++ = *s++;
	 if (*s != 0) *t++ = *s++;
       }
      else if (*s != '!' && *s != '^') *t++ = *s++;
      else if (*s == '^') {
	 hadhist = TRUE;
	 h = get_history(cur_history-1,hbuf);
	 ++s;
	 p = mbuf;
	 while (*s != 0 && *s != '^') *p++ = *s++;
	 *p = 0;
	 if (*s == '^') ++s;
	 p = rbuf;
	 while (*s != 0 && *s != '^') *p++ = *s++;
	 *p = 0;
	 if (*s == '^') ++s;
	 h = subst_history(h,mbuf,rbuf);
	 if (h == NULL) {
	    printf("DDT: Bad history pattern\n");
	    text[0] = 0;
	    return;
	  };

	 while (*h != 0) *t++ = *h++;
       }
      else if (*s == '!') {
	 hadhist = TRUE;
	 nocol = FALSE;
	 ++s;
	 switch (*s) {
	    case '-' :
	       n = 0;
	       while (isdigit(*s)) n = n*10 + *s++ - '0';
	       sprintf(mbuf,"%d",cur_history-n);
	       h = get_history(cur_history-n,hbuf);
	       nocol = TRUE;
	       break;
	    case '0' : case '1' : case '2' : case '3' : case '4' :
	    case '5' : case '6' : case '7' : case '8' : case '9' :
	       n = 0;
	       while (isdigit(*s)) n = n*10 + *s++ - '0';
	       sprintf(mbuf,"%d",cur_history);
	       h = get_history(n,hbuf);
	       break;
	    case '$' : case '*' : case '^' : case '%' : case ':' :
	       h = get_history(cur_history-1,hbuf);
	       nocol = TRUE;
	       break;
	    case '!' :
	       h = get_history(cur_history-1,hbuf);
	       ++s;
	       break;
	    case '?' :
	       ++s;
	       p = mbuf;
	       while (*s != 0 && *s != '?') *p++ = *s++;
	       if (*s == '?') ++s;
	       *p = 0;
	       h = match_history(mbuf,hbuf);
	       break;
	    default :
	       p = mbuf;
	       while (*s != 0 && *s != ':' && !isspace(*s)) *p++ = *s++;
	       *p = 0;
	       h = match_history(mbuf,hbuf);
	       break;
	  };

	 if (h == NULL) {
	    printf("DDT: History event '%s' not found\n",mbuf);
	    text[0] = 0;
	    return;
	  };

	 if (nocol || *s == ':') {
	    if (*s == ':') ++s;
	    switch (*s) {
	       case '0' : case '1' : case '2' : case '3' : case '4' :
	       case '5' : case '6' : case '7' : case '8' : case '9' :
		  fr = 0;
		  while (isdigit(*s)) fr = fr*10 + *s++ - '0';
		  to = fr;
		  break;
	       case '^' :
		  ++s;
		  fr = to = 1;
		  break;
	       case '$' :
		  ++s;
		  fr = to = -1;
		  break;
	       case '*' :
		  ++s;
		  fr = 1;
		  to = -1;
		  break;
	       default :
		  fr = 0;
		  to = -1;
		  break;
	     };
	    if (*s == '-') {
	       ++s;
	       switch (*s) {
		  case '0' : case '1' : case '2' : case '3' : case '4' :
		  case '5' : case '6' : case '7' : case '8' : case '9' :
		     to = 0;
		     while (isdigit(*s)) to = to*10 + *s++ - '0';
		     break;
		  case '^' :
		     to = 1;
		     break;
		  case '$' :
		     to = -1;
		     break;
		  case '*' :
		     to = -1;
		     break;
		  default :
		     fr = 0;
		     to = -2;
		     break;
		};
	     }
	    else if (*s == '*') to = -1;
	    h = history_words(h,fr,to);
	  };

	 while (*s == ':') {
	    ++s;
	    switch (*s) {
	       case 'h' :
	       case 'r' :
	       case 'e' :
	       case 't' :
		  h = NULL;
		  ++s;
		  break;
	       default :
		  h = NULL;
		  break;
	     };
	  };

	 if (h == NULL) {
	    printf("DDT: Bad history modifier\n");
	    text[0] = 0;
	    return;
	  };

	 while (*h != 0) *t++ = *h++;
       };
    };

   *t = 0;
   if (hadhist) {
      strcpy(text,rslt);
      if (pfg && all_files == NULL) printf("%s\n",text);
    };

   return hadhist;
};





/************************************************************************/
/*									*/
/*	get_history -- get history item by id				*/
/*									*/
/************************************************************************/


static String
get_history(id,buf)
   Integer id;
   String buf;
{
   HISTORY h;

   for (h = last_hist; h != NULL; h = h->last) {
      if (h->number == id) {
	 strcpy(buf,h->value);
	 return buf;
       };
    };

   return NULL;
};





/************************************************************************/
/*									*/
/*	match_history -- find history that matches given item		*/
/*									*/
/************************************************************************/


static String
match_history(pat,buf)
   String pat;
   String buf;
{
   HISTORY h;
   Integer ln;

   ln = strlen(pat);
   for (h = last_hist; h != NULL; h = h->last) {
      if (strncmp(h->value,pat,ln) == 0) {
	 strcpy(buf,h->value);
	 return buf;
       };
    };

   return NULL;
};





/************************************************************************/
/*									*/
/*	subst_history -- do substitution in history string		*/
/*									*/
/************************************************************************/


static String
subst_history(h,pat,rslt)
   String h;
   String pat;
   String rslt;
{
   Character buf[MAX_LINE_SIZE];
   String s,t,p;
   Integer ln;
   Boolean fnd;

   s = h;
   t = buf;
   fnd = FALSE;
   ln = strlen(pat);

   while (*s != 0) {
      if (!fnd && strncmp(s,pat,ln) == 0) {
	 p = rslt;
	 while (*p != 0) *t++ = *p++;
	 s += ln;
	 fnd = TRUE;
       }
      else *t++ = *s++;
    };
   *t = 0;

   if (!fnd) return NULL;

   strcpy(h,buf);

   return h;
};




/************************************************************************/
/*									*/
/*	history_words -- return selective list of words from string	*/
/*									*/
/************************************************************************/


static String
history_words(h,fr,to)
   String h;
   Integer fr,to;
{
   Character buf[MAX_LINE_SIZE];
   String args[MAXARGS];
   String s,t;
   Integer i,wct,qfg;

   if (fr == 0 && to == -1) return h;

   wct = 0;
   s = h;
   qfg = 0;
   t = buf;
   args[wct] = t;
   while (isspace(*s)) ++s;

   while (*s != 0) {
      if (qfg != 0 && *s == qfg) {
	 qfg = 0;
	 *t++ = *s++;
       }
      else if (*s == '\\') {
	 *t++ = *s++;
	 if (*s != 0) *t++ = *s++;
       }
      else if (qfg == 0 && (*s == '\'' || *s == '"')) {
	 qfg = *s;
	 *t++ = *s++;
       }
      else if (qfg != 0) *t++ = *s++;
      else if (isspace(*s)) {
	 if (t != args[wct]) {
	    *t++ = 0;
	    ++wct;
	    args[wct] = t;
	  };
	 ++s;
       }
      else if (*s == ';' || *s == '>' || *s == '<') {
	 if (t != args[wct]) {
	    *t++ = 0;
	    ++wct;
	    args[wct] = t;
	  };
	 *t++ = *s++;
       }
      else *t++ = *s++;
    };

   if (t != args[wct]) {
      *t++ = 0;
      ++wct;
      args[wct] = t;
    };

   if (fr < 0) fr += wct;
   if (to < 0) to += wct;
   if (to >= wct) to = wct-1;

   s = h;
   wct = 0;
   for (i = fr; i <= to; ++i) {
      if (wct != 0) *s++ = ' ';
      t = args[i];
      while (*t != 0) *s++ = *t++;
    };
   *s = 0;

   return h;
};





/************************************************************************/
/*									*/
/*	save_history -- save command for history purposes		*/
/*	pop_history -- unsave last history command			*/
/*									*/
/************************************************************************/


static void
save_history(text)
   String text;
{
   HISTORY h;

   while (isspace(*text)) ++text;

   if (*text == 0) return;

   h = PALLOC(HISTORY_INFO);
   h->value = SALLOC(text);
   h->number = ++cur_history;
   h->next = NULL;
   h->last = last_hist;
   if (last_hist != NULL) last_hist->next = h;
   else first_hist = h;
   last_hist = h;

   while (first_hist != NULL && first_hist->number < cur_history - history_cnt) {
      h = first_hist;
      first_hist = h->next;
      if (first_hist != NULL) first_hist->last = NULL;
      SFREE(h->value);
      free(h);
    };
};





static void
pop_history()
{
   HISTORY h;

   h = last_hist;
   last_hist = h->last;
   if (last_hist != NULL) last_hist->next = NULL;
   else first_hist = NULL;
   SFREE(h->value);
   free(h);
};





/************************************************************************/
/*									*/
/*	do_alias -- handle aliases					*/
/*									*/
/************************************************************************/


static void
do_alias(text)
   String text;
{
   ALIAS a;
   String s;
   Character buf[MAX_LINE_SIZE];
   Integer lvl;


   for (lvl = 0; lvl < 10; ++lvl) {
      s = text;
      while (isspace(*s)) ++s;
      for (a = all_alias; a != NULL; a = a->next) {
	 if (strncmp(s,a->name,a->length) == 0 &&
		(isspace(s[a->length]) || s[a->length] == 0)) {
	    save_history(text);
	    s += a->length;
	    while (isspace(*s)) ++s;
	    strcpy(buf,a->value);
	    if (!do_history(buf,FALSE)) {
	       strcat(buf," ");
	       strcat(buf,s);
	     };
	    pop_history();
	    break;
	  };
       };
      if (a == NULL) break;
      strcpy(text,buf);
    };


   if (lvl >= 10) {
      printf("DDT: Recursive alias\n");
      text[0] = 0;
    };

};





/************************************************************************/
/*									*/
/*	do_variable -- do variable substitution 			*/
/*									*/
/************************************************************************/


static void
do_variable(text)
   String text;
{
   Character buf[MAX_LINE_SIZE],what[1024];
   String s,t,p;
   Integer qfg,endch;
   VAR v;

   s = text;
   t = buf;
   qfg = 0;

   while (*s != 0) {
      if (qfg != 0 && *s == qfg) { qfg = 0; ++s; }
      else if (*s == '\\' && s[1] != 0) {
	 ++s;
	 *t++ = *s++;
       }
      else if (qfg == '\'') *t++ = *s++;
      else if (*s == '$' && s[1] == '$' && s[2] != 0 && !isspace(s[2])) {
	 s += 2;
	 p = what;
	 if (*s == '{') { endch = '}';  ++s; }
	 else endch = 0;
	 while (isalnum(*s) && *s != endch) *p++ = *s++;
	 *p = 0;
	 if ((endch != 0 && *s != endch && *s != ':') || what[0] == 0) {
	    printf("DDT: Bad variable syntax\n");
	    text[0] = 0;
	    return;
	  };
	 for (v = all_vars; v != NULL; v = v->next) {
	    if (STREQL(v->name,what)) break;
	  };
	 if (v == NULL) {
	    printf("DDT: Undefined variable %s\n",what);
	    text[0] = 0;
	    return;
	  };
	 p = v->value;
	 if (*s == ':') {
	    ++s;
	    switch (*s) {
	       case 'h' :
	       case 'r' :
	       case 'e' :
	       case 't' :
		  p = NULL;
		  ++s;
		  break;
	       default :
		  p = NULL;
		  break;
	     };
	  };
	 if (p == NULL || (endch != 0 && *s != endch)) {
	    printf("DDT: Bad variable syntax\n");
	    text[0] = 0;
	    return;
	  };
	 if (endch != 0 && *s == endch) ++s;
	 while (*p != 0) *t++ = *p++;
	}
      else if (*s == '"' || *s == '\'') qfg = *s++;
      else *t++ = *s++;
    };
   *t = 0;

   strcpy(text,buf);
};





/************************************************************************/
/*									*/
/*	check_builtin -- handle built in commands			*/
/*									*/
/************************************************************************/


typedef struct _BUILTIN {
   String name;
   voidFunction_Ptr rtn;
} BUILTIN;


static BUILTIN our_commands[] = {
   { "alias", handle_alias },
   { "history", handle_history },
   { "set", handle_set },
   { "source", handle_source },
   { "unalias", handle_unalias },
   { "unset", handle_unset },
   { NULL, NULL }
};




static Boolean
check_builtin(text)
   String text;
{
   String s;
   Integer i,j;

   s = text;
   while (isspace(*s)) ++s;

   for (i = 0; our_commands[i].name != NULL; ++i) {
      j = strlen(our_commands[i].name);
      if (strncmp(s,our_commands[i].name,j) == 0 && (isspace(s[j]) || s[j] == 0)) {
	 s = &s[j];
	 while (isspace(*s)) ++s;
	 (*our_commands[i].rtn)(s);
	 return TRUE;
       };
    };

   return FALSE;
};




/************************************************************************/
/*									*/
/*	handle_alias -- handle the alias command			*/
/*									*/
/************************************************************************/


static void
handle_alias(args)
   String args;
{
   Character what[1024];
   Character buf[MAX_LINE_SIZE];
   ALIAS a;
   String s,t;
   Integer i,qfg;

   if (args[0] == 0) {
      for (a = all_alias; a != NULL; a = a->next) {
	 printf("%s\t%s\n",a->name,a->value);
       };
      return;
    };

   s = args;
   t = what;

   while (*s != 0 && !isspace(*s)) *t++ = *s++;
   *t = 0;
   while (isspace(*s)) ++s;

   if (*s == 0) {
      for (a = all_alias; a != NULL; a = a->next) {
	 if (STREQL(a->name,what)) {
	    printf("%s\t%s\n",a->name,a->value);
	    break;
	  };
       };
      return;
    };

   t = buf;
   qfg = 0;

   while (*s != 0) {
      if (qfg != 0 && *s == qfg) { qfg = 0; ++s; }
      else if (*s == '\\' && s[1] != 0) {
	 ++s;
	 *t++ = *s++;
       }
      else if (qfg != 0) *t++ = *s++;
      else if (*s == '"' || *s == '\'') qfg = *s++;
      else *t++ = *s++;
    };
   *t = 0;

   DDT_input_set_alias(what,buf);
};





/************************************************************************/
/*									*/
/*	handle_unalias -- handle unalias command			*/
/*									*/
/************************************************************************/


static void
handle_unalias(args)
   String args;
{
   if (args[0] != 0) {
      DDT_input_set_alias(args,NULL);
    };
};






/************************************************************************/
/*									*/
/*	handle_set -- handle the set command				*/
/*									*/
/************************************************************************/


static void
handle_set(args)
   String args;
{
   Character what[1024];
   Character buf[MAX_LINE_SIZE];
   VAR v;
   String s,t;
   Integer i,qfg;

   if (args[0] == 0) {
      for (v = all_vars; v != NULL; v = v->next) {
	 printf("%s\t%s\n",v->name,v->value);
       };
      return;
    };

   s = args;
   t = what;

   while (*s != 0 && !isspace(*s)) *t++ = *s++;
   *t = 0;
   while (isspace(*s)) ++s;

   if (*s == 0) {
      for (v = all_vars; v != NULL; v = v->next) {
	 if (STREQL(v->name,what)) {
	    printf("%s\t%s\n",v->name,v->value);
	    break;
	  };
       };
      return;
    };

   t = buf;
   qfg = 0;

   while (*s != 0) {
      if (qfg != 0 && *s == qfg) { qfg = 0; ++s; }
      else if (*s == '\\' && s[1] != 0) {
	 ++s;
	 *t++ = *s++;
       }
      else if (qfg != 0) *t++ = *s++;
      else if (*s == '"' || *s == '\'') qfg = *s++;
      else *t++ = *s++;
    };
   *t = 0;

   DDT_input_set_var(what,buf);
};





/************************************************************************/
/*									*/
/*	handle_unalias -- handle unalias command			*/
/*									*/
/************************************************************************/


static void
handle_unset(args)
   String args;
{
   if (args[0] != 0) {
      DDT_input_set_var(args,NULL);
    };
};






/************************************************************************/
/*									*/
/*	handle_history -- handle history command			*/
/*									*/
/************************************************************************/


static void
handle_history(args)
   String args;
{
   HISTORY h;

   if (isdigit(args[0])) {
      history_cnt = atol(args);
    }
   else {
      for (h = first_hist; h != NULL; h = h->next) {
	 printf("%d %s\n",h->number,h->value);
       };
    };
};





/************************************************************************/
/*									*/
/*	handle_source -- handle source command				*/
/*									*/
/************************************************************************/


static void
handle_source(args)
   String args;
{
   DDT_input_source(args);
};





/************************************************************************/
/*									*/
/*	split_command -- split line to get off first command		*/
/*									*/
/************************************************************************/


static void
split_command(frst,rest)
   String frst;
   String rest;
{
   String s;
   Integer qfg;

   rest[0] = 0;

   if (end_of_line != 0) {
      s = frst;
      while (*s != 0) {
	 if (*s == end_of_line) break;
	 if (qfg != 0 && *s == qfg) qfg = 0;
	 else if (qfg == 0 && (*s == '"' || *s == '\'')) qfg = *s;
	 else if (qfg == 0 && *s == '\\' && s[1] != 0) ++s;
	 ++s;
       };
      if (*s != 0) {
	 *s++ = 0;
	 strcpy(rest,s);
       };
    };
};





/* end of ddtinput.c */
