/************************************************************************/
/*									*/
/*		dbgwin.c						*/
/*									*/
/*	Window manipulation routines for debugger interface		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "dbg_local.h"
#include <leaf.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define NOSYS_MSG	"DEBUG NO SYSTEM %1s"
#define FOCUS_MSG	"DEBUG FOCUS %1s %2s %3s %4d"
#define AT_MSG		"DEBUG AT %1s %2s %3s %4d"
#define EXIT_MSG	"DEBUG FINISH %1s"
#define RUN_MSG 	"DEBUG START %1s"
#define CONT_MSG	"DEBUG CONTINUE %1s"
#define ERROR_MSG	"DEBUG STOP_ERROR %1s %s %s %d %d %2s"
#define NEWSYS_MSG	"DEBUG NEWSYS %1s %2s"
#define PICK_MSG	"DEBUG PICK %1s %2s 2 %3d %4r"

#define EXTRA_BUTTONS	3




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;
static	Boolean 	popup_error;
static	Integer 	num_default_btn;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	dbg_setup_window();
static	int		dbg_control();
static	void		check_windows();
static	int		dbg_edit_control();
static	void		dbg_remove();
static	void		handle_nosys();
static	void		handle_newsys();
static	void		handle_exit();
static	void		handle_focus();
static	void		handle_at();
static	void		handle_run();
static	void		handle_cont();
static	void		handle_error();
static	void		handle_pick();
static	String		get_system_name();
static	void		format_location();
static	int		quit_btn();
static	Integer 	set_btn_size();






/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define BTTN_WIN	2
#define DEDT_WIN	3
#define STS0_AREA	4
#define STS1_AREA	5



static	LEAF_DATA	dbg_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DEDT_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(BTTN_WIN),
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   { BTTN_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(STS0_AREA),
	     LEAF_COORD_RX, LEAF_COORD_SIZE_LINE(LEAF_VALUE_PARAM(0)) },
	NULL, NULL },
   { STS0_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(STS1_AREA),
	     LEAF_COORD_RX, LEAF_COORD_TEXT_LINE },
	NULL, NULL },
   { STS1_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_TEXT_LINE },
	NULL, NULL },
   LEAF_END
};



static STEM_PDM_DATA	dbg_menus[] = {
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX,  "Buttons",      NULL },
   { STEM_PSTATE_BTN,	"Remove",       DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Add",          DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Modify",       DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Reload",       DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Default",      DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Save Local",   DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Save Global",  DBG_btn_menu },
   { STEM_PSTATE_BTN,	"Quit",         quit_btn },

   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	DBG_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DBG_win_init()
{
   AUXD hdl,db;
   String msg;

   all_windows = NULL;

   hdl = AUXDget_handle(NULL,"DBG");

   popup_error = AUXDget_defined(hdl,"POPUP_ERROR");

   num_default_btn = 0;
   for (db = AUXDget_handle(hdl,"BUTTON"); db != NULL; db = AUXDnext_handle(db))
      ++num_default_btn;

   msg = AUXDget_info(hdl,"NOSYS_MSG");
   if (msg != NULL) MSGregister(msg,handle_nosys,1,NULL);

   msg = AUXDget_info(hdl,"NEWSYS_MSG");
   if (msg != NULL) MSGregister(msg,handle_newsys,2,NULL);

   msg = AUXDget_info(hdl,"FOCUS_MSG");
   if (msg != NULL) MSGregister(msg,handle_focus,4,NULL);

   msg = AUXDget_info(hdl,"AT_MSG");
   if (msg != NULL) MSGregister(msg,handle_at,4,NULL);

   msg = AUXDget_info(hdl,"EXIT_MSG");
   if (msg != NULL) MSGregister(msg,handle_exit,1,NULL);

   msg = AUXDget_info(hdl,"RUN_MSG");
   if (msg != NULL) MSGregister(msg,handle_run,1,NULL);

   msg = AUXDget_info(hdl,"CONT_MSG");
   if (msg != NULL) MSGregister(msg,handle_cont,1,NULL);

   msg = AUXDget_info(hdl,"ERROR_MSG");
   if (msg != NULL) MSGregister(msg,handle_error,2,NULL);

   msg = AUXDget_info(hdl,"PICK_MSG");
   if (msg != NULL) MSGregister(msg,handle_pick,4,NULL);
};





/************************************************************************/
/*									*/
/*	DBG_create -- create a new debugger window			*/
/*									*/
/************************************************************************/


DBG_WIN
DBG_create(w,typ,file,cfile)
   ASH_WINDOW w;
   String typ;
   String file;
   String cfile;
{
   register DBG_WIN dw;
   Character buf[64];
   String core;
   String s,t;

   if (file == NULL || *file == NULL) {
      file = get_system_name(w,&core);
      if (file == NULL) return NULL;
      cfile = NULL;
    }
   else {
      file = SALLOC(file);
    };

   if (cfile != NULL) core = SALLOC(cfile);
   else {
      core = index(file,' ');
      if (core != NULL) {
	 while (isspace(*core)) *core++ = 0;
	 if (*core == 0) core = NULL;
       };
    };

   dw = PALLOC(DBG_WIN_INFO);
   dw->window = w;
   dw->filename = file;
   dw->corename = core;
   dw->exec_info = NULL;
   dw->buttons = NULL;
   dw->function = NULL;
   dw->line = 0;
   dw->use_gdb = FALSE;

   if (typ == NULL) typ = "dbg";
   if (typ != NULL && STREQL(typ,"gdbg")) dw->use_gdb = TRUE;

   dw->btn_size = set_btn_size(dw);

   PROTECT;
   all_windows = CONS(dw,all_windows);
   UNPROTECT;

   ASHset_window_id(w,"dbg");
   ASHset_window_defaults(w);

   if (!dbg_setup_window(dw)) return NULL;

   if (dw->menu_win != NULL) {
      STEMpdm_define(dw->menu_win,dw,dbg_menus);
    };

   DBG_btn_setup(dw);

   MSGcommand_name("ddt",buf);
   if (getenv("DDT") != NULL) strcpy(buf,getenv("DDT"));

   if (dw->corename != NULL && dw->corename[0] == 0) dw->corename = NULL;

   t = (dw->use_gdb ? "gddt" : "ddt");

   s = MSGinq_lock_file();
   if (s == NULL) {
      dw->exec_info = EDTexec(dw->dedt_win,dw->menu_win,NULL,buf,t,"-f","-p",
				 dw->filename,dw->corename,0);
    }
   else {
      dw->exec_info = EDTexec(dw->dedt_win,dw->menu_win,NULL,buf,t,"-f","-p",
				 "-msg",s, dw->filename,dw->corename,0);
    };

   EDTexec_prohibit_close(dw->exec_info);

   return dw;
};





/************************************************************************/
/*									*/
/*	dbg_setup_window -- define the portions of an annotation window */
/*									*/
/************************************************************************/


static Boolean
dbg_setup_window(dw)
   DBG_WIN dw;
{
   Character buf[1024];

   LEAFsetup_window(dw->window,dbg_leaf,dw);
   LEAFset_parameters(dw->window,1,dw->btn_size);

   LEAFset_control(dw->window,dbg_control);
   LEAFredraw(dw->window);

   sprintf(buf,"dbg: %s",dw->filename);
   ASHset_window_name(dw->window,buf);

   dw->dedt_win = LEAFinq_window(dw->window,DEDT_WIN,0);
   if (dw->dedt_win == NULL) {
      printf("Window too small to run debugger interface\n");
      ASHremove(dw->window);
      return FALSE;
    };

   ASHset_control(dw->dedt_win,dbg_edit_control);
   LEAFset_refresh(dw->window,check_windows);

   dw->menu_win = LEAFinq_window(dw->window,MENU_WIN,0);
   dw->btn_win = LEAFinq_window(dw->window,BTTN_WIN,0);

   BIOnew_input_window(dw->window);
   BIOset_cursor_standard(dw->window,ASH_CURSOR_SMALL_BULLSEYE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	dbg_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
dbg_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register Sequence l;
   register DBG_WIN dw;

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->window == w) break;
    };

   if (dw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      dbg_remove(dw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   DBG_WIN dw;
   ASH_WINDOW ew;
   Sequence l;
   Integer i;

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->window == w) break;
    };

   if (dw == NULL) return;

   ew = LEAFinq_window(w,DEDT_WIN,0);

   if (ew != dw->dedt_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };

   i = set_btn_size(dw);
   if (i != dw->btn_size) {
      dw->btn_size = i;
      LEAFset_parameters(dw->window,1,dw->btn_size);
      LEAFredraw(dw->window);
    };
};





/************************************************************************/
/*									*/
/*	dbg_edit_control -- handle control messages from editor win	*/
/*									*/
/************************************************************************/


static int
dbg_edit_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register DBG_WIN dw;
   Sequence l;

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->dedt_win == w) break;
    };

   if (dw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove(dw->window);
    };

   return ASH_CONTROL_REJECT;
};






/************************************************************************/
/*									*/
/*	dbg_remove -- remove given debugger interface			*/
/*									*/
/************************************************************************/


static void
dbg_remove(dw)
   DBG_WIN dw;
{
   PROTECT;

   all_windows = REMOB(dw,all_windows);

   if (dw->exec_info != NULL) EDTexec_stop(dw->exec_info);

   UNPROTECT;

   dw->window = NULL;

   if (dw->filename != NULL) free(dw);
};





/************************************************************************/
/*									*/
/*	handle_nosys -- handle a no system message			*/
/*	handle_exit -- handle exit message				*/
/*	handle_newsys -- handle a new system (changed system) message	*/
/*									*/
/************************************************************************/


static void
handle_nosys(sys,rid)
   String sys;
   Integer rid;
{
   DBG_WIN dw;
   Character buf[256];
   Sequence l;
   String core;

   if (rid >= 0) MSGreply(rid,NULL);
   MSGsync_allow();

   strcpy(buf,sys);

   forin (dw,DBG_WIN,l,all_windows) {
      if (FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL || dw->window == NULL) return;

   ASHset_window_name(dw->window,"dbg: <No System>");

   dw->filename = NULL;
   while (dw->filename == NULL && dw->window != NULL) {
      dw->filename = get_system_name(dw->window,&core);
      dw->corename = core;
    };

   if (dw->filename != NULL) {
      if (dw->corename == NULL) {
	 MSGsenda("DDT SET %s NEWSYS %s",buf,dw->filename);
       }
      else {
	 MSGsenda("DDT SET %s NEWSYS %c%s %s%c",buf,
		     LIT_STRING,dw->filename,dw->corename,LIT_STRING);
       };
    };
};





static void
handle_newsys(sys,new,rid)
   String sys;
   String new;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && dw->filename[0] != 0 && STRNEQ(dw->filename,"*") &&
	     sys != NULL && STREQL(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   dw->filename = SALLOC(new);
   dw->corename = NULL;
};





static void
handle_exit(sys,rid)
   String sys;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   DBG_exit(dw);
};





/************************************************************************/
/*									*/
/*	handle_focus -- handle focus change message			*/
/*	handle_at -- handle location change message			*/
/*	handle_run -- handle running message				*/
/*	handle_cont -- handle continue message				*/
/*	handle_error -- handle stop on error message			*/
/*									*/
/************************************************************************/


static void
handle_focus(sys,fil,fun,line,rid)
   String sys;
   String fil;
   String fun;
   Integer line;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;
   Character buf[1024];
   String s;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   if (dw->function != NULL) SFREE(dw->function);
   dw->function = SALLOC(fun);
   dw->line = line;

   format_location(fil,fun,line,"FOCUS: ",buf);

   s = (String) LEAFinq_data(dw->window,STS1_AREA,0);
   if (s != NULL && STREQL(s,buf)) return;

   LEAFset_data(SALLOC(buf),dw->window,STS1_AREA,0);
   if (s != NULL) SFREE(s);

   LEAFredraw(dw->window);
};





static void
handle_at(sys,fil,fun,line,rid)
   String sys;
   String fil;
   String fun;
   Integer line;
{
   DBG_WIN dw;
   Sequence l;
   Character buf[1024];
   String s;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   format_location(fil,fun,line,"AT:    ",buf);

   s = (String) LEAFinq_data(dw->window,STS0_AREA,0);
   if (s != NULL && STREQL(s,buf)) return;

   LEAFset_data(SALLOC(buf),dw->window,STS0_AREA,0);
   if (s != NULL) SFREE(s);

   LEAFredraw(dw->window);

   DBG_btn_mode(dw,DBG_MODE_STOP);
};





static void
handle_run(sys,rid)
   String sys;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;
   Character buf[1024];
   String s;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   sprintf(buf,"Execution starting ...");

   s = (String) LEAFinq_data(dw->window,STS0_AREA,0);
   if (s != NULL && STREQL(s,buf)) return;

   LEAFset_data(SALLOC(buf),dw->window,STS0_AREA,0);
   if (s != NULL) SFREE(s);

   LEAFredraw(dw->window);

   DBG_btn_mode(dw,DBG_MODE_RUN);
};





static void
handle_cont(sys,rid)
   String sys;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;
   Character buf[1024];
   String s;

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL) return;

   sprintf(buf,"Execution continues ...");

   s = (String) LEAFinq_data(dw->window,STS0_AREA,0);
   if (s != NULL && STREQL(s,buf)) return;

   LEAFset_data(SALLOC(buf),dw->window,STS0_AREA,0);
   if (s != NULL) SFREE(s);

   LEAFredraw(dw->window);

   DBG_btn_mode(dw,DBG_MODE_CONTINUE);
};





static void
handle_error(sys,msg,rid)
   String sys;
   String msg;
   Integer rid;
{
   DBG_WIN dw;
   Sequence l;
   Character buf[10240];

   if (rid >= 0) MSGreply(rid,NULL);

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL || dw->window == NULL || !popup_error) return;

   if (islower(msg[0])) msg[0] = toupper(msg[0]);

   sprintf(buf,"%s\n\n%%M   %%a\n",msg);
   STEMdialog1(dw->window,buf);
};





/************************************************************************/
/*									*/
/*	handle_pick -- handle pick alternative message			*/
/*									*/
/************************************************************************/


static void
handle_pick(sys,orig,ct,alts,rid)
   String sys;
   String orig;
   Integer ct;
   String alts;
   Integer rid;
{
   String s,p;
   Character buf[1024],menu[10240];
   Integer i;
   Sequence l;
   DBG_WIN dw;

   forin (dw,DBG_WIN,l,all_windows) {
      if (dw->filename != NULL && FIELDsystem_match(sys,dw->filename)) break;
    };

   if (dw == NULL || dw->window == NULL) {
      if (rid >= 0) MSGreply(rid,NULL);
      return;
    };

   MSGsync_allow();
   sprintf(menu,"%%CChoose proper alternative for %s:\n\n",orig);

   s = alts;
   for (i = 0; i < ct; ++i) {
      sprintf(buf,"%%0.%do ",i);
      strcat(menu,buf);

      p = buf;

      while (*s != 0 && *s != ';') *p++ = *s++;
      if (*s == ';') ++s;
      *p = 0;

      strcat(menu,buf);
      strcat(menu,"\n");
    };

   strcat(menu,"\n   %a%M   %c");

   i = 0;
   if (!STEMdialog1(dw->window,menu,&i)) i = -1;

   if (rid >= 0) {
      if (i < 0) MSGreply(rid,NULL);
      else {
	 sprintf(buf,"%d",i);
	 MSGreply(rid,buf);
       };
    };
};






/************************************************************************/
/*									*/
/*	get_system_name -- get system name from user			*/
/*									*/
/************************************************************************/


static String
get_system_name(w,core)
   ASH_WINDOW w;
   String * core;
{
   String menu[10];
   Character buf[1024],cbuf[1024];
   String file;

   *core = NULL;

   menu[0] = "%CDebugger Window\n\n";
   menu[1] = "Debug: %0.48t\n";
   menu[2] = "Core:  %1.48t\n";
   menu[3] = "   %a%M   %c";
   menu[4] = 0;
   buf[0] = 0;
   cbuf[0] = 0;

   if (!STEMdialog(w,menu,buf,cbuf)) {
      if (ASHinq_valid_window(w)) ASHremove(w);
      return NULL;
    };
   if (buf[0] == 0) return NULL;

   file = SALLOC(buf);

   if (cbuf[0] != 0) *core = SALLOC(cbuf);

   sprintf(cbuf,"dbg: %s",file);
   ASHset_window_name(w,cbuf);

   return file;
};






/************************************************************************/
/*									*/
/*	format_location -- format a location string			*/
/*									*/
/************************************************************************/


static void
format_location(file,func,line,pfx,buf)
   String file;
   String func;
   Integer line;
   String pfx;
   String buf;
{
   Character abf[1024];

   if (line != 0 && STRNEQ(func,"*")) {
      sprintf(abf,"%sat line %d of routine %s",pfx,line,func);
    }
   else if (STRNEQ(func,"*")) {
      sprintf(abf,"%sin routine %s",pfx,func);
    }
   else if (line != 0) {
      sprintf(abf,"%sat line %d",pfx,line);
    }
   else strcpy(abf,pfx);

   if (STRNEQ(file,"*")) {
      sprintf(buf,"%s in file %s",abf,file);
    }
   else strcpy(buf,abf);
};





/************************************************************************/
/*									*/
/*	quit_btn -- handle quit request 				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
quit_btn(dw,mnm,btn)
   DBG_WIN dw;
   String mnm;
   String btn;
{
   ASHremove(dw->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	set_btn_size -- compute # lines of buttons to display		*/
/*									*/
/************************************************************************/


static Integer
set_btn_size(dw)
   DBG_WIN dw;
{
   Integer i;
   Integer bpl;
   Integer lx,by,rx,ty;
   DBG_BTN db;

   if (dw->buttons == NULL) i = num_default_btn;
   else {
      i = 0;
      for (db = dw->buttons; db != NULL; db = db->next) ++i;
    };

   if (ASHinq_valid_window(dw->window)) {
      ASHinq_size(dw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      bpl = BUTTONS_PER_LINE*abs(rx-lx)/500;
    }
   else bpl = BUTTONS_PER_LINE;

   i = (i+bpl-1)/bpl;
   i = DBG__button_line_size*i+5;

   return i;
};






/* end of dbgwin.c */
