/************************************************************************/
/*									*/
/*		cbrowwin.c						*/
/*									*/
/*	Window manipulation routines for class browser			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "cbrow_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	cb_setup_window();
static	void		cb_check_windows();
static	int		cbrow_control();
static	void		cbrow_remove();
static	String		cb_request_system();
static	void		handle_cbrow_exec();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define DISP_WIN	2



static	LEAF_DATA	cbrow_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DISP_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	CBROW_win_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
CBROW_win_init()
{
   AUXD hdl;
   String msg;

   all_windows = NULL;

   hdl = AUXDget_handle(NULL,"CBROW");

   if (hdl != NULL) {
      msg = AUXDget_info(hdl,"DEBUG_AT");
      if (msg != NULL) MSGregister(msg,handle_cbrow_exec,3,NULL);
      msg = AUXDget_info(hdl,"DEBUG_ENTER");
      if (msg != NULL) MSGregister(msg,handle_cbrow_exec,3,NULL);
    };
};





/************************************************************************/
/*									*/
/*	CBROW_create -- create a new cbrow graph window 		*/
/*									*/
/************************************************************************/


CBROW_WIN
CBROW_create(w,name)
   ASH_WINDOW w;
   String name;
{
   register CBROW_WIN cw;
   Character buf[256];

   if (name == NULL) name = cb_request_system(w);
   else name = SALLOC(name);

   if (name == NULL) return NULL;

   sprintf(buf,"cbrowse: %s",name);
   ASHset_window_name(w,buf);
   ASHset_window_id(w,"CBROW");
   ASHset_window_defaults(w);

   cw = PALLOC(CBROW_WIN_INFO);
   cw->window = w;
   cw->system = name;
   cw->menu_win = NULL;
   cw->disp_win = NULL;
   cw->text_win = NULL;
   cw->edit_info = NULL;
   cw->classes = NULL;
   cw->strings = NULL;
   cw->gid = NULL;
   cw->display = NULL;

   PROTECT;
   all_windows = CONS(cw,all_windows);
   UNPROTECT;

   if (!cb_setup_window(cw)) return NULL;

   if (ASHinput_lock(cw->window,TRUE) != ASH_LOCK_GRANTED) return NULL;

   CBROW_class_setup(cw);
   CBROW_disp_setup(cw);
   CBROW_menu_setup(cw);

   CBROW_disp_redraw(cw);

   if (cw->disp_win != NULL) ASHvisible(cw->disp_win,TRUE);

   ASHinput_unlock(cw->window);

   if (cw->classes == NULL) return NULL;

   return cw;
};





/************************************************************************/
/*									*/
/*	CBROW_find_window -- find cbrow window given ash window 	*/
/*	CBROW_find_system -- find cbrow window given system name	*/
/*									*/
/************************************************************************/


CBROW_WIN
CBROW_find_window(w)
   ASH_WINDOW w;
{
   Sequence l;
   CBROW_WIN cw;

   forin (cw,CBROW_WIN,l,all_windows) {
      if (cw->window == w) break;
      else if (cw->disp_win == w) break;
      else if (cw->menu_win == w) break;
      else if (cw->text_win == w) break;
    };

   return cw;
};





CBROW_WIN
CBROW_find_system(sys)
   String sys;
{
   Sequence l;
   CBROW_WIN cw;

   forin (cw,CBROW_WIN,l,all_windows) {
      if (FIELDsystem_match(cw->system,sys)) break;
    };

   return cw;
};





/************************************************************************/
/*									*/
/*	cb_setup_window -- define the portions of an cbrow window	*/
/*									*/
/************************************************************************/


static Boolean
cb_setup_window(cw)
   CBROW_WIN cw;
{
   LEAFsetup_window(cw->window,cbrow_leaf,cw);

   LEAFset_control(cw->window,cbrow_control);
   LEAFredraw(cw->window);

   cw->disp_win = LEAFinq_window(cw->window,DISP_WIN,0);
   if (cw->disp_win == NULL) {
      fprintf(stderr,"CBROW: Window too small to run cbrow graph viewer\n");
      ASHremove(cw->window);
      return FALSE;
   };

   cw->menu_win = LEAFinq_window(cw->window,MENU_WIN,0);

   LEAFset_refresh(cw->window,cb_check_windows);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	cb_check_windows -- check for user doing a stupid resize	*/
/*									*/
/************************************************************************/


static void
cb_check_windows(w)
   ASH_WINDOW w;
{
   CBROW_WIN cw;
   ASH_WINDOW dw;

   cw = CBROW_find_window(w);

   if (cw == NULL) return;

   dw = LEAFinq_window(w,DISP_WIN,0);

   if (dw != cw->disp_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	cbrow_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
cbrow_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register CBROW_WIN cw;

   cw = CBROW_find_window(w);

   if (cw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      cbrow_remove(cw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	cbrow_remove -- remove given cbrow				*/
/*									*/
/************************************************************************/


static void
cbrow_remove(cw)
   CBROW_WIN cw;
{
   PROTECT;
   all_windows = REMOB(cw,all_windows);
   UNPROTECT;

   cw->window = NULL;
   if (cw->text_win != NULL && ASHinq_valid_window(cw->text_win))
      ASHremove(cw->text_win);
 };





/************************************************************************/
/*									*/
/*	cb_request_system -- request system name from user		*/
/*									*/
/************************************************************************/


static String
cb_request_system(w)
   ASH_WINDOW w;
{
   String systems[32];
   register Integer i,ct,sct;
   Character btn[32][64];
   Character nbuf[64];
   String menu[40];
   Integer val;
   String name;

   sct = FIELDinq_systems(32,systems);
   if (sct == 1) return systems[0];

   ct = 0;
   menu[ct++] = "%CClass Browser\n";

   menu[ct++] = "System: %1.32t\n";

   for (i = 0; i < sct; ++i) {
      sprintf(btn[i],"    %%0.%do %s",i,systems[i]);
      menu[ct++] = btn[i];
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   val = 0;
   nbuf[0] = 0;

   if (!STEMdialog(w,menu,&val,nbuf)) return NULL;

   if (nbuf[0] != 0 && nbuf[0] != ' ') {
      name = SALLOC(nbuf);
    }
   else if (val < sct) name = systems[val];
   else name = NULL;

   return name;
};





/************************************************************************/
/*									*/
/*	handle_cbrow_exec -- handle DEBUG AT messages			*/
/*									*/
/************************************************************************/


static void
handle_cbrow_exec(sys,fil,fun,rid)
   String sys;
   String fil;
   String fun;
   Integer rid;
{
   CBROW_WIN cw;
   CBROW_MEMB cm;

   if (rid >= 0) MSGreply(rid,NULL);

   cw = CBROW_find_system(sys);
   if (cw == NULL) return;

   if (ASHinput_lock(cw->window,FALSE) != ASH_LOCK_GRANTED) return;

   cm = CBROW_class_find_member(cw,fun);

   CBROW_disp_eval(cw,cm);

   ASHinput_unlock(cw->window);
};






/* end of cbrowwin.c */
