/************************************************************************/
/*									*/
/*		buildwin.c						 */
/*									*/
/*	Window manipulation routines for building windows		 */
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "build_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	void		check_windows();
static	int		build_control();
static	void		build_remove();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define PROJECT_AREA	2
#define FILE_AREA	3
#define STATUS_AREA	4
#define DISPLAY_WIN	5



static	LEAF_DATA	build_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { PROJECT_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	BUILD_button_project, (int) "Project:  " },
   { FILE_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(PROJECT_AREA) },
	BUILD_button_file, (int) "File:  " },
   { STATUS_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(FILE_AREA) },
	BUILD_button_display, (int) "Display:  NONE"  },
   { DISPLAY_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(STATUS_AREA) },
	NULL, NULL },
   LEAF_END
};





static	LEAF_DATA	transcript_leaf[] = {
   LEAF_ROOT(NULL),
   { PROJECT_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT, LEAF_COORD_RX, LEAF_COORD_TY },
	BUILD_button_project, (int) "Project:  " },
   { DISPLAY_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT_LINE(PROJECT_AREA) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	BUILD_win_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
BUILD_win_init()
{
   all_windows = NULL;
};





/************************************************************************/
/*									*/
/*	BUILD_create -- create a new make interface window		*/
/*									*/
/************************************************************************/


BUILD_WIN
BUILD_create(w,proj,tonly)
   ASH_WINDOW w;
   String proj;
   Boolean tonly;
{
   BUILD_WIN bw;
   BUILD_PROJ bp;
   BUILD_FILE bf;

   bw = PALLOC(BUILD_WIN_INFO);
   bw->window = w;
   bw->willow_fg = FALSE;
   bw->disp_win = NULL;
   bw->project = NULL;
   bw->file = NULL;
   bw->edit_info = NULL;
   bw->disp_type = BUILD_DISP_NONE;
   bw->trans_only = tonly;

   bw->method = DEFAULT_METHOD;
   bw->connmethod = DEFAULT_CONNMETHOD;
   bw->fixed = DEFAULT_FIXED;
   bw->standard = DEFAULT_STANDARD;
   bw->centered = DEFAULT_CENTERED;
   bw->gid = NULL;
   bw->items = NULL;

   PROTECT;
   all_windows = CONS(bw,all_windows);
   UNPROTECT;

   ASHset_window_id(w,"build");
   ASHset_window_defaults(w);

   if (!setup_window(bw,tonly)) return FALSE;

   BUILD_menu_setup(bw,TRUE);

   if (proj != NULL && *proj != 0) {
      bp = BUILD_find_project(proj);
      if (bp != NULL) {
	 bf = BUILD_find_file(proj,BUILD_TYPE_UNKNOWN,&bp);
	 if (bp != NULL) {
	    BUILD_set_project(bw,bp);
	    if (bf != NULL) BUILD_set_file(bw,bf);
	  };
       };
    };

   if (bw->project == NULL) {
      bp = BUILD_find_project(MSGinq_wd());
      if (bp != NULL) BUILD_set_project(bw,bp);
    };

   while (bw->project == NULL) {
      BUILD_get_project(bw);
    };

   if (tonly) BUILD_set_display(bw,BUILD_DISP_TRANSCRIPT);

   return bw;
};





/************************************************************************/
/*									*/
/*	BUILD_find_window -- find build window given ash window 	*/
/*									*/
/************************************************************************/


BUILD_WIN
BUILD_find_window(w)
   ASH_WINDOW w;
{
   register Sequence l;
   register BUILD_WIN bw;

   forin (bw,BUILD_WIN,l,all_windows) {
      if (bw->window == w) break;
      else if (bw->menu_win == w) break;
      else if (bw->disp_win == w) break;
    };

   return bw;
};





/************************************************************************/
/*									*/
/*	BUILD_find_window_by_project -- find window given project	*/
/*									*/
/************************************************************************/


BUILD_WIN
BUILD_find_window_by_project(bp)
   BUILD_PROJ bp;
{
   Sequence l;
   BUILD_WIN bw;

   forin (bw,BUILD_WIN,l,all_windows) {
      if (bw->project == bp) break;
    };

   return bw;
};





/************************************************************************/
/*									*/
/*	BUILD_update_project -- update all instances of project 	*/
/*									*/
/************************************************************************/


void
BUILD_update_project(bp)
   BUILD_PROJ bp;
{
   Sequence l;
   BUILD_WIN bw;

   forin (bw,BUILD_WIN,l,all_windows) {
      if (bw->project == bp) {
	 BUILD_disp_update(bw,TRUE,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	BUILD_set_project -- set project for window			*/
/*	BUILD_set_file -- set file for window				*/
/*	BUILD_set_display -- set display for window			*/
/*									*/
/************************************************************************/


void
BUILD_set_project(bw,bp)
   BUILD_WIN bw;
   BUILD_PROJ bp;
{
   Character buf[1024];
   BUILD_FILE bf;

   if (bw->project == bp) return;

   bw->project = bp;

   if (bp != NULL) {
      if (bp->system_name != NULL) {
	 sprintf(buf,"Project:  %s in %s",bp->system_name,bp->system_dir);
       }
      else {
	 sprintf(buf,"Project:  %s",bp->system_dir);
       }
    }
   else sprintf(buf,"Project:  ");

   LEAFset_data(SALLOC(buf),bw->window,PROJECT_AREA,0);
   LEAFredraw(bw->window);

   sprintf(buf,"build: %s",bp->system_name);
   ASHset_window_name(bw->window,buf);

   BUILD_menu_setup(bw,FALSE);

   bw->file = NULL;
   bw->macro = NULL;
   if (bp->system_name != NULL) {
      bf = BUILD_find_file(bp->system_name,BUILD_TYPE_UNKNOWN,&bp);
    }
   else bf = NULL;

   bw->file = NULL;
   BUILD_set_file(bw,bf);
};





void
BUILD_set_file(bw,bf)
   BUILD_WIN bw;
   BUILD_FILE bf;
{
   Character buf[256];

   if (bf == bw->file && bf != NULL) return;

   BUILD_disp_update(bw,TRUE,FALSE);

   bw->file = bf;
   bw->macro = NULL;

   if (!bw->trans_only) {
      if (bf != NULL) sprintf(buf,"File:  %s",bf->name);
      else sprintf(buf,"File:  ");
      LEAFset_data(SALLOC(buf),bw->window,FILE_AREA,0);
      LEAFredraw(bw->window);
      BUILD_disp_update(bw,FALSE,TRUE);
    };
};




void
BUILD_set_macro(bw,bm)
   BUILD_WIN bw;
   BUILD_MACRO bm;
{
   Character buf[256];

   if (bm == bw->macro) return;

   BUILD_disp_update(bw,TRUE,FALSE);

   bw->file = NULL;
   bw->macro = bm;

   if (!bw->trans_only) {
      if (bm != NULL) sprintf(buf,"Macro:  %s",bm->name);
      else sprintf(buf,"File:  ");
      LEAFset_data(SALLOC(buf),bw->window,FILE_AREA,0);
      LEAFredraw(bw->window);
      BUILD_disp_update(bw,FALSE,TRUE);
    };
};




void
BUILD_set_display(bw,dt)
   BUILD_WIN bw;
   BUILD_DISP dt;
{
   Character buf[256];
   String s;

   if (bw->disp_type == dt) return;

   if (bw->disp_type != BUILD_DISP_NONE) {
      BUILD_disp_remove(bw);
    };

   bw->disp_type = dt;

   switch (dt) {
      case BUILD_DISP_NONE :
	 s = "NONE";
	 break;
      case BUILD_DISP_TRANSCRIPT :
	 s = "MAKE Transcript";
	 break;
      case BUILD_DISP_INFO :
	 s = "File/Macro Information";
	 break;
      case BUILD_DISP_GRAPH :
	 s = "Dependency Graph";
	 break;
      case BUILD_DISP_FILES :
	 s = "Project Files";
	 break;
    };

   if (!bw->trans_only) {
      sprintf(buf,"Display:  %s",s);
      LEAFset_data(SALLOC(buf),bw->window,STATUS_AREA,0);
      LEAFredraw(bw->window);
    };

   BUILD_disp_setup(bw);
};





/************************************************************************/
/*									*/
/*	BUILD_update -- make sure window is up to date			*/
/*									*/
/************************************************************************/


void
BUILD_update(bw)
   BUILD_WIN bw;
{
   if (BUILD_data_cleanup(bw->project)) {
      BUILD_menu_setup(bw,FALSE);
    };
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of a build window		*/
/*									*/
/************************************************************************/


static Boolean
setup_window(bw,tonly)
   BUILD_WIN bw;
   Boolean tonly;
{
   if (tonly) LEAFsetup_window(bw->window,transcript_leaf,bw);
   else LEAFsetup_window(bw->window,build_leaf,bw);

   LEAFset_control(bw->window,build_control);
   LEAFredraw(bw->window);

   if (tonly) bw->menu_win = NULL;
   else bw->menu_win = LEAFinq_window(bw->window,MENU_WIN,0);

   bw->disp_win = LEAFinq_window(bw->window,DISPLAY_WIN,0);

   if (bw->disp_win == NULL ||
	  (bw->menu_win == NULL && !tonly)) {
      fprintf(stderr,"BUILD: Window too small to run display\n");
      ASHremove(bw->window);
      return FALSE;
    };

   LEAFset_refresh(bw->window,check_windows);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   BUILD_WIN bw;
   ASH_WINDOW mw,dw;

   bw = BUILD_find_window(w);

   if (bw->menu_win == NULL) mw = NULL;
   else mw = LEAFinq_window(w,MENU_WIN,0);

   dw = LEAFinq_window(w,DISPLAY_WIN,0);

   if (mw != bw->menu_win || dw != bw->disp_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	build_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
build_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register BUILD_WIN bw;

   bw = BUILD_find_window(w);

   if (bw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      build_remove(bw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	build_remove -- remove given build window			*/
/*									*/
/************************************************************************/


static void
build_remove(bw)
   BUILD_WIN bw;
{
   PROTECT;

   all_windows = REMOB(bw,all_windows);

   UNPROTECT;

   bw->window = NULL;
};





/* end of buildwin.c */
