/************************************************************************/
/*									*/
/*		buildmenu.c						*/
/*									*/
/*	Menu handling routines for BUILD package			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "build_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_NAMES	20
#define MAKE_MENU	"Make"
#define CMD_MENU	"Command"



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	int		command_button();
static	int		make_btn();
static	int		compile_btn();
static	int		quit_btn();
static	int		set_project_btn();
static	int		project_opt_btn();
static	int		save_project_btn();
static	int		set_file_btn();
static	int		file_opt_btn();
static	int		set_macro_btn();
static	int		macro_opt_btn();
static	int		display_btn();
static	int		update_btn();
static	int		reinit_btn();
static	int		depend_btn();
static	BUILD_PROJ	define_project();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static STEM_PDM_DATA	build_menus[] = {
   { STEM_PSTATE_MENU,	MAKE_MENU,	NULL },
   { STEM_PSTATE_BTN,	"Make Current", make_btn },
   { STEM_PSTATE_BTN,	"Make Default", make_btn },
   { STEM_PSTATE_BTN,	"Make",         make_btn },
   { STEM_PSTATE_BTN,	"Compile",      compile_btn },
   { STEM_PSTATE_BTN,	"Quit",         quit_btn },

   { STEM_PSTATE_MENU,	CMD_MENU,	NULL },

   { STEM_PSTATE_MENU,	"Project",      NULL },
   { STEM_PSTATE_BTN,	"Set Project",  set_project_btn },
   { STEM_PSTATE_BTN,	"Options",      project_opt_btn },
   { STEM_PSTATE_BTN,	"Save",         save_project_btn },
   { STEM_PSTATE_BTN,	"Reinitialize", reinit_btn },
   { STEM_PSTATE_BTN,	"Dependencies", depend_btn },

   { STEM_PSTATE_MENU,	"File",         NULL },
   { STEM_PSTATE_BTN,	"Set File",     set_file_btn },
   { STEM_PSTATE_BTN,	"Set Macro",    set_macro_btn },
   { STEM_PSTATE_BTN,	"Set Default",  set_file_btn },
   { STEM_PSTATE_BTN,	"Properties",   file_opt_btn },

   { STEM_PSTATE_MENU,	"Windows",      NULL },
   { STEM_PSTATE_BTN,	"Update",       update_btn },
   { STEM_PSTATE_BTN,	"Transcript",   display_btn },
   { STEM_PSTATE_BTN,	"Information",  display_btn },
   { STEM_PSTATE_BTN,	"Graph",        display_btn },
   { STEM_PSTATE_BTN,	"None",         display_btn },

   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	BUILD_menu_init -- module initialization			*/
/*									*/
/************************************************************************/


void
BUILD_menu_init()
{
};





/************************************************************************/
/*									*/
/*	BUILD_menu_setup -- setup menu for window			*/
/*									*/
/************************************************************************/


void
BUILD_menu_setup(bw,newfg)
   BUILD_WIN bw;
   Boolean newfg;
{
   String names[MAX_NAMES];
   Integer i,ct;

   if (bw->menu_win == NULL) return;

   if (newfg) {
      STEMpdm_remove(bw->menu_win);
      STEMpdm_define(bw->menu_win,bw,build_menus);
    }
   else {
      STEMpdm_menu_remove_btns(bw->menu_win,CMD_MENU);
    };

   ct = BUILD_data_inq_info(bw->project,BUILD_TYPE_COMMAND,0,
			       MAX_NAMES,names);
   for (i = 0; i < ct; ++i) {
      STEMpdm_btn_add(bw->menu_win,CMD_MENU,names[i],command_button);
    };

   STEMpdm_btn_add(bw->menu_win,CMD_MENU,"Other ...",make_btn);
};





/************************************************************************/
/*									*/
/*	BUILD_button_project -- handle hits in PROJECT area		*/
/*	BUILD_button_file -- handle hits in FILE area			*/
/*	BUILD_button_display -- handle hits in DISPLAY area		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

int
BUILD_button_project(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   BUILD_WIN bw;

   w = RIPinq_window(rgn);
   bw = BUILD_find_window(w);

   if (bw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   set_project_btn(bw,"DUMMY","DUMMY");

   return TRUE;
};





/*ARGSUSED*/

int
BUILD_button_file(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   BUILD_WIN bw;

   w = RIPinq_window(rgn);
   bw = BUILD_find_window(w);

   if (bw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   if (bw->macro == NULL) {
      set_file_btn(bw,"DUMMY","DUMMY");
    }
   else {
      set_macro_btn(bw,"DUMMY","DUMMY");
    };

   return TRUE;
};





/*ARGSUSED*/

int
BUILD_button_display(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   BUILD_WIN bw;
   String menus[16];
   Integer ct,vl;

   w = RIPinq_window(rgn);
   bw = BUILD_find_window(w);

   if (bw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   ct = 0;
   menus[ct++] = "%CDisplay Selection\n";

   menus[ct++] = "%0.1o Transcript";
   menus[ct++] = "%0.2o Information";
   menus[ct++] = "%0.3o Graph";
/* menus[ct++] = "%0.4o Files";         */
   menus[ct++] = "%0.0o None";

   menus[ct++] = "\n   %a%M   %c";
   menus[ct] = 0;

   vl = (Integer) bw->disp_type;

   if (!STEMdialog(bw->window,menus,&vl)) return FALSE;

   BUILD_set_display(bw,(BUILD_DISP) vl);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	BUILD_get_project -- get project from user or system		*/
/*									*/
/************************************************************************/


void
BUILD_get_project(bw)
   BUILD_WIN bw;
{
   set_project_btn(bw,"DUMMY","INIT");
};





/************************************************************************/
/*									*/
/*	BUILD_select_file -- get file name from user			*/
/*									*/
/************************************************************************/


BUILD_FILE
BUILD_select_file(prmpt,bw,ty,nty)
   String prmpt;
   BUILD_WIN bw;
   BUILD_TYPE ty;
   BUILD_TYPE nty;
{
   Character hdr[128],fbuf[128];
   String menus[MAX_NAMES+10];
   Integer ct,fct,vl,mfg;
   String nms[MAX_NAMES*2];
   BUILD_PROJ bp;
   BUILD_FILE bf;
   BUILD_TYPE oty;
   STEM_DIALOG_LIST sdl;

   BUILD_disp_update(bw,TRUE,TRUE);

   ct = 0;
   sprintf(hdr,"%%CSelect File for %s\n",prmpt);
   menus[ct++] = hdr;
   menus[ct++] = "File name: %0.48t\n";
   fbuf[0] = 0;

   oty = ty & ~BUILD_TYPE_ASK_MACRO;
   fct = BUILD_data_inq_info(bw->project,oty,nty,MAX_NAMES*2-1,nms);
   sdl.btns = nms;
   nms[fct] = NULL;
   sdl.choice = -1;

   menus[ct++] = "       %1.40.10l\n";

   if (ty & BUILD_TYPE_ASK_MACRO) {
      menus[ct++] = "\n   %2o Select Macro";
    };
   mfg = 0;

   menus[ct++] = "\n   %a%M   %c";
   menus[ct] = NULL;

   if (!STEMdialog(bw->window,menus,fbuf,&sdl,&mfg)) return NULL;

   vl = sdl.choice;
   if (fbuf[0] == 0 && vl >= 0) strcpy(fbuf,nms[vl]);

   if (fbuf[0] != 0 && bw->project != NULL) {
      bf = BUILD_data_file(bw->project,fbuf,FALSE);
    }
   else if (mfg) {
      set_macro_btn(bw,"DUMMY","DUMMY");
      bf = NULL;
    }
   else {
      bp = bw->project;
      bf = BUILD_find_file(fbuf,ty,&bp);
      if (bw->project == NULL && bp != NULL) BUILD_set_project(bw,bp);
    };

   return bf;
};





/************************************************************************/
/*									*/
/*	command_button -- handle command request			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
command_button(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   ASHinput_lock_makelong(bw->window);

   BUILD_cmd_make(bw->project,btn,-1);

   return TRUE;
};




/************************************************************************/
/*									*/
/*	make_btn -- handle make request 				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
make_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   STEM_DIALOG_LIST sdl;
   Integer nct;
   String names[MAX_NAMES+2];
   String menus[64];
   Integer ct,i;
   Character buf[64];

   if (bw->project == NULL) return FALSE;

   if (STREQL(menu,CMD_MENU)) {
      nct = BUILD_data_inq_info(bw->project,BUILD_TYPE_COMMAND|BUILD_TYPE_SYSTEM,0,
				   MAX_NAMES,names);
    }
   else {
      nct = BUILD_data_inq_info(bw->project,BUILD_TYPE_SYSTEM,
				   BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH,
				   MAX_NAMES,names);
    };

   names[nct] = NULL;

   ct = 0;
   menus[ct++] = "%CSelect Item to Make\n";
   menus[ct++] = "Other: %1.40t\n";
   menus[ct++] = "       %0.40.10l\n";
   menus[ct++] = "   %a%M   %c";
   menus[ct] = 0;

   sdl.choice = 0;
   sdl.btns = names;
   buf[0] = 0;

   if (STREQL(btn,"Make Default")) {
      if (bw->project->system_name != NULL) strcpy(buf,bw->project->system_name);
    }
   else if (STREQL(btn,"Make Current") && bw->file != NULL) {
      strcpy(buf,bw->file->name);
    }
   else {
      if (bw->file != NULL && bw->file->name != NULL) {
	 for (i = 0; i < nct; ++i) {
	    if (STREQL(bw->file->name,names[i])) {
	       sdl.choice = i;
	       break;
	     };
	  };
       };

      if (!STEMdialog(bw->window,menus,&sdl,buf)) return FALSE;

      if (buf[0] == 0 && sdl.choice < 0) return FALSE;

      if (buf[0] == 0) strcpy(buf,names[sdl.choice]);
    };

   if (buf[0] != 0) {
      ASHinput_lock_makelong(bw->window);
      BUILD_cmd_make(bw->project,buf,-1);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	compile_btn -- handle compile request				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
compile_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_FILE bf;

   bf = BUILD_select_file("Compilation",bw,BUILD_TYPE_SOURCE,
			     BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH);
   if (bf == NULL) return FALSE;

   ASHinput_lock_makelong(bw->window);

   BUILD_cmd_compile(bw->project,bf->name,-1);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	quit_btn -- handle quit command 				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
quit_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   if (bw->window != NULL) ASHremove(bw->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	set_project_btn -- set project for window			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
set_project_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_PROJ bps[MAX_NAMES],bp;
   String menus[64];
   Character mbuf[MAX_NAMES][64];
   Character dbuf[64],nbuf[64];
   Integer ct,pct;
   Integer i,vl,quit;

   pct = BUILD_data_inq_projects(MAX_NAMES-1,bps);

   ct = 0;
   menus[ct++] = "%CCurrent Project\n";

   menus[ct++] = "%0.0o Current working directory\n";
   vl = 0;

   for (i = 0; i < pct; ++i) {
      if (bps[i]->system_name == NULL) continue;
      sprintf(mbuf[i],"%%0.%do %s",i+1,bps[i]->system_name);
      menus[ct++] = mbuf[i];
      if (bw->project == bps[i]) vl = i+1;
    };

   sprintf(mbuf[pct],"\n%%0.%do New Project:",pct+1);
   menus[ct++] = mbuf[pct];
   menus[ct++] = "    Directory: %1.48t";
   menus[ct++] = "    Name:      %2.48t";
   dbuf[0] = 0;
   nbuf[0] = 0;
   if (bw->project == NULL) vl = pct+1;

   if (bw->menu_win == NULL) {
      menus[ct++] = "\n%3o Quit";
    };

   menus[ct++] = "\n   %a%M   %c";
   menus[ct] = 0;

   quit = FALSE;

   if (!STEMdialog(bw->window,menus,&vl,dbuf,nbuf,&quit)) return FALSE;

   if (quit) {
      ASHremove(bw->window);
      return TRUE;
    };

   if (vl == 0) bp = define_project(MSGinq_wd(),NULL);
   else if (vl > pct) bp = define_project(dbuf,nbuf);
   else bp = bps[i-1];

   BUILD_set_project(bw,bp);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	project_opt_btn -- project option button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
project_opt_btn(bw,mnm,btn)
   BUILD_WIN bw;
   String mnm;
   String btn;
{
   String menu[32];
   Integer ct;
   Character nbuf[80],tbuf[80],sbuf[80],mbuf[80];
   Integer copt,sopt,topt,iopt,dopt;
   BUILD_PROJ bp;

   if (bw->project == NULL) return FALSE;

   bp = bw->project;

   ct = 0;
   menu[ct++] = "%CProject Options\n";

   menu[ct++] = "%4o Build as much as possible (-k)";
   menu[ct++] = "%5o Show commands; don't build (-n)";
   menu[ct++] = "%6o Touch out of date files; don't build";
   menu[ct++] = "%7o Ignore bad status on all commands";
   menu[ct++] = "%8o Use automatic dependency analysis";
   copt = bp->continue_opt;
   sopt = bp->show_opt;
   topt = bp->touch_opt;
   iopt = bp->ignore_opt;
   dopt = bp->autodepend;

   menu[ct++] = "\nDefault build: %0.64t\n";
   if (bp->system_name == NULL) nbuf[0] = 0;
   else strcpy(nbuf,bp->system_name);

   menu[ct++] = "Transcript:    %1.64t\n";
   if (bp->transcript == NULL) tbuf[0] = 0;
   else strcpy(tbuf,bp->transcript);

   menu[ct++] = "Source directory: %2.64t\n";
   if (bp->source_dir == NULL) sbuf[0] = 0;
   else strcpy(sbuf,bp->source_dir);

   menu[ct++] = "Project file:     %3.64t\n";
   if (bp->make_name == NULL) mbuf[0] = 0;
   else strcpy(mbuf,bp->make_name);

   menu[ct++] = "   %a%M   %c";
   menu[ct] = 0;

   if (!STEMdialog(bw->window,menu,nbuf,tbuf,sbuf,mbuf,
		      &copt,&sopt,&topt,&iopt,&dopt)) return FALSE;

   if (nbuf[0] != 0 && (bp->system_name == NULL || STRNEQ(nbuf,bp->system_name))) {
      if (bp->system_name != NULL) SFREE(bp->system_name);
      bp->system_name = SALLOC(nbuf);
    };

   if (tbuf[0] == 0) bp->transcript = NULL;
   else if (bp->transcript == NULL || STRNEQ(bp->transcript,tbuf)) {
      if (bp->trans_fd != NULL) fclose(bp->trans_fd);
      if (bp->transcript != NULL) SFREE(bp->transcript);
      bp->transcript = SALLOC(tbuf);
      bp->trans_fd = NULL;
    };

   if (sbuf[0] == 0) bp->source_dir = NULL;
   else if (bp->source_dir == NULL || STRNEQ(bp->source_dir,sbuf)) {
      if (bp->source_dir != NULL) SFREE(bp->source_dir);
      bp->source_dir = SALLOC(sbuf);
    };

   if (mbuf[0] != 0) {
      if (bp->make_name == NULL || STRNEQ(bp->make_name,mbuf)) {
	 if (bp->make_name != NULL) SFREE(bp->make_name);
	 bp->make_name = SALLOC(mbuf);
       };
    };

   bp->continue_opt = copt;
   bp->show_opt = sopt;
   bp->touch_opt = topt;
   bp->ignore_opt = iopt;
   bp->autodepend = dopt;

   bw->project = NULL;
   BUILD_set_project(bw,bp);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	save_project_btn -- save request				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
save_project_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_disp_update(bw,TRUE,TRUE);

   BUILD_file_gen_makefile(bw->project,TRUE);
};





/************************************************************************/
/*									*/
/*	set_file_btn -- handle set file request 			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
set_file_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_FILE bf;

   if (STREQL(btn,"Set Default")) {
      bf = BUILD_select_file("Display",bw,BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH,0);
    }
   else {
      bf = BUILD_select_file("Display",bw,
				BUILD_TYPE_UNKNOWN|BUILD_TYPE_ASK_MACRO,
				BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH);
      if (bf == NULL && bw->macro != NULL) return TRUE;
    };

   if (bf == NULL) return FALSE;

   BUILD_set_file(bw,bf);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	file_opt_btn -- handle file options				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
file_opt_btn(bw,mnm,btn)
   BUILD_WIN bw;
   String mnm;
   String btn;
{
   String menu[64];
   Integer ct,mct;
   Character hdr[128];
   Integer typ[2],flgs[6];
   Integer mac;
   Character mdef[MAX_NAMES][64];
   Sequence l;
   BUILD_MACRO bm;

   if (bw->project == NULL) return FALSE;

   if (bw->macro != NULL) return macro_opt_btn(bw,mnm,btn);
   if (bw->file == NULL) return FALSE;

   ct = 0;
   sprintf(hdr,"%%CFile Options for %s\n",bw->file->name);
   menu[ct++] = hdr;

   menu[ct++] = "File type:";
   menu[ct++] = "%0.1o Source        %0.2o Header        %0.4o Binary";
   menu[ct++] = "%0.8o System        %0.16o Command       %0.32o Default";
   menu[ct++] = "%0.64o Directory     %0.16777216o Dependency    %0.33554432o Library";
   menu[ct++] = "%0.128o Match Rule    %0.256o Basic File\n";
   menu[ct++] = "%1.256o C             %1.512o Pascal        %1.1024o YACC";
   menu[ct++] = "%1.2048o Lex";
   typ[0] = BUILD_TYPE_BASE(bw->file->type);
   typ[1] = BUILD_TYPE_LANG(bw->file->type);

   menu[ct++] = "\n%2o Use automatic dependency analysis";
   flgs[0] = bw->file->autodepend;
   menu[ct++] = "%3o Standard option (not editable)\n";
   flgs[1] = bw->file->library;
   menu[ct++] = "%4o Silect make";
   flgs[2] = bw->file->silent;
   menu[ct++] = "%5o Ignore bad status on make";
   flgs[3] = bw->file->ignore;
   menu[ct++] = "%6o Don't remove on bad make";
   flgs[4] = bw->file->precious;

   mac = bw->file->match_bits;
   mct = 0;
   forin (bm,BUILD_MACRO,l,bw->project->macros) {
      if (bm->option_name != NULL) {
	 sprintf(mdef[mct],"%%7.%df %s",bm->bit_value,bm->option_name);
	 menu[ct++] = mdef[mct];
	 ++mct;
       };
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   if (!STEMdialog(bw->window,menu,&typ[0],&typ[1],&flgs[0],&flgs[1],
		      &flgs[2],&flgs[3],&flgs[4],&mac))
      return FALSE;

   bw->file->type &= ~(BUILD_TYPE_BASE_BITS|BUILD_TYPE_LANG_BITS);
   bw->file->type |= typ[0]|typ[1];

   bw->file->autodepend = flgs[0];
   bw->file->library = flgs[1];
   bw->file->silent = flgs[2];
   bw->file->ignore = flgs[3];
   bw->file->precious = flgs[4];

   bw->file->match_bits = mac;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	set_macro_btn -- set macro to look at				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
set_macro_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   String menus[64];
   Character nbuf[64];
   Integer ct,mct;
   Integer i,vl,ffg;
   String s;
   BUILD_MACRO bm;
   String bms[MAX_NAMES*2];
   STEM_DIALOG_LIST sdl;

   if (bw->project == NULL) return FALSE;

   mct = BUILD_data_inq_macros(bw->project,MAX_NAMES*2-1,bms);
   bms[mct] = NULL;

   ct = 0;
   menus[ct++] = "%CSelect Macro\n";

   menus[ct++] = "       %0.40.10l\n";

   sdl.choice = -1;
   sdl.btns = bms;

   menus[ct++] = "Other: %1.48t";
   nbuf[0] = 0;

   menus[ct++] = "\n   %2o Select File";
   ffg = 0;

   menus[ct++] = "\n   %a%M   %c";
   menus[ct] = 0;

   if (!STEMdialog(bw->window,menus,&sdl,nbuf,&ffg)) return FALSE;

   vl = sdl.choice;
   if (vl < 0 && nbuf[0] == 0) {
      if (ffg) {
	 i = set_file_btn(bw,menu,btn);
	 return i;
       }
      else return FALSE;
    }
   else if (nbuf[0] != 0) {
      s = nbuf;
      bm = BUILD_find_macro(bw->project,s);
      if (bm == NULL) BUILD_data_macro_define(bw->project,nbuf,NULL,TRUE,FALSE);
    }
   else s = bms[vl];

   bm = BUILD_find_macro(bw->project,s);

   BUILD_set_macro(bw,bm);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	macro_opt_btn -- set options for macro				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
macro_opt_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   String menus[64];
   Character hdr[128],obuf[64];
   Integer ct;
   Integer typ[3];
   Integer lib;

   if (bw->macro == NULL) return FALSE;

   if (BUILD_set_flags(bw)) return TRUE;

   ct = 0;
   sprintf(hdr,"%%COptions for macro %s\n",bw->macro->name);
   menus[ct++] = hdr;

   menus[ct++] = "Default type: %2o";
   menus[ct++] = "%0.1o Source        %0.2o Header        %0.4o Binary";
   menus[ct++] = "%0.8o System        %0.16o Command       %0.32o Default";
   menus[ct++] = "%0.64o Directory";
   menus[ct++] = "%1.256o C             %1.512o Pascal        %1.1024o YACC";
   menus[ct++] = "%1.2048o Lex";
   typ[0] = BUILD_TYPE_BASE(bw->macro->dflt_type);
   typ[1] = BUILD_TYPE_LANG(bw->macro->dflt_type);
   typ[2] = (bw->macro->dflt_type != 0);

   menus[ct++] = "Option name: %3.32t\n";
   if (bw->macro->option_name == NULL) obuf[0] = 0;
   else strcpy(obuf,bw->macro->option_name);

   menus[ct++] = "%4o Library macro (not editable)\n";
   lib = bw->macro->library;

   menus[ct++] = "   %a%M   %c";
   menus[ct] = 0;

   if (!STEMdialog(bw->window,menus,&typ[0],&typ[1],&typ[2],obuf,&lib))
      return FALSE;

   if (typ[2]) bw->macro->dflt_type = typ[0]|typ[1];
   else bw->macro->dflt_type = 0;

   BUILD_data_macro_option(bw->project,bw->macro,obuf);
   BUILD_disp_update(bw,FALSE,TRUE);

   if (!lib) bw->macro->library = lib;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	display_btn -- handle display choice selection			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
display_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_DISP vl;

   if (STREQL(btn,"Transcript")) vl = BUILD_DISP_TRANSCRIPT;
   else if (STREQL(btn,"Information")) vl = BUILD_DISP_INFO;
   else if (STREQL(btn,"Graph")) vl = BUILD_DISP_GRAPH;
   else if (STREQL(btn,"Files")) vl = BUILD_DISP_FILES;
   else if (STREQL(btn,"None")) vl = BUILD_DISP_NONE;
   else return FALSE;

   BUILD_set_display(bw,vl);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	update_btn -- handle update request for display 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
update_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_disp_update(bw,TRUE,TRUE);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	reinit_btn -- handle user selected reinitialization		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
reinit_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_PROJ bp;

   ASHinput_lock_makelong(bw->window);

   bp = bw->project;

   BUILD_data_remove(bp);
   BUILD_file_setup(bp);

   bw->project = NULL;
   BUILD_set_project(bw,bp);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	depend_btn -- handle automatic dependency analysis		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
depend_btn(bw,menu,btn)
   BUILD_WIN bw;
   String menu;
   String btn;
{
   BUILD_PROJ bp;

   ASHinput_lock_makelong(bw->window);

   bp = bw->project;

   BUILD_file_depends(bp);

   BUILD_disp_update(bw,TRUE,TRUE);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	define_project -- define a new project				*/
/*									*/
/************************************************************************/


static BUILD_PROJ
define_project(dir,nam)
   String dir;
   String nam;
{
   Character buf[1024];
   BUILD_PROJ bp;
   String s;

   if (dir == NULL || dir[0] == 0) return NULL;
   if (nam == NULL || nam[0] == 0) {
      s = rindex(dir,'/');
      if (s == NULL) return NULL;
      nam = &s[1];
    };

   sprintf(buf,"%s/%s",dir,nam);

   bp = BUILD_find_project(buf);

   return bp;
};





/* end of buildmenu.c */

