/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		buildlex.l						*/
/*									*/
/*	Lexical definitions for make interface				*/
/*									*/
/************************************************************************/

#define LEXDEBUG
#define allprint(c) printf("0x%x",c)
#define sprint(s) printf("%s",s)

#undef YYLMAX
#define YYLMAX	10240


typedef struct _FILE_DATA *	FILE_DATA;
typedef struct _FILE_DATA {
   Integer line;
   String name;
   Boolean lib;
   FILE * file;
   FILE_DATA previous;
} FILE_DATA_INFO;

static	FILE_DATA	file_stack;
static	String		include_pfx;




static	void		start_read();
static	String		fix_macro_line();
static	void		make_include_file();



%}

%START START MACRO SHELL DEP PRAG INCL
%a 10240
%o 10240

WHITE	([ \t\r])+
WHITE0	([ \t\r])*

STRING	(\"([^\n"\\]|(\\.))*\")

ID	([a-zA-Z_@$?0-9./~%][-%a-zA-Z_$0-9@?./~()\[\]]*)
NULLID	"(null)"

INT	(\-?[0-9]+)

CHAR0	(\'.\')
CHAR1	(\'\\.\')
CHAR2	(\'\\([0-7]{1,3})\')
CHAR	({CHAR0}|{CHAR1}|{CHAR2})

CMMT	(^"#".*"\n")

CLINE	("\n\t"|";")
SLINE	(([^\n]|\\\n)*)
MLINE	("\n\t")

EOL	([\n])
SPACE	([ \t]|(\\\n))



%%


<START>{ID}		{ yylval.string = SALLOC(yytext);
			  return LX_ID;
			}
<START>"::"             { BEGIN DEP;
			  return LX_COLCOL;
			}
<START>":"              { BEGIN DEP;
			  return ':';
			}
<START>"="{SPACE}*      { BEGIN MACRO;
			  return '=';
			}
<START>^"#AUTODEPEND\n" { return LX_AUTODEPEND; }
<START>{EOL}		;


<DEP>{ID}		{ yylval.string = SALLOC(yytext);
			  return LX_ID;
			}
<DEP>{CLINE}		{ BEGIN SHELL;
			  return ';';
			}

<MACRO>{SLINE}		{ yylval.string = fix_macro_line(yytext);
			  return LX_TEXT;
			}

<SHELL>{SLINE}		{ yylval.string = SALLOC(yytext);
			  return LX_TEXT;
			}
<SHELL>{MLINE}		{ return LX_SEOL; }


<PRAG>{INT}		{ yylval.string = SALLOC(yytext);
			  return LX_INT;
			}
<PRAG>{ID}		{ yylval.string = SALLOC(yytext);
			  return LX_ID;
			}
<PRAG>{NULLID}		{ yylval.string = SALLOC(yytext);
			  return LX_ID;
			}
<PRAG>{STRING}		{ yytext[yyleng-1] = 0;
			  yylval.string = SALLOC(&yytext[1]);
			  return LX_STRING;
			}

<INCL>.*\n		{ make_include_file(yytext);
			  BEGIN START;
			}

^"#FILE "/.*\n          { BEGIN PRAG;
			  return LX_FILEINFO;
			}
^"#MACRO "/.*\n         { BEGIN PRAG;
			  return LX_MACROINFO;
			}
^"#PROJECT "/.*\n       { BEGIN PRAG;
			  return LX_PROJECTINFO;
			}
^"#".*"\n"              ;
{WHITE} 		;
"\\\n"                  ;

{EOL}			{ BEGIN START; return LX_EOL; }

.			{ return yytext[0]; }

^"include"[ \t]         { BEGIN INCL; }

%%



static void
start_read()
{
   BEGIN START;
   include_pfx = (String) getenv("INCLUDE_PATH");
};





static String
fix_macro_line(s)
   String s;
{
   Integer ln;
   String p,q;

   if (index(s,'\\') == NULL) return SALLOC(s);

   ln = strlen(s);
   q = (String) alloca(ln+10);

   p = q;
   while (*s != 0) {
      if (*s == '\\' && s[1] == '\n') ++s;
      *p++ = *s++;
    };
   *p = 0;

   return SALLOC(q);
};





static void
make_include_file(s)
   String s;
{
   String t,p;
   Character buf[1024];
   FILE * newf;
   FILE_DATA fd;

   while (isspace(*s)) ++s;

   t = s;
   while (!isspace(*t)) ++t;
   *t = 0;

   if (access(s,4) < 0 && s[0] != '/' && include_pfx != NULL) {
      for (t = include_pfx; t != NULL; t = index(t,':')) {
	 while (*t == ':') ++t;
	 strcpy(buf,t);
	 p = index(buf,':');
	 *p = 0;
	 strcat(buf,s);
	 if (access(s,4) >= 0) {
	    s = buf;
	    break;
	  };
       };
    };

   newf = fopen(s,"r");
   if (newf == NULL) {
      fprintf(stderr,"Can't open include file %s\n",s);
      return;
    };

   fd = PALLOC(FILE_DATA_INFO);
   fd->name = cur_file;
   fd->line = yylineno;
   fd->file = yyin;
   fd->previous = file_stack;
   fd->lib = cur_lib;
   file_stack = fd;

   cur_file = SALLOC(s);
   cur_lib = TRUE;
   yylineno = 1;
   yyin = newf;
};





yywrap()
{
   FILE_DATA fd;

   if (file_stack == NULL) return 1;

   fclose(yyin);

   fd = file_stack;
   file_stack = fd->previous;
   yyin = fd->file;
   yylineno = fd->line;
   if (cur_file != NULL) SFREE(cur_file);
   cur_file = fd->name;
   cur_lib = fd->lib;

   free(fd);

   return 0;
};





/* end of buildlex.l */
