/************************************************************************/
/*									*/
/*		builddisp.c						*/
/*									*/
/*	Display routines for Make interface				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "build_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_BTN 	512





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		update_info_display();
static	void		edit_rules();
static	Integer 	edit_rule_control();
static	void		edit_macro();
static	Integer 	edit_macro_control();
static	void		show_depend();
static	void		show_macro_files();
static	Integer 	button_handler();
static	Integer 	header_btn();
static	void		update_transcript();
static	Integer 	transcript_control();
static	void		transcript_clear();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define HEADER_AREA	1
#define MENU_WIN	2
#define EDIT_WIN	3



static	LEAF_DATA	build_disp_leaf[] = {
   LEAF_ROOT(NULL),
   { HEADER_AREA, LEAF_TYPE_TEXT,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	header_btn, NULL },
   { MENU_WIN, LEAF_TYPE_STEM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(HEADER_AREA) },
	button_handler, NULL },
   { EDIT_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





static	LEAF_DATA	build_disp_single[] = {
   LEAF_ROOT(NULL),
   { EDIT_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	BUILD_disp_init -- module initialization			*/
/*									*/
/************************************************************************/


void
BUILD_disp_init()
{
};





/************************************************************************/
/*									*/
/*	BUILD_disp_setup -- setup display for window			*/
/*									*/
/************************************************************************/


void
BUILD_disp_setup(bw)
   BUILD_WIN bw;
{
   BUILD_update(bw);

   if (bw->disp_win == NULL) return;

   ASHvisible(bw->disp_win,FALSE);

   switch (bw->disp_type) {
      case BUILD_DISP_INFO :
	 LEAFsetup_window(bw->disp_win,build_disp_leaf,bw);
	 break;
      default :
	 LEAFsetup_window(bw->disp_win,build_disp_single,bw);
	 break;
    };

   LEAFredraw(bw->disp_win);
   bw->edit_win = LEAFinq_window(bw->disp_win,EDIT_WIN,0);
   bw->edit_info = NULL;

   switch (bw->disp_type) {
      default :
      case BUILD_DISP_NONE :
	 break;
      case BUILD_DISP_TRANSCRIPT :
	 update_transcript(bw);
	 break;
      case BUILD_DISP_INFO :
	 STEMmenu_refresh(LEAFinq_window(bw->disp_win,MENU_WIN,0),NULL);
	 update_info_display(bw,FALSE);
	 break;
      case BUILD_DISP_GRAPH :
	 BUILD_graph_setup(bw);
	 break;
      case BUILD_DISP_FILES :
	 break;
    };

   ASHvisible(bw->disp_win,TRUE);
};





/************************************************************************/
/*									*/
/*	BUILD_disp_update -- update display (no change in type) 	*/
/*									*/
/************************************************************************/


void
BUILD_disp_update(bw,savefg,keepfg)
   BUILD_WIN bw;
   Boolean savefg;
   Boolean keepfg;
{
   BUILD_update(bw);

   if (bw->disp_win == NULL) return;

   switch (bw->disp_type) {
      default :
      case BUILD_DISP_NONE :
	 break;
      case BUILD_DISP_TRANSCRIPT :
	 if (keepfg) update_transcript(bw);
	 break;
      case BUILD_DISP_INFO :
	 update_info_display(bw,savefg);
	 break;
      case BUILD_DISP_GRAPH :
	 if (keepfg) BUILD_graph_update(bw);
	 break;
      case BUILD_DISP_FILES :
	 break;
    };
};





/************************************************************************/
/*									*/
/*	BUILD_disp_remove -- remove current display			*/
/*									*/
/************************************************************************/


void
BUILD_disp_remove(bw)
   BUILD_WIN bw;
{
   Integer lx,by,rx,ty;

   if (bw->disp_win == NULL) return;

   switch (bw->disp_type) {
      default :
      case BUILD_DISP_NONE :
	 break;
      case BUILD_DISP_TRANSCRIPT :
	 transcript_clear(bw);
	 break;
      case BUILD_DISP_INFO :
	 if (bw->edit_info != NULL) {
	    EDTclose_save(bw->edit_info);
	    bw->edit_info = NULL;
	  };
	 break;
      case BUILD_DISP_GRAPH :
	 BUILD_graph_remove(bw);
	 break;
      case BUILD_DISP_FILES :
	 break;
    };

   LEAFremove(bw->disp_win);
   ASHinq_size(bw->disp_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclear_box(bw->disp_win,lx,by,rx,ty);
   bw->edit_info = NULL;
   bw->edit_win = NULL;

   bw->disp_type = BUILD_DISP_NONE;
};





/************************************************************************/
/*									*/
/*	BUILD_disp_transcript -- handle transcript output		*/
/*									*/
/************************************************************************/


void
BUILD_disp_transcript(bw,lin)
   BUILD_WIN bw;
   String lin;
{
   if (bw->disp_win == NULL ||
	  bw->disp_type != BUILD_DISP_TRANSCRIPT ||
	  bw->edit_info == NULL)
      return;

   EDTshell_output(bw->edit_info,lin);
};





/************************************************************************/
/*									*/
/*	update_info_display -- update information display		*/
/*									*/
/************************************************************************/


static void
update_info_display(bw,savefg)
   BUILD_WIN bw;
   Boolean savefg;
{
   if (bw->file != NULL) {
      LEAFset_data("Dependencies",bw->disp_win,HEADER_AREA,0);
    }
   else if (bw->macro != NULL) {
      LEAFset_data("Files Included",bw->disp_win,HEADER_AREA,0);
    };

   LEAFredraw(bw->disp_win);

   if (bw->edit_info != NULL) {
      if (savefg) EDTclose_save(bw->edit_info);
      else EDTclose_abort(bw->edit_info);
      bw->edit_info = NULL;
    };
   STEMmenu_refresh(LEAFinq_window(bw->disp_win,MENU_WIN,0),NULL);

   if (bw->file != NULL) {
      show_depend(bw);
      edit_rules(bw);
    }
   else if (bw->macro != NULL) {
      show_macro_files(bw);
      edit_macro(bw);
    };
};





/************************************************************************/
/*									*/
/*	edit_rules -- setup an editor on the rules of current file	*/
/*	edit_rule_control -- handle editor control messages		*/
/*									*/
/************************************************************************/


static void
edit_rules(bw)
   BUILD_WIN bw;
{
   ASH_WINDOW ew;
   String s;

   ew = bw->edit_win;

   ASHset_control(ew,edit_rule_control);

   s = bw->file->build;
   if (s == NULL && bw->file->dflt != NULL) s = bw->file->dflt->build;

   bw->edit_info = EDTedit_string(ew,s);
   EDTprohibit_close(bw->edit_info);
};





static Integer
edit_rule_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   BUILD_WIN bw;
   String s;
   BUILD_FILE bf;

   bw = BUILD_find_window(ASHinq_parent(w));
   if (bw == NULL) return ASH_CONTROL_REJECT;

   bf = bw->file;
   if (bf == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$INQ_SAVE_CLOSE") || STREQL(msg,"EDIT$SAVE")) {
      s = EDTinq_contents(bw->edit_info);
      if (s != NULL) {
	 BUILD_data_set_rules(bw->project,bf,s);
	 SFREE(s);
	 if (STREQL(msg,"EDIT$SAVE")) BUILD_update(bw);
       };
    }
   else if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove_control(w,edit_rule_control);
      bw->edit_info = NULL;
    };

   return ASH_CONTROL_REJECT;
};






/************************************************************************/
/*									*/
/*	edit_macro -- setup an editor on the contents of a macro	*/
/*	edit_macro_control -- handle editor control messages		*/
/*									*/
/************************************************************************/


static void
edit_macro(bw)
   BUILD_WIN bw;
{
   ASH_WINDOW ew;
   String s;

   ew = bw->edit_win;

   ASHset_control(ew,edit_macro_control);

   s = bw->macro->body;

   bw->edit_info = EDTedit_string(ew,s);
   EDTprohibit_close(bw->edit_info);
};





static Integer
edit_macro_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   BUILD_WIN bw;
   String s;
   BUILD_MACRO bm;

   bw = BUILD_find_window(ASHinq_parent(w));
   if (bw == NULL) return ASH_CONTROL_REJECT;

   bm = bw->macro;
   if (bm == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$INQ_SAVE_CLOSE") || STREQL(msg,"EDIT$SAVE")) {
      s = EDTinq_contents(bw->edit_info);
      if (s != NULL) {
	 BUILD_data_macro_body(bw->project,bm,s);
	 SFREE(s);
	 if (STREQL(msg,"EDIT$SAVE")) BUILD_update(bw);
       };
    }
   else if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove_control(w,edit_macro_control);
      bw->edit_info = NULL;
    };

   return ASH_CONTROL_REJECT;
};






/************************************************************************/
/*									*/
/*	show_depend -- show dependencies for current file		*/
/*									*/
/************************************************************************/


static void
show_depend(bw)
   BUILD_WIN bw;
{
   STEM_BUTTON btns[MAX_BTN];
   Integer bct;
   Sequence l;
   BUILD_DEPEND bd;
   ASH_WINDOW sw;

   sw = LEAFinq_window(bw->disp_win,MENU_WIN,0);

   bct = 0;
   forin (bd,BUILD_DEPEND,l,bw->file->depends) {
      btns[bct].name = bd->file->name;
      if (bd->automatic || bd->macro || bd->implicit)
	 btns[bct].flags = STEM_FLAG_LOLITE;
      else
	 btns[bct].flags = 0;
      ++bct;
    };
   btns[bct].name = 0;

   STEMmenu_refresh(sw,btns);
};





/************************************************************************/
/*									*/
/*	show_macro_files -- show files used by macro			*/
/*									*/
/************************************************************************/


static void
show_macro_files(bw)
   BUILD_WIN bw;
{
   STEM_BUTTON btns[MAX_BTN];
   Integer bct;
   BUILD_FILE bf;
   Character buf[1024];
   BUILD_PROJ bp;
   Integer i;
   String s;
   ASH_WINDOW sw;

   sw = LEAFinq_window(bw->disp_win,MENU_WIN,0);
   bp = bw->project;

   bct = 0;
   s = bw->macro->body;
   if (s == NULL) s = "";

   i = 0;
   while (*s != 0) {
      if (isspace(*s)) {
	 if (i != 0) {
	    buf[i] = 0;
	    bf = BUILD_find_file(buf,BUILD_TYPE_UNKNOWN,&bp);
	    if (bf != NULL) {
	       btns[bct].name = bf->name;
	       btns[bct].flags = 0;
	       ++bct;
	     };
	  }
	 i = 0;
       }
      else buf[i++] = *s;
      ++s;
    };

   if (i != 0) {
      buf[i] = 0;
      bf = BUILD_find_file(buf,BUILD_TYPE_UNKNOWN,&bp);
      if (bf != NULL) {
	 btns[bct].name = bf->name;
	 btns[bct].flags = 0;
	 ++bct;
       };
    };

   btns[bct].name = 0;

   STEMmenu_refresh(sw,btns);
};





/************************************************************************/
/*									*/
/*	button_handler -- handle stem button picks			*/
/*	depend_btn -- handle button for dependencies			*/
/*									*/
/************************************************************************/


/* ARGSUSED */

static Integer
button_handler(bw,btn,mask,w)
   BUILD_WIN bw;
   Integer btn;
   Integer mask;
   ASH_WINDOW w;
{
   BUILD_DEPEND bd;
   Boolean fg;

   fg = TRUE;

   if (mask & RIP_BTN_NONE) return FALSE;

   if (bw->file != NULL) {
      bd = NTH(BUILD_DEPEND,bw->file->depends,btn);
      if (bd == NULL) fg = FALSE;
      else if (bd->automatic || bd->macro || bd->implicit) fg = FALSE;
      else bw->file->depends = REMOB(bd,bw->file->depends);
      show_depend(bw);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	header_btn -- handle hits on header button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
header_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   BUILD_WIN bw;
   Boolean fg;
   BUILD_FILE bf;

   bw = BUILD_find_window(RIPinq_window(rgn));

   if (bw == NULL) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   fg = TRUE;

   if (bw->file != NULL) {
      bf = BUILD_select_file("Dependency",bw,0,BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH);
      if (bf == NULL) fg = FALSE;
      else {
	 BUILD_data_depend(bw->project,bw->file,bf,FALSE);
	 show_depend(bw);
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	update_transcript -- setup transcript display			*/
/*	transcript_control -- handle control messages on transcript	*/
/*									*/
/************************************************************************/


static void
update_transcript(bw)
   BUILD_WIN bw;
{
   String s;

   transcript_clear(bw);

   if (bw->project == NULL) s = NULL;
   else s =  bw->project->trans_data;

   ASHset_control(bw->edit_win,transcript_control);

   bw->edit_info = EDTedit_string(bw->edit_win,s);
   EDTprohibit_close(bw->edit_info);
};





static Integer
transcript_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   BUILD_WIN bw;

   bw = BUILD_find_window(ASHinq_parent(w));
   if (bw == NULL) return ASH_CONTROL_REJECT;

   if (bw->edit_info == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove_control(w,transcript_control);
      bw->edit_info = NULL;
    }

   return ASH_CONTROL_REJECT;
};





static void
transcript_clear(bw)
   BUILD_WIN bw;
{
   if (bw->edit_info != NULL) {
      EDTclose_abort(bw->edit_info);
      bw->edit_info = NULL;
    };
};




/* end of builddisp.c */
