/************************************************************************/
/*									*/
/*		annotwin.c						*/
/*									*/
/*	Window manipulation routines for annotation windows		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "annot_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define ANNOTATION_WIDTH	50
#define STATE_WIDTH		85




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	int		annot_control();
static	void		check_windows();
static	int		annot_edit_control();
static	void		annot_remove();
static	ANNOT_WDESC	request_window_type();
static	int		set_system();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define ANNO_WIN	2
#define MESG_WIN	3
#define EDIT_WIN	4
#define STAT_WIN	5



static	LEAF_DATA	annot_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { ANNO_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(MESG_WIN),
	     LEAF_COORD_SIZE_LINE(LEAF_VALUE_PARAM(0)), LEAF_COORD_NEXT(MENU_WIN) },
	ANNOT_annot_btn, NULL },
   { MESG_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY, LEAF_COORD_RX, LEAF_COORD_TEXT_LINE },
	NULL, NULL },
   { EDIT_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_NEXT(ANNO_WIN), LEAF_COORD_NEXT(MESG_WIN),
	     LEAF_COORD_NEXT(STAT_WIN), LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   { STAT_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_SIZE_LINE(LEAF_VALUE_PARAM(1)), LEAF_COORD_NEXT(MESG_WIN),
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





static STEM_PDM_DATA	annot_menus[] = {
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX,  "Annotate",     NULL },
   { STEM_PSTATE_BTN,	"Set System",   set_system },
   { STEM_PSTATE_BTN,	"Search Again", ANNOT_act_search_again },
   { STEM_PSTATE_BTN,	"Search",       ANNOT_act_search },
   { STEM_PSTATE_BTN,	"Remove All",   ANNOT_act_remove },
   { STEM_PSTATE_BTN,	"Rotate",       ANNOT_act_rotate },
   { STEM_PSTATE_BTN,	"Update",       ANNOT_act_update },
   { STEM_PSTATE_BTN,	"Display",      ANNOT_act_display },
   { STEM_PSTATE_BTN,	"Monitor",      ANNOT_act_monitor },
   { STEM_PSTATE_BTN,	"Quit",         ANNOT_edit_quit },

   { STEM_PSTATE_END }
};

static STEM_PDM_DATA	annot_menus_search[] = {
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX,  "Annotate",     NULL },
   { STEM_PSTATE_BTN,	"Set System",   set_system },
   { STEM_PSTATE_BTN,	"Remove All",   ANNOT_act_remove },
   { STEM_PSTATE_BTN,	"Rotate",       ANNOT_act_rotate },
   { STEM_PSTATE_BTN,	"Update",       ANNOT_act_update },
   { STEM_PSTATE_BTN,	"Display",      ANNOT_act_display },
   { STEM_PSTATE_BTN,	"Monitor",      ANNOT_act_monitor },
   { STEM_PSTATE_BTN,	"Quit",         ANNOT_edit_quit },

   { STEM_PSTATE_MENU,	"Search",       NULL },
   { STEM_PSTATE_BTN,	"Search",       ANNOT_act_search },
   { STEM_PSTATE_BTN,	"Search Again", ANNOT_act_search_again },

   { STEM_PSTATE_END }
};

static STEM_PDM_DATA	annot_menus_none[] = {
   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	ANNOT_win_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
ANNOT_win_init()
{
   all_windows = NULL;
};





/************************************************************************/
/*									*/
/*	ANNOT_create -- create a new annotation window			*/
/*									*/
/************************************************************************/


ANNOT_WIN
ANNOT_create(w,type,file)
   ASH_WINDOW w;
   String type;
   String file;
{
   ANNOT_WIN aw;
   ANNOT_WDESC awd;
   String menu[32];
   Character buf[64],sbuf[64];

   if (type == NULL) awd = NULL;
   else awd = ANNOT_find_window_desc(type);

   if (awd == NULL) awd = request_window_type(w);
   if (awd == NULL) return NULL;

   if (file != NULL && *file == NULL) file = NULL;

   aw = PALLOC(ANNOT_WIN_INFO);
   aw->window = w;
   aw->filename = NULL;
   aw->wd = NULL;
   aw->system = NULL;
   aw->desc = awd;
   aw->level = aw->desc->level;
   aw->edit_info = NULL;
   aw->willow_fg = FALSE;
   aw->top_line[0] = -1;
   aw->remove_fg = FALSE;
   aw->state_setup = FALSE;
   aw->search_type = NULL;
   aw->search_line = 0;
   aw->search_dir = 1;
   aw->allowed = LCOPY(aw->desc->allowed);
   aw->set_on = LCOPY(aw->desc->set_on);
   aw->new_annot = NULL;
   aw->cur_annot = NULL;
   aw->alt_annot = NULL;
   aw->exec_btn = NULL;
   aw->changing_file = FALSE;
   aw->linevalid = FALSE;
   aw->mod_time = 0;

   PROTECT;
   all_windows = CONS(aw,all_windows);
   UNPROTECT;

   ASHset_window_id(w,aw->desc->name);
   ASHset_window_defaults(w);

   ANNOT_msg_setup(aw);

   while (aw->filename == NULL) {
      if (file == NULL || *file == NULL) {
	 if (!aw->desc->autofile) {
	    menu[0] = "%CAnnotation Window\n\n";
	    menu[1] = "Name:   %0.32t\n";
	    menu[2] = "System: %1.32t\n";
	    menu[3] = "   %a%M   %c";
	    menu[4] = 0;
	    if (file == NULL) buf[0] = 0;
	    else strcpy(buf,file);
	    if (aw->system == NULL) sbuf[0] = 0;
	    else strcpy(sbuf,aw->system);
	    if (!STEMdialog(w,menu,buf,sbuf) || buf[0] == 0) {
	       if (aw->filename == NULL) return NULL;
	       strcpy(buf,aw->filename);
	     }
	    else {
	       file = buf;
	       if (aw->system != NULL) SFREE(aw->system);
	       if (sbuf[0] == 0 || STREQL(sbuf,"*")) aw->system = NULL;
	       else aw->system = SALLOC(sbuf);
	     };
	  }
       };

      if (!ANNOT_setup_windows(aw,file)) return NULL;

      file = NULL;

      if (aw->set_on != NULL) break;
    };

   return aw;
};





/************************************************************************/
/*									*/
/*	ANNOT_setup_windows -- setup once we have a file		*/
/*									*/
/************************************************************************/


Boolean
ANNOT_setup_windows(aw,file)
   ANNOT_WIN aw;
   String file;
{
   ANNOT_BDESC abd;
   Sequence l;
   Character buf[128];

   if (file == NULL) return TRUE;

   if (!aw->desc->create && access(file,4) < 0) return TRUE;

   aw->filename = SALLOC(file);
   aw->wd = MSGinq_wd();
   aw->wd = SALLOC(aw->wd);

   if (!setup_window(aw)) return FALSE;

   ANNOT_set_window_name(aw);

   if (aw->menu_win != NULL) {
      if (aw->annot_win != NULL && aw->desc->search_for == NULL)
	 STEMpdm_define(aw->menu_win,aw,annot_menus);
      else if (aw->annot_win != NULL)
	 STEMpdm_define(aw->menu_win,aw,annot_menus_search);
      else
	 STEMpdm_define(aw->menu_win,aw,annot_menus_none);
      if (aw->state_pdm == STATE_PDM) {
	 STEMpdm_menu_add(aw->menu_win,SELECTION_MENU);
       };
      if (aw->desc->buttons != NULL) {
	 STEMpdm_menu_add(aw->menu_win,"Commands");
	 forin (abd,ANNOT_BDESC,l,aw->desc->buttons) {
	    STEMpdm_btn_add(aw->menu_win,"Commands",abd->name,ANNOT_msg_menu);
	  };
       };
      if (aw->annot_win != NULL && aw->desc->search_for != NULL) {
	 forin (abd,ANNOT_BDESC,l,aw->desc->search_for) {
	    sprintf(buf,"Search for %s",abd->name);
	    STEMpdm_btn_add(aw->menu_win,"Search",buf,ANNOT_act_search_for);
	  };
       };
    };

   ANNOT_edit_setup(aw);
   ANNOT_annot_setup(aw);

   if (aw->desc->nowindow) return FALSE;

   if (aw->state_win != NULL) ASHvisible(aw->state_win,TRUE);
   if (aw->edit_win != NULL) ASHvisible(aw->edit_win,TRUE);
   if (aw->annot_win != NULL) ASHvisible(aw->annot_win,TRUE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_find_window -- find annotation window given ash window	*/
/*									*/
/************************************************************************/


ANNOT_WIN
ANNOT_find_window(w)
   ASH_WINDOW w;
{
   Sequence l;
   ANNOT_WIN aw;

   forin (aw,ANNOT_WIN,l,all_windows) {
      if (aw->window == w) break;
      else if (aw->edit_win == w) break;
      else if (aw->annot_win == w) break;
    };

   return aw;
};





/************************************************************************/
/*									*/
/*	ANNOT_set_window_name -- set name for window			*/
/*									*/
/************************************************************************/


void
ANNOT_set_window_name(aw)
   ANNOT_WIN aw;
{
   Character buf[256];
   String file;

   file = (aw->filename == NULL ? "<NONE>" : aw->filename);

   if (aw->desc->title == NULL) {
      sprintf(buf,"%s: %s",aw->desc->name,file);
    }
   else {
      sprintf(buf,aw->desc->title,file);
    };

   ASHset_window_name(aw->window,buf);
};





/********************************************************************************/
/*										*/
/*	ANNOT_for_all_windows -- call a routine for all windows 		*/
/*										*/
/********************************************************************************/


void
ANNOT_for_all_windows(rtn,a1,a2)
   Function_Ptr rtn;
   Universal a1,a2;
{
   ANNOT_WIN aw;
   Sequence l;

   forin (aw,ANNOT_WIN,l,all_windows) {
      (*rtn)(aw,a1,a2);
    };
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of an annotation window	*/
/*									*/
/************************************************************************/


static Boolean
setup_window(aw)
   ANNOT_WIN aw;
{
   Integer annoct,showct;
   Sequence l;
   ANNOT_DESC ad;

   annoct = 0;
   showct = 0;
   forin (ad,ANNOT_DESC,l,aw->allowed) {
      if (ad->display_level > aw->level) continue;
      if (!ad->hilite) ++showct;
      if (!ad->private) {
	 if (annoct == 0) aw->cur_annot = aw->alt_annot = ad;
	 else if (annoct == 1) aw->alt_annot = ad;
	 ++annoct;
       };
    };

   LEAFsetup_window(aw->window,annot_leaf,aw);

   if (showct == 0) LEAFset_parameters(aw->window,2,0,0);
   else if (annoct <= 1 || aw->desc->choicepdm || aw->desc->nopdm)
      LEAFset_parameters(aw->window,2,ANNOTATION_WIDTH,0);
   else LEAFset_parameters(aw->window,2,ANNOTATION_WIDTH,STATE_WIDTH);

   LEAFset_control(aw->window,annot_control);
   LEAFredraw(aw->window);

   if (annoct <= 1 || aw->desc->nopdm) aw->state_pdm = STATE_NONE;
   else if (aw->desc->choicepdm) aw->state_pdm = STATE_PDM;
   else aw->state_pdm = STATE_MENU;

   aw->edit_win = LEAFinq_window(aw->window,EDIT_WIN,0);
   if (aw->edit_win == NULL) {
      fprintf(stderr,"ANNOT: Window too small to run annotator\n");
      ASHremove(aw->window);
      return FALSE;
    };

   ASHset_control(aw->edit_win,annot_edit_control);
   LEAFset_refresh(aw->window,check_windows);

   aw->menu_win = LEAFinq_window(aw->window,MENU_WIN,0);
   aw->annot_win = LEAFinq_window(aw->window,ANNO_WIN,0);
   aw->state_win = LEAFinq_window(aw->window,STAT_WIN,0);
   aw->mesg_win = LEAFinq_window(aw->window,MESG_WIN,0);

   HELPmouse_register(aw->annot_win,"Add/Delete Annotation","Information about Annotation",
			 "Add/Delete Annotation");
   if (aw->state_win != NULL)
      HELPmouse_register(aw->state_win,"Select Annotation Type","Select Annotation Type",
			    "Select Annotation Type");

   return TRUE;
};





/************************************************************************/
/*									*/
/*	annot_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
annot_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   Sequence l;
   ANNOT_WIN aw;

   forin (aw,ANNOT_WIN,l,all_windows) {
      if (aw->window == w) break;
    };

   if (aw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      annot_remove(aw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   ANNOT_WIN aw;
   ASH_WINDOW ew;
   Sequence l;

   forin (aw,ANNOT_WIN,l,all_windows) {
      if (aw->window == w) break;
    };

   if (aw == NULL) return;

   ew = LEAFinq_window(w,EDIT_WIN,0);

   if (ew != aw->edit_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	annot_edit_control -- handle ASH messages for editor		*/
/*									*/
/************************************************************************/


static int
annot_edit_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   Sequence l;
   ANNOT_WIN aw;

   forin (aw,ANNOT_WIN,l,all_windows) {
      if (aw->edit_win == w) break;
    };

   if (aw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"EDIT$REMOVE")) {
      ASHremove(aw->window);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	annot_remove -- remove given annotation 			*/
/*									*/
/************************************************************************/


static void
annot_remove(aw)
   ANNOT_WIN aw;
{
   PROTECT;

   all_windows = REMOB(aw,all_windows);

   UNPROTECT;

   if (aw->filename != NULL) {
/*    free(aw); 	Dont free -- message problems */
      aw->filename = NULL;
      aw->wd = NULL;
      aw->window = NULL;
      aw->edit_win = NULL;
      aw->edit_info = NULL;
      aw->annot_win = NULL;
      aw->menu_win = NULL;
      aw->mesg_win = NULL;
      aw->state_win = NULL;
      ANNOT_clear(aw);
    }
   else aw->remove_fg = TRUE;
};





/************************************************************************/
/*									*/
/*	request_window_type -- get window type from user		*/
/*									*/
/************************************************************************/


static ANNOT_WDESC
request_window_type(w)
   ASH_WINDOW w;
{
   ANNOT_WDESC awd[32];
   Integer i,ct,wct;
   Character btn[32][64];
   String menu[40];
   Integer val;

   wct = ANNOT_inq_window_types(32,awd);
   if (wct == 0) return NULL;
   if (wct == 1) return awd[0];

   ct = 0;
   menu[ct++] = "%CAnnotation Window Type\n";

   for (i = 0; i < wct; ++i) {
      sprintf(btn[i],"%%0.%do %s",i,awd[i]->name);
      menu[ct++] = btn[i];
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   val = 0;
   if (!STEMdialog(w,menu,&val)) return NULL;

   return awd[val];
};





/************************************************************************/
/*									*/
/*	set_system -- handle system set request for this window 	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
set_system(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   Character sbuf[256];

   if (aw->system == NULL) sbuf[0] = 0;
   else strcpy(sbuf,aw->system);

   if (!FIELDsystem_request(aw->window,aw->desc->name,sbuf)) return FALSE;

   if (aw->system != NULL) {
      SFREE(aw->system);
      aw->system = NULL;
    };

   if (sbuf[0] == 0 || STREQL(sbuf,"*")) aw->system = NULL;
   else aw->system = SALLOC(sbuf);

   return TRUE;
};





/* end of annotwin.c */





