/************************************************************************/
/*									*/
/*		annotedit.c						*/
/*									*/
/*	Editor interface for annotation windows 			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "annot_local.h"
#include "fixannot.h"



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define ANNOT_SPACE	256



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	edit_bounds();
static	Integer 	edit_file();
static	void		release_file();
static	void		save_annotations();
static	void		set_time();




/************************************************************************/
/*									*/
/*	ANNOT_edit_init -- module initialization			*/
/*									*/
/************************************************************************/


void
ANNOT_edit_init()
{
};





/********************************************************************************/
/*										*/
/*	ANNOTedit_readonly -- make editor portion readonly or editable		*/
/*										*/
/********************************************************************************/


void
ANNOTedit_readonly(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   ANNOT_WIN aw;

   aw = ANNOT_find_window(w);
   if (aw == NULL || aw->edit_info == NULL) return;

   EDTenable(aw->edit_info,fg);
   WINDact_enable(aw->window,fg);
};





/************************************************************************/
/*									*/
/*	ANNOTedit_close -- close request				*/
/*									*/
/************************************************************************/


Boolean
ANNOTedit_close(w)
   ASH_WINDOW w;
{
   ANNOT_WIN aw;
   Boolean fg;

   aw = ANNOT_find_window(w);
   if (aw == NULL || aw->edit_info == NULL) return FALSE;

   fg = EDTclose(aw->edit_info);
   if (fg) aw->edit_info = NULL;

   return fg;
};





/************************************************************************/
/*									*/
/*	ANNOT_edit_setup -- setup editor in window			*/
/*									*/
/************************************************************************/


void
ANNOT_edit_setup(aw)
   ANNOT_WIN aw;
{
   EDT_DATA ed;

   if (aw->edit_win == NULL) return;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = aw->desc->readonly;
   ed->create = aw->desc->create;
   ed->temporary = FALSE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = aw->desc->savectl;
   ed->mode = aw->desc->mode;
   ed->disable = aw->desc->disable;

   ed->bounds_rtn = edit_bounds;
   ed->input_rtn = NULL;
   ed->mouse_rtn = ANNOT_edit_mouse;
   ed->file_rtn = edit_file;

   ed->input_filter = NULL;
   ed->output_filter = NULL;

   aw->edit_info = EDTdefine_editor(aw->edit_win,aw->filename,ed,aw,
				       aw->menu_win,aw->mesg_win);

   release_file(aw);
};





/************************************************************************/
/*									*/
/*	ANNOT_change_file -- change file and/or line number		*/
/*									*/
/************************************************************************/


Boolean
ANNOT_change_file(aw,file,line,alt,join)
   ANNOT_WIN aw;
   String file;
   Integer line;
   Boolean alt;
   Boolean join;
{
   String ofile;
   Integer i;

   if (aw->changing_file) {
      if (file != NULL && aw->new_file != NULL && MSGfile_compare(file,aw->new_file))
	 file = NULL;
      if (file == NULL) {
	 aw->new_line = line;
	 return TRUE;
       };
      return FALSE;
    }
   else {
      aw->changing_file = TRUE;
      aw->new_file = file;
      aw->new_line = line;
    };

   if (file != NULL) {
      ofile = aw->filename;
      if (aw->edit_info != NULL) {
	 ANNOT_clear(aw);
	 if (!EDTset_file(aw->edit_info,file)) {
	    aw->filename = ofile;
	    aw->wd = MSGinq_wd();
	    aw->wd = SALLOC(aw->wd);
	    return FALSE;
	  };
       };
      if (STRNEQ(aw->filename,file)) aw->filename = SALLOC(file);
      ANNOT_set_window_name(aw);
    };

   if (aw->edit_info != NULL) {
      i = EDTlast_line(aw->edit_info);
      if (aw->new_line > i && i > 0) aw->new_line = i;
      if (join) EDTjoin_views(aw->edit_info);
      if (alt) EDTgoto_alt(aw->edit_info,aw->new_line);
      else EDTgoto(aw->edit_info,aw->new_line);
    };

   aw->changing_file = FALSE;
   aw->new_file = NULL;
   aw->new_line = 0;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_edit_quit -- handle quit button on menu			*/
/*									*/
/************************************************************************/


Integer
ANNOT_edit_quit(aw,mnm,bnm)
   ANNOT_WIN aw;
   String mnm;
   String bnm;
{
   if (aw->edit_info != NULL) {
      if (!EDTclose(aw->edit_info)) return FALSE;
      aw->edit_info = NULL;
    };

   if (aw->window != NULL) ASHremove(aw->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	edit_bounds -- handle bounds changes				*/
/*									*/
/************************************************************************/


static Integer
edit_bounds(aw)
   ANNOT_WIN aw;
{
   if (aw->edit_info != NULL) ANNOT_update(aw,FALSE);
};





/************************************************************************/
/*									*/
/*	edit_file -- handle changes in files				*/
/*									*/
/************************************************************************/


static Integer
edit_file(aw,typ,name)
   ANNOT_WIN aw;
   EDT_FILE_TYPE typ;
   String name;
{
   switch (typ) {
      case EDT_FILE_PREOPEN :
	 MSGfile_check(name);
	 break;

      case EDT_FILE_OPEN :
	 if (aw->filename == NULL || STRNEQ(aw->filename,name)) {
	    if (aw->filename != NULL) SFREE(aw->filename);
	    aw->filename = SALLOC(name);
	    aw->wd = MSGinq_wd();
	    aw->wd = SALLOC(aw->wd);
	    ANNOT_set_window_name(aw);
	  };
	 release_file(aw);
	 ANNOT_send_open(aw);
	 break;

      case EDT_FILE_INQ_ABORT :
      case EDT_FILE_INQ_SAVE :
	 return TRUE;

      case EDT_FILE_ABORT :
	 ANNOT_clear(aw);
	 ANNOT_send_close(aw,FALSE);
	 break;

      case EDT_FILE_SAVE :
	 save_annotations(aw,name);
	 ANNOT_clear(aw);
	 ANNOT_send_close(aw,TRUE);
	 break;

      case EDT_FILE_SECURE :
	 save_annotations(aw,name);
	 break;

      default :
	 break;
    };

   return 0;
};





/********************************************************************************/
/*										*/
/*	release_file -- release opened file and handle prereading		*/
/*										*/
/********************************************************************************/


static void
release_file(aw)
   ANNOT_WIN aw;
{
   Integer i,ch,pid;
   Character buf[1024],mbuf[64];
   Character tbuf[10][1024];
   FILE * inf;
   Integer line,col,flags;
   String s,str[10];
   ANNOT_DESC ad;

   set_time(aw);

   if (aw->edit_info == NULL) return;

   aw->new_annot = NULL;
   if (aw->desc->parse) {
      sprintf(mbuf,"/tmp/ANNOTXXXXXX");
      mktemp(mbuf);
      MSGcommand_name("fixannot",buf);
      if ((pid = vfork()) == 0) {
	 execl(buf,"fixannot","-l","-f",mbuf,aw->filename,NULL);
	 _exit(127);
       };
      while (getpgrp(pid) > 0) {
	 CMPXselect(0);
       };

      inf = fopen(mbuf,"r");
      while (inf != NULL && fscanf(inf,"%d %d %d",&line,&col,&flags) == 3) {
	 while ((ch = getc(inf)) == ' ');

	 for (i = 0; i < 10; ++i) tbuf[i][0] = 0;

	 i = 0;
	 s = tbuf[i++];
	 while (ch != '\n' && ch != 0) {
	    if (ch == '~') {
	       *s = 0;
	       s = tbuf[i++];
	     }
	    else *s++ = ch;
	    ch = getc(inf);
	  };
	 *s = 0;

	 ad = ANNOT_find_desc(aw,tbuf[0]);
	 if (ad != NULL && MEMQ(ad,aw->allowed)) {
	    for (i = 0; i < NUM_TEXT; ++i) {
	       if (tbuf[i+1][0] == 0) str[i] = NULL;
	       else str[i] = tbuf[i+1];
	     };
	    ANNOT_do_add(aw,ad,line,col,str[0],str[1],str[2]);
	  };
       };
      if (inf != NULL) fclose(inf);
      unlink(mbuf);
    };
};





/************************************************************************/
/*									*/
/*	save_annotations -- save annotations with output file		*/
/*									*/
/************************************************************************/


static void
save_annotations(aw,name)
   ANNOT_WIN aw;
   String name;
{
   ANNOTATION an;
   Integer i,pid;
   Character buf[1024],lbuf[1024],tbuf[64];
   FILE * otf;

   set_time(aw);
   ANNOT_act_update(aw,"EDIT","UPDATE");

   if (!aw->desc->parse) return;

   sprintf(tbuf,"/tmp/ANNOTXXXXXX");
   mktemp(tbuf);

   otf = fopen(tbuf,"w");
   for (an = aw->first_annot; an != NULL; an = an->next) {
      if (an->desc->save) {
	 strcpy(lbuf,an->desc->name);

	 for (i = 0; i < NUM_TEXT; ++i) {
	    strcat(lbuf,"~");
	    if (an->text[i] != NULL) strcat(lbuf,an->text[i]);
	  };

	 fprintf(otf,"%d %d %d %s\n",an->line,an->column,
		    FIXANNOT_SEARCH,lbuf);

       };
    };
   fclose(otf);

   MSGcommand_name("fixannot",buf);

   if ((pid = vfork()) == 0) {
      execl(buf,"fixannot","-d","-f",tbuf,name,NULL);
      _exit(127);
    };
   while (getpgrp(pid) > 0) {
      CMPXselect(0);
    };

   unlink(tbuf);
};





/************************************************************************/
/*									*/
/*	set_time -- set time for file					*/
/*									*/
/************************************************************************/


static void
set_time(aw)
   ANNOT_WIN aw;
{
   struct stat sbuf;

   if (aw->filename == NULL) aw->mod_time = 0;
   else if (stat(aw->filename,&sbuf) < 0) aw->mod_time = 0;
   else aw->mod_time = sbuf.st_mtime;
};





/* end of annotedit.c */

