/************************************************************************/
/*									*/
/*		annotdata.c						*/
/*									*/
/*	Annotation data structure routines				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "annot_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_SHOW	16

#define LEFT_MARGIN	2




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	ANNOTATION	find_first();
static	ANNOTATION	find_previous();
static	Boolean 	show_annotation();
static	void		annot_display();





/************************************************************************/
/*									*/
/*	ANNOT_data_init -- module initialization			*/
/*									*/
/************************************************************************/


void
ANNOT_data_init()
{
};





/************************************************************************/
/*									*/
/*	ANNOT_add -- add annotation for a file				*/
/*									*/
/************************************************************************/


ANNOTATION
ANNOT_add(aw,desc,line,col)
   ANNOT_WIN aw;
   ANNOT_DESC desc;
   Integer line,col;
{
   ANNOTATION an;
   ANNOTATION prv;
   Integer i;

   if (aw->edit_info == NULL) return NULL;

   if (desc->unique) {
      an = ANNOT_search(aw,desc->unique_with,0,1);
      if (an != NULL) {
	 if (ANNOT_inq_line(an) == line && ANNOT_inq_char(an) == col &&
		 !an->desc->hilite) return an;
	 ANNOT_remove(an);
       };
    }
   else if (!desc->multiple) {
      for (an = find_first(aw,line);
	   an != NULL && ANNOT_inq_line(an) == line;
	   an = an->next) {
	 if (an->desc == desc) return an;
       };
    };

   prv = find_previous(aw,line,NULL);

   an = PALLOC(ANNOTATION_INFO);
   an->desc = desc;
   an->mark = EDTmarker(aw->edit_info,line,col);
   if (an->mark == NULL) return NULL;
   an->window = aw;
   an->line = line;
   an->column = col;
   an->value = 0;
   an->frozen = 0;
   for (i = 0; i < NUM_TEXT; ++i) an->text[i] = NULL;

   if (desc->save) EDTforce_save(aw->edit_info);

   if (prv == NULL) {
      an->next = aw->first_annot;
      aw->first_annot = an;
    }
   else {
      an->next = prv->next;
      prv->next = an;
    };

   ANNOT_display(aw,line,TRUE);

   return an;
};



/************************************************************************/
/*									*/
/*	ANNOT_remove -- remove annotation from window			*/
/*	ANNOT_remove_all -- remove all annotations of given type	*/
/*	ANNOT_clear -- clear all annotations				*/
/*									*/
/************************************************************************/


void
ANNOT_remove(an)
   ANNOTATION an;
{
   ANNOT_WIN aw;
   ANNOTATION an1;

   aw = an->window;
   an1 = find_previous(aw,0,an);

   if (an1 == NULL) aw->first_annot = an->next;
   else an1->next = an->next;

   ANNOT_display(aw,ANNOT_inq_line(an),TRUE);

   if (aw->last_display == an) aw->last_display = NULL;

   if (an->frozen == 0) free(an);
   else an->frozen = -1;
};




void
ANNOT_user_remove(an,force)
   ANNOTATION an;
   Boolean force;
{
   ANNOT_send_remove(an);

   if (force || an->desc->display || an->desc->msg_remove == NULL) ANNOT_remove(an);
};





void
ANNOT_remove_all(aw,typ)
   ANNOT_WIN aw;
   ANNOT_DESC typ;
{
   ANNOTATION an,nan;

   for (an = aw->first_annot; an != NULL; an = nan) {
      nan = an->next;
      if (an->desc == typ) ANNOT_user_remove(an,TRUE);
    };
};





void
ANNOT_clear(aw)
   ANNOT_WIN aw;
{
   ANNOTATION an,nan;
   Integer lx,by,rx,ty;

   for (an = aw->first_annot; an != NULL; an = nan) {
      nan = an->next;
      ANNOT_remove(an);
    };

   if (aw->annot_win != NULL) {
      ASHsensitive_remove_all(aw->annot_win);

      ASHinq_size(aw->annot_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      ASHclip(aw->annot_win,FALSE);
      ASHclear_box(aw->annot_win,lx,by,rx,ty);
      aw->top_line[0] = -1;
    };
};





/************************************************************************/
/*									*/
/*	ANNOT_rotate -- rotate displayed annotation for line		*/
/*									*/
/************************************************************************/


void
ANNOT_rotate(an)
   ANNOTATION an;
{
   ANNOTATION fst,nan,prv;
   Integer line;

   line = ANNOT_inq_line(an);
   prv = find_previous(an->window,NULL,an);

   fst = NULL;
   for (nan = an->next; nan != NULL; nan = nan->next) {
      if (ANNOT_inq_line(nan) != line) break;
      fst = nan;
    };

   if (fst == NULL) return;

   if (prv == NULL) an->window->first_annot = an->next;
   else prv->next = an->next;
   fst->next = an;
   an->next = nan;

   ANNOT_display(an->window,line,TRUE);
};






/************************************************************************/
/*									*/
/*	ANNOT_search -- search for a given annotation			*/
/*									*/
/************************************************************************/


ANNOTATION
ANNOT_search(aw,type,line,dir)
   ANNOT_WIN aw;
   ANNOT_DESC type;
   Integer line;
   Integer dir;
{
   ANNOTATION an,xan,lan;

   if (dir > 0) {
      if (line > 0) {
	 xan = find_previous(aw,line+1,NULL);
	 if (xan != NULL) xan = xan->next;
	 else xan = aw->first_annot;
       }
      else {
	 xan = aw->first_annot;
       };
      for (an = xan; an != NULL; an = an->next) {
	 if (an->desc == type) break;
	 if (an->desc->unique_with == type) break;
       };
    }
   else {
      an = NULL;
      if (line > 0) xan = ANNOT_find(aw,line,NULL);
      else xan = NULL;
      for (lan = aw->first_annot; lan != NULL; lan = lan->next) {
	 if (xan != NULL && lan == xan) break;
	 if (xan == NULL && line > 0 && ANNOT_inq_line(lan) >= line) break;
	 if (lan->desc == type) an = lan;
       };
    };

   return an;
};




/************************************************************************/
/*									*/
/*	ANNOT_find -- find current annotation at given line		*/
/*									*/
/************************************************************************/


ANNOTATION
ANNOT_find(aw,line,type)
   ANNOT_WIN aw;
   Integer line;
   ANNOT_DESC type;
{
   ANNOTATION an,fst;

   fst = find_first(aw,line);

   if (type != NULL) {
      for (an = fst; an != NULL; an = an->next) {
	 if (ANNOT_inq_line(an) != line) {
	    an = NULL;
	    break;
	  }
	 else if (an->desc == type) break;
       };
      fst = an;
    };

   return fst;
};





/************************************************************************/
/*									*/
/*	ANNOT_find_value -- find by type value and line 		*/
/*									*/
/************************************************************************/


ANNOTATION
ANNOT_find_value(aw,ad,line,value)
   ANNOT_WIN aw;
   ANNOT_DESC ad;
   Integer line;
   Integer value;
{
   ANNOTATION an,fan;

   fan = NULL;

   for (an = find_first(aw,line);
	an != NULL && ANNOT_inq_line(an) == line;
	an = an->next) {
      if (an->desc == ad && an->value == value) {
	 fan = an;
	 break;
       };
    };

   if (fan == NULL) {
      for (an = aw->first_annot; an != NULL; an = an->next) {
	 if (an->value == value) {
	    fan = an;
	    break;
	  };
       };
    };

   return fan;
};





/************************************************************************/
/*									*/
/*	ANNOT_iterate -- iterate over all annotations			*/
/*									*/
/************************************************************************/


void
ANNOT_iterate(aw,fct,data)
   ANNOT_WIN aw;
   Function_Ptr fct;
   Universal data;
{
   ANNOTATION an,nan;

   for (an = aw->first_annot; an != NULL; an = nan) {
      nan = an->next;
      (*fct)(an,data);
    };
};





/************************************************************************/
/*									*/
/*	ANNOT_update -- handle editor bounds moved			*/
/*									*/
/************************************************************************/


void
ANNOT_update(aw,force)
   ANNOT_WIN aw;
   Boolean force;
{
   Integer top,bot;
   ANNOTATION an,nan;
   Integer i,j;
   Integer lx,by,rx,ty;
   Integer ct,wid,tops[MAX_PANES],bots[MAX_PANES];

   if (aw->annot_win == NULL || aw->filename == NULL || aw->edit_info == NULL) return;

   aw->linevalid = FALSE;

   for (an = aw->first_annot; an != NULL; an = nan) {
      nan = an->next;
      i = ANNOT_inq_line(an);
      if (i < 0) {
	 ANNOT_remove(an);
	 force = TRUE;
       }
      else {
	 if (an->line != i) {
	    an->line = i;
	    force = TRUE;
	  };
	 i = ANNOT_inq_char(an);
	 an->column = i;
       };
    };

   aw->linevalid = TRUE;

   ct = EDTinq_bounds(aw->edit_info,&wid,tops,bots,MAX_PANES);

   if (!force) {
      for (i = 0; i < ct; ++i) {
	 if (tops[i] != aw->top_line[i] || bots[i] != aw->bot_line[i]) break;
       };
      if (i < ct) force = TRUE;
    };

   if (!force || ct <= 0) return;

   ASHbatch_mode(TRUE);

   for (i = 0; i < ct; ++i) {
      aw->top_line[i] = tops[i];
      aw->bot_line[i] = bots[i];
    };

   top = tops[0];
   bot = bots[0];
   for (i = 1; i < ct; ++i) {
      if (tops[i] < top) top = tops[i];
      if (bots[i] > bot) bot = bots[i];
    };

   ASHsensitive_remove_all(aw->annot_win);

   ASHinq_size(aw->annot_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclip(aw->annot_win,FALSE);
   ASHclear_box(aw->annot_win,lx,by,rx,ty);

   for (i = top; i <= bot; ++i) {
      ct = EDTinq_line_pos(aw->edit_info,i,bots,tops,MAX_PANES);
      if (ct > 0) ANNOT_display(aw,i,FALSE);
      if (aw->desc->sensitive != 0) {
	 for (j = 0; j < ct; ++j) {
	    ty = tops[j];
	    by = bots[j];
	    ASHsensitive_area(aw->annot_win,lx,by,rx,ty,aw->desc->sensitive);
	  };
       };
    };

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	ANNOT_display -- display an annotation				*/
/*									*/
/************************************************************************/


void
ANNOT_display(aw,line,clear)
   ANNOT_WIN aw;
   Integer line;
   Boolean clear;
{
   ANNOTATION an;
   Integer ct;
   ANNOTATION show[MAX_SHOW];
   Boolean hilite;

   if (line < 0 || aw->window == NULL) return;

   ct = 0;
   hilite = FALSE;

   for (an = find_first(aw,line); an != NULL && ANNOT_inq_line(an) == line; an = an->next) {
      if (show_annotation(an,aw->level) && ct < MAX_SHOW) {
	 if (an->desc->hilite) hilite = TRUE;
	 else show[ct++] = an;
       };
      aw->last_display = an;
    };

   if (aw->annot_win == NULL) ct = 0;

   if (ct == 0 && !clear) return;

   annot_display(aw,line,ct,show,hilite);
};





/************************************************************************/
/*									*/
/*	ANNOT_inq_line -- return line for annotation			*/
/*	ANNOT_inq_char -- return character position for annotation	*/
/*									*/
/************************************************************************/


Integer
ANNOT_inq_line(an)
   ANNOTATION an;
{
   if (an->window->linevalid) return an->line;

   return EDTmark_line(an->window->edit_info,an->mark);
};





Integer
ANNOT_inq_char(an)
   ANNOTATION an;
{
   if (an->window->linevalid) return an->column;

   return EDTmark_char(an->window->edit_info,an->mark);
};





/************************************************************************/
/*									*/
/*	ANNOT_get_line -- get line number given Y coord 		*/
/*									*/
/************************************************************************/


Integer
ANNOT_get_line(aw,y)
   ANNOT_WIN aw;
   Integer y;
{
   return EDTcorrelate_line(aw->edit_info,y);
};





/************************************************************************/
/*									*/
/*	find_first -- find first annotation for a line			*/
/*	find_previous -- find annotation previous to a line		*/
/*									*/
/************************************************************************/


static ANNOTATION
find_first(aw,line)
   ANNOT_WIN aw;
   Integer line;
{
   ANNOTATION an;

   if (aw->last_display == NULL || ANNOT_inq_line(aw->last_display) >= line)
      an = aw->first_annot;
   else an = aw->last_display;

   for ( ; an != NULL && ANNOT_inq_line(an) < line; an = an->next);

   if (an != NULL && ANNOT_inq_line(an) != line) an = NULL;

   return an;
};





static ANNOTATION
find_previous(aw,line,annot)
   ANNOT_WIN aw;
   Integer line;
   ANNOTATION annot;
{
   ANNOTATION an,pan;

   pan = NULL;
   for (an = aw->first_annot; an != NULL; an = an->next) {
      if (annot == NULL && ANNOT_inq_line(an) >= line) break;
      else if (an == annot) break;
      pan = an;
    };

   return pan;
};





/************************************************************************/
/*									*/
/*	show_annotation -- determine if annotation should be displayed	*/
/*									*/
/************************************************************************/


static Boolean
show_annotation(an,lvl)
   ANNOTATION an;
   Integer lvl;
{
   Boolean fg;

   fg = TRUE;

   if (an->desc->display_level > lvl) fg = FALSE;

   if (an->desc->display_text == NULL) fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	annot_display -- display an annotation				*/
/*									*/
/************************************************************************/


static void
annot_display(aw,line,act,ans,hilite)
   ANNOT_WIN aw;
   Integer line;
   Integer act;
   ANNOTATION ans[];
   Boolean hilite;
{
   Integer lx,by,rx,ty;
   ANNOTATION an;
   Integer ct,lby[MAX_PANES],lty[MAX_PANES];
   Integer xo,yo,ft,lx0,xe,ye;
   Integer i,j,tc;

   if (hilite && aw->edit_info != NULL) EDThilite_line(aw->edit_info,line);
   if (aw->annot_win == NULL) return;

   ct = EDTinq_line_pos(aw->edit_info,line,lby,lty,MAX_PANES);

   if (ct == 0) return;

   ASHinq_size(aw->annot_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   lx0 = lx;

   for (i = 0; i < ct; ++i) {
      by = lby[i];
      ty = lty[i];
      ASHclip_region(aw->annot_win,lx0,by,rx,ty);
      ASHclip(aw->annot_win,TRUE);
      ASHclear_box(aw->annot_win,lx0,by,rx,ty);

      lx = lx0+LEFT_MARGIN;
      for (j = 0; j < act; ++j) {
	 an = ans[j];
	 ft = an->desc->display_font;
	 ASHinq_text_offset(ft,an->desc->display_text,&xo,&yo);
	 ASHinq_text_extent(ft,an->desc->display_text,&xe,&ye);
	 yo += (by-ty-ye)/2;
	 ASHfont(aw->annot_win,ft);
	 tc = ASHtext_color(aw->annot_win,an->desc->color);
	 ASHtext(aw->annot_win,lx+xo,by-yo,an->desc->display_text);
	 ASHtext_color(aw->annot_win,tc);
	 ASHinq_text_next(ft,an->desc->display_text,&xo,&yo);
	 lx += xo;
       };
    };
};





/* end of annotdata.c */
