/*
mailbox.c 1.4 10/6/92
	Handles all mailbox file manipulations for the POP3 server.

	Written by W. Gregg Stefancik (wstef@eng.clemson.edu)
*/

static char *sccs_id = "@(#)mailbox.c	1.4 (wstef@eng.clemson.edu) 10/6/92";

#include "pop3d.h"

#define POP_LOCK "/tmp/pop.%s"

#define MAILBOX_FILE	"/usr/spool/mail/%s"
#define MAILBOX_LOCK	"/usr/spool/mail/%s.lock"
#define TMP_MAILBOX_PFX	"pop3m"
#define TMP_NEWMAIL_PFX	"pop3n"

char mail_lock_name[100];
char *tmp_mbox_name, *tmp_new_name;
int  tmp_mbox_fd = -1;
int  total_msgs, last_msg_rd, cur_msgs;
long cur_size;
struct mail_ent *me_head=NULL, *me_tail=NULL;
struct stat mb_stat;
FILE *mbox_fp;

lock_popbox(user)
char *user;
{
  char pop_lock[100];
  int lock_fd, pid;

  sprintf(pop_lock,POP_LOCK,user);
  if((lock_fd=open(pop_lock,O_CREAT|O_EXCL|O_WRONLY,0644)) < 0)
    return(FALSE);
  pid=getpid();
  /* jam the process id into the lock file */
  write(lock_fd,&pid,sizeof(pid));
  close(lock_fd);
  return(TRUE);
}

unlock_popbox(user)
char *user;
{
  char pop_lock[100];

  sprintf(pop_lock,POP_LOCK,user);
  unlink(pop_lock);
  return(TRUE);
}

read_mailbox(user)
char *user;
{
  char mailbox_name[100];
  int mailbox_fd;

  cur_size=0;
  if((tmp_mbox_name=tempnam(NULL,TMP_MAILBOX_PFX)) == NULL)
  {
    if(debug)
      fprintf(stderr,"read_mail: couldn't get scratch mbox name\n");
    return(FALSE);
  }
  if((tmp_mbox_fd=open(tmp_mbox_name,O_CREAT|O_RDWR|O_TRUNC,0600)) < 0)
  {
    if(debug)
      fprintf(stderr,"read_mail: couldn't create scratch mbox\n");
    return(FALSE);
  }
  /* lock and copy contents of /usr/spool/mail to tmp_mbox */
  if(!lock_spool_mail(user))
  {
    close(tmp_mbox_fd);
    unlink(tmp_mbox_name);
    free(tmp_mbox_name);
    return(FALSE);
  }
  sprintf(mailbox_name,MAILBOX_FILE,user);
  if((mailbox_fd=open(mailbox_name,O_RDONLY)) < 0)
  {
    unlock_spool_mail(user); /* if non-existant mail box or can't read unlock */
    if(errno != ENOENT)
    {
      if(debug)
	fprintf(stderr,"read_mailbox: unexpected errno %d on open %s",errno,
         mailbox_name);
      close(tmp_mbox_fd);
      unlink(tmp_mbox_name);
      free(tmp_mbox_name);
      return(FALSE);
    }
    mb_stat.st_size=0; /* fill in the info for call update */
    mb_stat.st_mtime=0;
  }
  else
  {
    if(!fd_copy(mailbox_fd,tmp_mbox_fd))
    {
      if(debug)
	fprintf(stderr,"read_mail: error copying mailbox %d\n",errno);
      close(mailbox_fd);
      unlock_spool_mail(user);
      close(tmp_mbox_fd);
      unlink(tmp_mbox_name);
      free(tmp_mbox_name);
      return(FALSE);
    }
    /* do the stat thing so we can tell if things change underneath us */
    fstat(mailbox_fd,&mb_stat);
    cur_size=mb_stat.st_size;
    close(mailbox_fd);
    unlock_spool_mail(user);
  }
  /* grock down the mail */
  total_msgs=last_msg_rd=cur_msgs=0;
  me_head=me_tail=NULL;
  if(mailbox_fd < 0)
    return(TRUE);	/* no messages in maildrop/box */
  if((mbox_fp = fdopen(tmp_mbox_fd,"r")) == NULL)
  {
    if(debug)
      fprintf(stderr,"read_mailbox: could not get mbox_fp from mbox_fd\n");
    close(tmp_mbox_fd);
    unlink(tmp_mbox_name);
    free(tmp_mbox_name);
    return(FALSE);
  }
  /* get to the beginning so we can process */
  rewind(mbox_fp);
  parse_down_mbox(mbox_fp);
  cur_msgs=total_msgs;
  if(debug)
    show_entries(me_head);
  last_msg_rd=find_new_mail(me_head) - 1;
  return(TRUE);
}

parse_down_mbox(mbox_fp)
FILE *mbox_fp;
{
  char inline[256],is_ln_st=TRUE;
  long cur_pos, last_pos;
  struct mail_ent *me_pre, *me_ptr=NULL;

  cur_pos=last_pos=0;
  while(fgets(inline,sizeof(inline),mbox_fp) != NULL)
  {
    if(!is_ln_st) continue;  /* did we read a partial last time ? */
    /* see if we really read an entire line or not */
    if(index(inline,'\n') == NULL)
      is_ln_st = FALSE;
    else
      is_ln_st = TRUE;
    if(strncmp(inline,"From ",5) == NULL)
    {
      me_pre=me_ptr;
      last_pos=cur_pos;
      cur_pos=ftell(mbox_fp) - strlen(inline);
      me_ptr=(struct mail_ent *)malloc(sizeof(struct mail_ent));
      me_ptr->me_msg_id = ++total_msgs;
      me_ptr->me_status=0;
      me_ptr->me_start_pos=cur_pos;
      if((me_ptr->me_prev=me_pre) == NULL)
	me_head=me_ptr;
      else
      {
	me_pre->me_next=me_ptr;
	me_tail = me_ptr;
	me_pre->me_length=cur_pos-last_pos;
      }
    }
    if(strncmp(inline,"Status: ",8) == NULL)
    {
      if(index(inline,'R') != NULL)
        me_ptr->me_status |= MST_READ;
      if(index(inline,'O') != NULL)
        me_ptr->me_status |= MST_OLD;
    }
  }
  if(me_ptr != NULL)
  {
    me_ptr->me_next=NULL;
    me_tail=me_ptr;
    me_ptr->me_length=ftell(mbox_fp) - cur_pos;
  }
}

find_new_mail(start)
struct mail_ent *start;
{
  struct mail_ent *mep;

  for(mep=start; mep != NULL; mep = mep->me_next)
    if(mep->me_status == MST_NEW)
      return(mep->me_msg_id);
  return(total_msgs+1);
}

list_all()
{
  char outline[100], found=FALSE;
  struct mail_ent *mep;

  for(mep=me_head; mep != NULL; mep=mep->me_next)
    if(!(mep->me_status & MST_DELETE))
    {
      if(!found)
      {
	found=!found;
	send_ok("");
      }
      sprintf(outline,"%d %d\r\n",mep->me_msg_id,mep->me_length);
      write(con_sock,outline,strlen(outline));
    }
  if(!found)
    send_err("no messages to list");
  else
    write(con_sock,".\r\n",3);
}

list_msg(msg_num)
int msg_num;
{
  struct mail_ent *mep;

  if(msg_num > total_msgs || msg_num < 1 || me_head == NULL)
    send_err("message number out of range");
  else
    for(mep=me_head; mep != NULL; mep=mep->me_next)
      if(mep->me_msg_id == msg_num)
      {
	if(mep->me_status & MST_DELETE)
 	  send_err("message had been deleted");
	else
	  send_ok("%d %d", mep->me_msg_id, mep->me_length);
	break;
      }
}

delete_msg(msg_num)
int msg_num;
{
  struct mail_ent *mep;

  if(msg_num > total_msgs || msg_num < 1 || me_head == NULL)
    send_err("message number out of range");
  else
    for(mep=me_head; mep != NULL; mep=mep->me_next)
      if(mep->me_msg_id == msg_num)
      {
	if(mep->me_status & MST_DELETE)
	  send_err("message has been deleted");
	else
	{
	  mep->me_status |= MST_DELETE;
	  cur_msgs--;
	  cur_size -= mep->me_length;
	  if(msg_num > last_msg_rd)
	    last_msg_rd = msg_num;
	  send_ok("message deleted");
	}
	break;
      }
}

retreve_msg(msg_num, lns_pst_hdr)
int msg_num, lns_pst_hdr;
{
  char buffer[101], *nl_ptr;
  char is_ln_st=TRUE, fst_bl_fnd=FALSE, stat_found=FALSE;
  int b_len, phdr_lns=0;
  long to_go;
  struct mail_ent *mep;

  if(msg_num > total_msgs || msg_num < 1 || me_head == NULL)
    send_err("message number out of range");
  else
    for(mep=me_head; mep != NULL; mep=mep->me_next)
      if(mep->me_msg_id == msg_num)
      {
	if(mep->me_status & MST_DELETE)
	  send_err("message has been deleted");
	else
	{
	  fseek(mbox_fp,mep->me_start_pos,0);
	  send_ok("here comes the message");
	  to_go = mep->me_length;
	  do
	  {
	    fgets(buffer,100,mbox_fp);
	    to_go -= (b_len=strlen(buffer));
	    if(is_ln_st)
	    {
	      if(!stat_found && !fst_bl_fnd)
	        if(strncmp("Status: ",buffer,8) == NULL)
		  stat_found=TRUE;
	      if(fst_bl_fnd && lns_pst_hdr > -1)
		if(++phdr_lns > lns_pst_hdr)
		  break;
	      if(!fst_bl_fnd)
		if(buffer[0] == '\n')
		{
		  if(!stat_found)
		    write(con_sock,"Status: R\r\n",11);
		  fst_bl_fnd=TRUE;
		}
	      if(buffer[0] == '.')
		write(con_sock,".",1);
	    }
	    if((nl_ptr=index(buffer,'\n')) != NULL)
	    {
	      is_ln_st=TRUE;
	      if(nl_ptr == buffer)
	      {
		strcpy(nl_ptr,"\r\n");
		b_len++;
	      }
	      else
		if(*(nl_ptr-1) != '\r')
		{
		  strcpy(nl_ptr,"\r\n");
		  b_len++;
		}
	    }
	    else
	      is_ln_st=FALSE;
	    write(con_sock,buffer,b_len);
	  } while(to_go > 0);
	  write(con_sock,".\r\n",3);
	  if(msg_num > last_msg_rd)
	    last_msg_rd = msg_num;
	  mep->me_status |= MST_READ;
	}
	break;
      }
}

reset_mailbox()
{
  struct mail_ent *mep;

  for(mep=me_head; mep != NULL; mep = mep->me_next)
  {
    if(mep->me_status & MST_DELETE)
    {
      mep->me_status ^= MST_DELETE;
      cur_msgs++;
      cur_size += mep->me_length;
    }
    if((mep->me_status & MST_READ) && !(mep->me_status & MST_OLD))
      mep->me_status ^= MST_READ;
  }
  last_msg_rd = find_new_mail(me_head) - 1;
  send_ok("maildrop reset %d messages %d octets",cur_msgs,cur_size);
}

free_mail_entries()
{
  struct mail_ent *mep, *nxt;

  if(me_head == NULL)
    return(TRUE);
  for(mep=me_head; mep != NULL; mep = nxt)
  {
    nxt = mep->me_next;
    free(mep);
  }
  me_head = NULL;
  me_tail = NULL;
  return(TRUE);
}

show_entries(start)
struct mail_ent *start;
{
  struct mail_ent *mep;

  for(mep=start; mep != NULL; mep = mep->me_next)
  {
    fprintf(stderr,"msg id: %d  status: %d\n",mep->me_msg_id,mep->me_status);
    fprintf(stderr,"start(length): %d (%d)\n\n",mep->me_start_pos,mep->me_length);
  }
}

update_mailbox(user)
char *user;
{
  char mailbox_name[100];
  int mailbox_fd, new_mail_fd = -1;
  struct mail_ent *mep;
  struct stat cmb_stat;

  if(mb_stat.st_size == 0)  /* no updating necessary when you have no mail */
  {
    close(tmp_mbox_fd);
    unlink(tmp_mbox_name);
    free(tmp_mbox_name);
    return(TRUE);
  }
  if(!lock_spool_mail(user))
  {
    if(debug)
      fprintf(stderr,"update_mail: could not get lock on spool mail");
    return(FALSE);
  }
  sprintf(mailbox_name,MAILBOX_FILE,user);
  if((mailbox_fd=open(mailbox_name,O_RDWR)) < 0)
  {
    if(debug)
      fprintf(stderr,"update_mail: can't open spool mail for %s\n",user);
    unlock_spool_mail(user);
    return(FALSE);
  }
  if(fstat(mailbox_fd,&cmb_stat) < 0)
  {
    if(debug)
      fprintf(stderr,"update_mail: can't stat spool mail for %s\n",user);
    unlock_spool_mail(user);
    return(FALSE);
  }
  if(mb_stat.st_size > cmb_stat.st_size)
  {
    if(debug)
      fprintf(stderr,"update_mail: mailbox shrunk during session.\n",user);
    unlock_spool_mail(user);
    return(FALSE);
  }
  if(mb_stat.st_size < cmb_stat.st_size)
  {
    if((tmp_mbox_name=tempnam(NULL,TMP_MAILBOX_PFX)) == NULL)
    {
      if(debug)
        fprintf(stderr,"update_mail: couldn't get scratch mbox name\n");
      return(FALSE);
    }
    if((new_mail_fd=open(tmp_new_name,O_CREAT|O_RDWR|O_TRUNC,0600)) < 0)
    {
      if(debug)
        fprintf(stderr,"update_mail: couldn't create scratch mbox\n");
      return(FALSE);
    }
    lseek(mailbox_fd,mb_stat.st_size,L_SET);
    fd_copy(mailbox_fd,new_mail_fd);
  }
  lseek(mailbox_fd,0l,L_SET); /* get to the beginning */
  ftruncate(mailbox_fd,0l); /* smash down the mail box */
  for(mep=me_head; mep != NULL; mep=mep->me_next)
    if(!(mep->me_status & MST_DELETE))
      write_mailbox_msg(mailbox_fd,mep);
  fclose(mbox_fp);
  close(tmp_mbox_fd);
  unlink(tmp_mbox_name);
  free(tmp_mbox_name);
  if(new_mail_fd > -1)
  {
    lseek(mailbox_fd,0l,L_XTND);
    lseek(new_mail_fd,0l,L_SET);
    fd_copy(new_mail_fd,mailbox_fd);
    close(new_mail_fd);
    unlink(tmp_new_name);
    free(tmp_new_name);
  }
  close(mailbox_fd);
  unlock_spool_mail(user);
  return(TRUE);
}

write_mailbox_msg(mailbox_fd,mep)
int mailbox_fd;
struct mail_ent *mep;
{
  char buffer[100];
  char is_ln_st=TRUE, fst_bl_fnd=FALSE, stat_found=FALSE;
  int b_len;
  long to_go;

  fseek(mbox_fp,mep->me_start_pos,0); /* get to start of message */
  to_go = mep->me_length;
  do
  {
    fgets(buffer,100,mbox_fp);
    to_go -= (b_len=strlen(buffer));
    if(is_ln_st)
    {
      if(!stat_found && !fst_bl_fnd)
        if(strncmp("Status: ",buffer,8) == NULL)
          stat_found=TRUE;
      if(!fst_bl_fnd)
        if(buffer[0] == '\n')
        {
          if(!stat_found && (mep->me_status & MST_READ))
            write(mailbox_fd,"Status: RO\n",11);
            fst_bl_fnd=TRUE;
        }
    }
    if(index(buffer,'\n') != NULL)
      is_ln_st=TRUE;
    else
      is_ln_st=FALSE;
    write(mailbox_fd,buffer,b_len);
  } while(to_go > 0);
}

lock_spool_mail(user)
char *user;
{
  int mail_lock_fd,pid,lock_tries=0;

  /* lock and copy contents of /usr/spool/mail to tmp_mbox */
  sprintf(mail_lock_name,MAILBOX_LOCK,user);
  while((mail_lock_fd=open(mail_lock_name,O_CREAT|O_EXCL|O_WRONLY,0644)) < 0)
  {
    sleep(1);
    if(++lock_tries > 5)
    {
      if(debug)
        fprintf(stderr,"lock_mail: couldn't get lock on %s's mail\n",user);
      return(FALSE);
    }
  }
  pid=getpid();
  write(mail_lock_fd, &pid, sizeof(pid));
  close(mail_lock_fd);
  return(TRUE);
}

fd_copy(source,dest)
int source, dest;
{
  char buffer[4096];
  int len;

  do
  {
    if((len=read(source,buffer,sizeof(buffer))) < 0)
      return(FALSE);
    if(len)
      if(write(dest,buffer,len) != len)
	return(FALSE);
  }
  while(len == sizeof(buffer));
  return(TRUE);
}
