(* APPLY function - Definition v3 page 77 *)

(*
$File: Interpreter/Apply.sml $
$Date: 1992/12/18 08:59:54 $
$Revision: 1.10 $
$Locker:  $
*)

(*$Apply:
	CORE_DYNOBJECT SPECIAL_VALUE BASIC_VALUE SORTED_FINMAP
	IO_STREAMS CRASH APPLY
 *)

functor Apply(structure CoreDynObject : CORE_DYNOBJECT

	      structure SpecialValue : SPECIAL_VALUE
		sharing type SpecialValue.SVal = CoreDynObject.SVal

	      structure BasicValue : BASIC_VALUE
		sharing type BasicValue.BasVal = CoreDynObject.BasVal

	      structure SortedFinMap: SORTED_FINMAP
		sharing type CoreDynObject.map = SortedFinMap.map
	  
	      structure IOStreams: IO_STREAMS
	      structure Crash : CRASH

	      val USE: string -> unit	(* use a file, raise host ML's Io if
					   unable to open it. *)

	      val RAISE: CoreDynObject.Val -> 'a
	     ): APPLY =
  struct
    type BasVal = BasicValue.BasVal
    type Val    = CoreDynObject.Val

    structure D = CoreDynObject
    structure B = BasicValue
    structure S = SpecialValue

    exception Use	(* APPLY provides a function FAIL_USE for the compiler
			   to call if compilation from a nested `use'
			   context fails. We catch it and return the correct
			   Kit exception in the implementation of the `use'
			   prim below. *)

    fun FAIL_USE() = raise Use

   (* tuple extraction. *)

    fun nth(v, i) =
      let
	open CoreDynObject
      in
	case v
	  of RECORDval env =>
	       (case SortedFinMap.lookup env (Lab.mk_IntegerLab i)
		  of Some v => v
		   | None => Crash.impossible "Apply.nth(lookup)"
	       )

	   | _ =>
	       Crash.impossible "Apply.nth(?)"
      end

   (* Value decomposition. *)

    fun valToINT(D.Sval sv) = S.unSValInt sv
      | valToINT _ = Crash.impossible "Apply.valToINT"

    and valToSTRING(D.Sval sv) = S.unSValString sv
      | valToSTRING _ = Crash.impossible "Apply.valToSTRING"

    and valToREAL(D.Sval sv) = S.unSValReal sv
      | valToREAL _ = Crash.impossible "Apply.valToREAL"

    and valToADDR v =
      case v
	of CoreDynObject.ADDRval a => a
	 | _ => Crash.impossible "Apply.valToADDR"

   (* unwrap List: turn a DynObject list into an ML one. *)

    fun unwrapList v =
      case v
	of D.CON0val c =>
	     (Crash.assert("Apply.unwrapList(CON0)", c = D.Con.con_NIL);
	      nil
	     )

	 | D.CON1val(c, v) =>
	     (Crash.assert("Apply.unwrapList(CON1)", c = D.Con.con_CONS);
	      let
		val hd = nth(v, 1)
		val tl = nth(v, 2)
	      in
		hd :: unwrapList tl
	      end
	     )

	 | _ =>
	     Crash.impossible "Apply.unwrapList"

   (* Value construction from integers, strings, reals: *)

    val valOfINT = D.Sval o S.mkSValInt
    and valOfSTRING = D.Sval o S.mkSValString
    and valOfREAL = D.Sval o S.mkSValReal

   (* Runtime values for `true' `false', `nil', unit: *)

    val val_TRUE = D.CON0val D.Con.con_TRUE
    val val_FALSE = D.CON0val D.Con.con_FALSE
    val val_NIL = D.CON0val D.Con.con_NIL
    val val_UNIT = D.unit

   (* How to build a boolean result: *)

    fun valOfBOOL true  = val_TRUE
      | valOfBOOL false = val_FALSE

   (* How to build lists: *)

    fun CONS(h, t) = D.CON1val(D.Con.con_CONS, D.pair(h, t))

   (* Polymorphic equality: *)

    fun equal(v1, v2) =
      case (v1, v2)
	of (D.Sval sv1, D.Sval sv2) => (sv1 = sv2)

	 | (D.CON0val con1, D.CON0val con2) => (con1 = con2)

	 | (D.CON1val(con1, v1), D.CON1val(con2, v2)) =>
	     (con1 = con2) andalso equal(v1, v2)

	 | (D.RECORDval map1, D.RECORDval map2) =>
	     SortedFinMap.equal equal (map1, map2)

			(* Exception packets can never be compared. *)
	 | (D.EXNAME0val _, _) => Crash.impossible "Apply.equal(EXNAME0, _)"
	 | (_, D.EXNAME0val _) => Crash.impossible "Apply.equal(_, EXNAME0)"
	 | (D.EXNAME1val _, _) => Crash.impossible "Apply.equal(EXNAME1, _)"
	 | (_, D.EXNAME1val _) => Crash.impossible "Apply.equal(_, EXNAME1)"

			(* All the basic values (prim and overloaded variables)
			   are functional, so comparison should never happen. *)
	 | (D.BASval _, _) => Crash.impossible "Apply.equal(BASval, _)"
	 | (_, D.BASval _) => Crash.impossible "Apply.equal(_, BASval)"

			(* Comparison of function types should never happen. *)
	 | (D.CLOSUREval _, _) => Crash.impossible "Apply.equal(CLOSURE, _)"
	 | (_, D.CLOSUREval _) => Crash.impossible "Apply.equal(_, CLOSURE)"

	 | (D.ADDRval a1, D.ADDRval a2) =>
	     (a1 = a2)	(* Reference equality. *)

			(* Other (legal) combinations such as CON0=CON1. *)
	 | _ => false


   (* Besides the overloaded variables, there is a single object of type BasVal: "prim".
      It is a function of type (int * 'a) -> 'b, where the integer is an
      index to the primitive function we need. Look at the prelude; the numbers
      it uses must match those below. Equality is prim 0.

      Oh, yes. It looks as if we'll have to have a persistent stream set
      here to make the external interface to Apply simple. Here it is: *)

    val theStreams = ref IOStreams.initialStreams

    open BasicValue 

    fun APPLY(b, v) =
    let
      fun arg n = nth(v, n)
      fun fst() = arg 1 and snd() = arg 2 and thd() = arg 3
    in
      case b of 
	ABS_INT =>
	        (valOfINT(abs(valToINT(v)))
		 handle Abs => RAISE CoreDynObject.exAbs
		   )
      | NEG_INT =>
		(valOfINT(~(valToINT(v)))
		 handle Neg => RAISE CoreDynObject.exNeg
		   )
      |	SUM_INT =>
		(valOfINT(valToINT(fst()) + valToINT(snd()))
		 handle Sum => RAISE CoreDynObject.exSum 
		   )
      |	DIFF_INT =>
		(valOfINT(valToINT(fst()) - valToINT(snd()))
		 handle Diff => RAISE CoreDynObject.exDiff
		   )
      |	PROD_INT =>
		(valOfINT(valToINT(fst()) * valToINT(snd()))
		 handle Prod => RAISE CoreDynObject.exProd
		   )
      |	LESS_INT =>
		valOfBOOL(valToINT(fst()) < valToINT(snd()))
      |	GREATER_INT => 
		valOfBOOL(valToINT(fst()) > valToINT(snd()))
      |	LESSEQ_INT =>
	        valOfBOOL(valToINT(fst()) <= valToINT(snd()))
      |	GREATEREQ_INT =>
	        valOfBOOL(valToINT(fst()) >= valToINT(snd()))
      |	ABS_REAL => 
	        (valOfREAL(abs(valToREAL(v)))
		 handle Abs => RAISE CoreDynObject.exAbs
		   )
      |	NEG_REAL =>
		(valOfREAL(~(valToREAL(v)))
		 handle Neg => RAISE CoreDynObject.exNeg
		   )
      |	SUM_REAL =>
		(valOfREAL(valToREAL(fst()) + valToREAL(snd()))
		 handle Sum => RAISE CoreDynObject.exSum
		   )
      |	DIFF_REAL =>
		(valOfREAL(valToREAL(fst()) - valToREAL(snd()))
		 handle Diff => RAISE CoreDynObject.exDiff
		   )
      |	PROD_REAL =>
		(valOfREAL(valToREAL(fst()) * valToREAL(snd()))
		 handle Prod => RAISE CoreDynObject.exProd
		   )
      |	LESS_REAL =>
		valOfBOOL(valToREAL(fst()) < valToREAL(snd()))
      |	GREATER_REAL =>
		valOfBOOL(valToREAL(fst()) > valToREAL(snd()))
      |	LESSEQ_REAL =>
	        valOfBOOL(valToREAL(fst()) <= valToREAL(snd()))
      |	GREATEREQ_REAL => 
	        valOfBOOL(valToREAL(fst()) >= valToREAL(snd()))
      | PRIM => 
	  let
	    val opcode = nth(v, 1)
	    val theArg = nth(v, 2)		(* single argument *)
	    fun arg n = nth(theArg, n)	(* several arguments *)
	    fun fst() = arg 1 and snd() = arg 2 and thd() = arg 3
	  in
	    case valToINT opcode
	      of  0 (* op = *) =>
		valOfBOOL(equal(fst(), snd()))

	    |  1 (* floor *) => Crash.unimplemented "floor"
	    |  2 (* real *) => Crash.unimplemented "real"
	    |  3 (* sqrt *) => Crash.unimplemented "sqrt"
	    |  4 (* sin *) => Crash.unimplemented "sin"
	    |  5 (* cos *) => Crash.unimplemented "cos"
	    |  6 (* arctan *) => Crash.unimplemented "arctan"
	    |  7 (* exp *) => Crash.unimplemented "exp"
	    |  8 (* ln *) => Crash.unimplemented "ln"
		 
	    |  9 (* size *) =>
		valOfINT(size(valToSTRING theArg))
		
	    | 10 (* chr *) =>
		let
		  val i = valToINT(fst())
		in
		  valOfSTRING(chr i)
		  handle Chr => RAISE(snd())
		end
	      
	    | 11 (* ord *) =>
		let
		  val s = valToSTRING(fst())
		in
		  valOfINT(ord s)
		  handle Ord => RAISE(snd())
		end
	      
	    | 12 (* explode *) =>
		let
		  val s = valToSTRING theArg
		  val l = explode s
		  fun f str value = CONS(valOfSTRING str, value)
		in
		  List.foldR f val_NIL l
		end
	      
	    | 13 (* implode *) =>
		let
		  val l = unwrapList theArg
		  val l' = map valToSTRING l
		in
		  valOfSTRING(implode l')
		end
	      
	    | 14 (* op / *) =>
		(valOfREAL(valToREAL(fst()) / valToREAL(snd()))
		 handle Quot => RAISE(thd())
		   )
		
	    | 15 (* op div *) =>
		(valOfINT(valToINT(fst()) div valToINT(snd()))
		 handle Div => RAISE(thd())
		   )
		
	    | 16 (* op mod *) =>
		(valOfINT(valToINT(fst()) mod valToINT(snd()))
		 handle Mod => RAISE(thd())
		   )
		
	    | 17 (* op := *) =>
		let
		  val addr = valToADDR(fst())
		  val new = snd()
		in
		  CoreDynObject.Store.add(addr, new)
		end
	      
	    | 18 (* ! *) =>
		CoreDynObject.Store.retrieve(valToADDR theArg)
		
		
	    | 19 (* open_in *) =>
		let
		  val file = valToSTRING(fst())
		    
		  val (id, theStreams') =
		    IOStreams.openIn (!theStreams) (file, fn () => RAISE(snd()))
		in
		  theStreams := theStreams';
		  valOfINT id
		end
	      
	    | 20 (* open_out *) =>
		let
		  val file = valToSTRING(fst())
		    
		  val (id, theStreams') =
		    IOStreams.openOut (!theStreams) (file, fn () => RAISE(snd()))
		in
		  theStreams := theStreams';
		  valOfINT id
		end
	      
	    | 21 (* input *) =>
		let
		  val id = valToINT(fst())
		  val n = valToINT(snd())
		  val stream = IOStreams.inputStream (!theStreams) id
		in
		  valOfSTRING(input(stream, n))
		end
	      
	    | 22 (* lookahead *) =>
		let
		  val id = valToINT theArg
		  val stream = IOStreams.inputStream (!theStreams) id
		in
		  valOfSTRING(lookahead stream)
		end
	      
	    | 23 (* close_in *) =>
		let
		  val id = valToINT theArg
		  val stream = IOStreams.inputStream (!theStreams) id
		in
		  close_in stream;
		  val_UNIT
		end
	      
	    | 24 (* end_of_stream *) =>
		let
		  val id = valToINT theArg
		  val stream = IOStreams.inputStream (!theStreams) id
		in
		  valOfBOOL(end_of_stream stream)
		end
	      
	    | 25 (* output *) =>
		let
		  val id = valToINT(fst())
		  val stream = IOStreams.outputStream (!theStreams) id
		  val text = valToSTRING(snd())
		  val exn = thd()
		in
		  (output(stream, text); val_UNIT)
		  handle Io _ => RAISE exn
		end
	      
	    | 26 (* close_out *) =>
		let
		  val id = valToINT theArg
		  val stream = IOStreams.outputStream (!theStreams) id
		in
		  close_out stream;
		  val_UNIT
		end
	      
	    | 27 (* use *) =>
		((USE(valToSTRING(fst())); val_UNIT)
		 handle Io _ => RAISE(snd())	(* snd arg is Io exception. *)
		      | Use => RAISE(thd())	(* thd arg is Use exception. *)
			  )
		
	    | 28 (* flush_out --- XXX NON STANDARD ML *) =>
		let 
		  val id = valToINT theArg
		  val stream = IOStreams.outputStream (!theStreams) id
		in
		  NonStandard.flush_out stream;
		  val_UNIT
		end
	      
	    | n => Crash.impossible("Apply.APPLY(#" ^ Int.string n ^ ")")
	  end

      |	_ => Crash.impossible("Apply.APPLY(Unresolved basic value)")
    end
  end;

