(*
$File: Compiler/CompilerEnv.sml $
$Date: 1992/09/18 18:04:01 $
$Revision: 1.2 $
$Locker:  $
*)

(*$CompilerEnv:
	VAR EXCON LVARS FINMAP PRETTYPRINT CRASH COMPILER_ENV
 *)

functor CompilerEnv(structure Var: VAR
		    structure Excon: EXCON
		    structure Lvars: LVARS
		    structure FinMap: FINMAP
		    structure PP: PRETTYPRINT
		      sharing type FinMap.StringTree = PP.StringTree

		    structure Crash: CRASH
		   ): COMPILER_ENV =
  struct
    type var = Var.var
    type longvar = Var.longvar
    type excon = Excon.excon
    type longexcon = Excon.longexcon
    type lvar = Lvars.lvar

    datatype result = LVAR of lvar
                    | ABS | NEG | PLUS | MINUS | MUL | LESS 
		    | GREATER | LESSEQ | GREATEREQ
		                (* ABS .. GREATEREQ are place-holders for 
				   the built-in overloaded operators 
				   The compiler must turn these into the
				   corresponding PRIM_APP(n, ...) in
				   the lambda language *)
                    | PRIM	(* `PRIM' is a place-holder for the built-in
				   prim function: int * 'a -> 'b. The compiler
				   must turn this, plus its constant argument,
				   into a PRIM_APP(n, ...) in the lambda
				   language. *)

   (* the representation of the environment is from strings to results,
      so that we can look up either variables or exceptions. We should
      be doing this correctly in context in CompileDec, since the
      elaborator has identified the mode of all the identifiers. *)

    type CEnv = (string, result) FinMap.map

    val emptyCEnv = FinMap.empty

    local 
      fun f (v, r) (cenv: CEnv) = FinMap.add(Var.pr_var v, r, cenv)
      open Var
    in
      val initialCEnv =
	List.foldL f emptyCEnv
	  [(var_PRIM, PRIM),
	   (var_ABS, ABS),
	   (var_NEG, NEG),
	   (var_PLUS, PLUS),
	   (var_MINUS, MINUS),
	   (var_MUL, MUL),
	   (var_LESS, LESS),
	   (var_GREATER, GREATER),
	   (var_LESSEQ, LESSEQ),
	   (var_GREATEREQ, GREATEREQ)]

    end
    

    fun declareVar(var, lv, ce) =
      FinMap.add(Var.pr_var var, LVAR lv, ce)

    fun declareExcon(excon, lv, ce) =
      FinMap.add(Excon.pr_excon excon, LVAR lv, ce)

    val plus = FinMap.plus

   (* lookupId needed to report top-level bindings. The other lookup functions
      are needed when compiling. *)

    type id = Var.id

    fun lookupId ce id =
      let
	val var = Var.mkVar id
      in
	case FinMap.lookup ce (Var.pr_var var)
	  of Some(LVAR lv) => lv

	   | Some PRIM =>
	       Crash.impossible("CompilerEnv.lookupId("
				^ Var.pr_var var ^ ") = PRIM"
			       )

	   | None =>
	       Crash.impossible("CompilerEnv.lookupId("
				^ Var.pr_var var ^ ") failed"
			       )
      end

    fun lookupLongvar ce longvar =
      case Var.decompose longvar
	of (nil, var) =>
	     (case FinMap.lookup ce (Var.pr_var var)
		of Some lv => lv
		 | None =>
		     Crash.impossible("CompilerEnv.lookupLongvar("
				      ^ Var.pr_var var ^ ")"
				     )
	     )

	 | _ => Crash.impossible("CompilerEnv.lookupLongvar(long: "
				 ^ Var.pr_longvar longvar ^ ")"
				)

   (* lookupExcon (as opposed to lookupLongexcon) is temporary, for looking
      up Match and Bind. *)

    fun lookupExcon ce excon =
      case FinMap.lookup ce (Excon.pr_excon excon)
	of Some(LVAR lv) => lv

	 | Some PRIM =>
	     Crash.impossible "CompilerEnv.lookupLongexcon: PRIM"

	 | None =>
	     Crash.impossible("CompilerEnv.lookupExcon "
			      ^ Excon.pr_excon excon
			     )

    fun lookupLongexcon ce longexcon =
      case Excon.decompose longexcon
	of (nil, excon) => lookupExcon ce excon
         | _ => Crash.impossible("CompilerEnv.lookupLongexcon(long: "
				 ^ Excon.pr_longexcon longexcon ^ ")"
				)

   (* lvarsOfCEnv is needed so that we can build the map from lvars to
      runtime objects (the dynamic env); see CompileAndRun. It's illegal
      to find an identifier mapping to a PRIM here. *)

    fun lvarsOfCEnv ce =
      FinMap.Fold (fn ((_, LVAR lv), rest) => lv :: rest
		    | ((_, PRIM), _) => Crash.impossible "lvarsOfCEnv"
		  ) nil ce

    type StringTree = FinMap.StringTree

    val layoutCEnv =
      FinMap.layoutMap
        {start="{", eq=" -> ", sep=", ", finish="}"}
	PP.LEAF
	(PP.layoutAtom(fn LVAR lv => Lvars.pr_lvar lv
			| PRIM => "PRIM"
		      )
	)
  end;
