(*
$File: Common/KitCompiler.sml $
$Date: 1993/03/05 14:38:40 $
$Revision: 1.74 $
$Locker: birkedal $
*)

(* Entire Kit Compiler system, linked together using functors only. The
   top-level modules are functors without arguments. *)

(*$TOOLS:
	BASIC_IO FINMAP SORTED_FINMAP PRETTYPRINT
	REPORT TIMESTAMP FLAGS LIST_HACKS CRASH
 *)

signature TOOLS =
  sig
    structure BasicIO: BASIC_IO
    structure FinMap: FINMAP
    structure SortedFinMap: SORTED_FINMAP

    structure PrettyPrint: PRETTYPRINT
      sharing type FinMap.StringTree
		   = SortedFinMap.StringTree
		   = PrettyPrint.StringTree

    structure Report: REPORT
      sharing type FinMap.Report
		   = SortedFinMap.Report
		   = PrettyPrint.Report
		   = Report.Report

    structure Timestamp: TIMESTAMP
    structure Flags: FLAGS
    structure ListHacks: LIST_HACKS
    structure Crash: CRASH
  end;

(*$Tools:
	BasicIO Crash Flags PrettyPrint
	FinMap SortedFinMap ListHacks Report Timestamp TOOLS
 *)

functor Tools(): TOOLS =
  struct
    structure BasicIO = BasicIO()
    structure Crash = Crash(structure BasicIO = BasicIO)
    structure Flags = Flags()
    structure Report = Report(structure BasicIO = BasicIO)
    structure Timestamp = Timestamp()

    structure PrettyPrint = PrettyPrint(structure Report = Report
					structure Crash = Crash
				       )

    structure FinMap = FinMap(structure Report = Report
			      structure PP = PrettyPrint
			     )

    structure SortedFinMap = SortedFinMap(structure Report = Report
					  structure PP = PrettyPrint
					 )

    structure ListHacks = ListHacks()
  end;

(*$ALL_INFO: DF_INFO ERROR_INFO SOURCE_INFO TYPE_INFO OVERLOADING_INFO GRAMMAR_INFO*)

signature ALL_INFO =
  sig
    structure DFInfo          : DF_INFO
    structure ErrorInfo       : ERROR_INFO
    structure SourceInfo      : SOURCE_INFO
    structure TypeInfo        : TYPE_INFO
    structure OverloadingInfo : OVERLOADING_INFO

    structure GrammarInfo : GRAMMAR_INFO
      sharing type DFInfo.info          = GrammarInfo.DFInfo
	  and type ErrorInfo.info       = GrammarInfo.ErrorInfo
	  and type SourceInfo.info      = GrammarInfo.SourceInfo
	  and type TypeInfo.info        = GrammarInfo.TypeInfo
	  and type OverloadingInfo.info = GrammarInfo.OverloadingInfo

	  and type DFInfo.StringTree
	           = SourceInfo.StringTree = TypeInfo.StringTree
		   = OverloadingInfo.StringTree = GrammarInfo.StringTree
  end;

(*$BASICS:
	TOOLS VAR CON SCON EXCON LAB TYNAME TYCON STATOBJECT
	STATOBJECT_PROP IDENT STRID SIGID FUNID
	TYVAR LEX_BASICS ALL_INFO
 *)

signature BASICS =
  sig
    structure Tools : TOOLS 
    structure StrId : STRID

    structure Ident : IDENT
      sharing type Ident.strid = StrId.strid

    structure Var : VAR
      sharing type Var.id     = Ident.id
	  and type Var.longid = Ident.longid
	  and type Var.strid  = StrId.strid

    structure Con : CON
      sharing type Con.longid = Ident.longid
	  and type Con.id     = Ident.id
	  and type Con.strid  = StrId.strid

    structure SCon : SCON

    structure Excon : EXCON
      sharing type Excon.longid = Ident.longid
	  and type Excon.id     = Ident.id
	  and type Excon.strid  = StrId.strid

    structure Lab : LAB
    structure TyVar : TYVAR

    structure TyCon : TYCON
      sharing type TyCon.strid = StrId.strid

    structure TyName : TYNAME
      sharing type TyName.tycon = TyCon.tycon

    structure StatObject : STATOBJECT
      sharing type StatObject.SyntaxTyVar = TyVar.SyntaxTyVar
	  and type StatObject.StringTree  = Tools.PrettyPrint.StringTree
	  and type StatObject.longexcon = Excon.longexcon
	  and type StatObject.longvar   = Var.longvar
	  and type StatObject.longcon   = Con.longcon
	  and type StatObject.longid    = Ident.longid
	  and type StatObject.TyName    = TyName.TyName
          and type StatObject.tycon     = TyCon.tycon
          and type StatObject.strid     = StrId.strid
	  and type StatObject.scon      = SCon.scon
	  and type StatObject.lab       = Lab.lab
	  and type StatObject.var       = Var.var
	  and type StatObject.id        = Ident.id

    structure StatObjectProp : STATOBJECT_PROP
      sharing type StatObjectProp.SyntaxTyVar  = TyVar.SyntaxTyVar
	  and type StatObjectProp.StringTree   = Tools.PrettyPrint.StringTree
	  and type StatObjectProp.TypeScheme   = StatObject.TypeScheme
	  and type StatObjectProp.Type         = StatObject.Type
	  and type StatObjectProp.Substitution = StatObject.Substitution
	  and type StatObjectProp.TyVar        = StatObject.TyVar
	  and type StatObjectProp.TypeFcn      = StatObject.TypeFcn
	  and type StatObjectProp.TyName       = TyName.TyName
	  and type StatObjectProp.var          = Var.var
	  and type StatObjectProp.id           = Ident.id

    structure SigId : SIGID
    structure FunId : FUNID

    structure LexBasics: LEX_BASICS
      sharing type LexBasics.Report = Tools.Report.Report

    structure AllInfo : ALL_INFO
      sharing type AllInfo.TypeInfo.longcon = Con.longcon
	  and type AllInfo.TypeInfo.lab = Lab.lab
	  and type AllInfo.ErrorInfo.Type = StatObject.Type
	  and type AllInfo.ErrorInfo.TyVar = StatObject.TyVar
	  and type AllInfo.ErrorInfo.TyName = TyName.TyName
	  and type AllInfo.ErrorInfo.TypeFcn = StatObject.TypeFcn
	  and type AllInfo.ErrorInfo.con = Con.con
	  and type AllInfo.ErrorInfo.lab = Lab.lab
	  and type AllInfo.ErrorInfo.excon = Excon.excon
	  and type AllInfo.ErrorInfo.tycon = TyCon.tycon
	  and type AllInfo.ErrorInfo.longid = StatObject.longid
	  and type AllInfo.ErrorInfo.longtycon = TyCon.longtycon
	  and type AllInfo.ErrorInfo.strid = StrId.strid
	  and type AllInfo.ErrorInfo.longstrid = StrId.longstrid
	  and type AllInfo.ErrorInfo.sigid = SigId.sigid
	  and type AllInfo.ErrorInfo.funid = FunId.funid
	  and type AllInfo.ErrorInfo.id = StatObjectProp.id
	  and type AllInfo.SourceInfo.pos = LexBasics.pos
	  and type AllInfo.GrammarInfo.StringTree = Tools.PrettyPrint.StringTree
          and type AllInfo.OverloadingInfo.Type = StatObject.Type
	  and type AllInfo.OverloadingInfo.StringTree = Tools.PrettyPrint.StringTree
  end;

(*$Basics:
	TOOLS Ident StrId SigId FunId TyVar Var Con SCon Excon Lab
	TyName TyCon StatObject STATOBJECT STATOBJECT_PROP LexBasics DFInfo
	SourceInfo ErrorInfo TypeInfo GrammarInfo OverloadingInfo BASICS
 *)

functor Basics(structure Tools: TOOLS): BASICS =
  struct
    structure Tools = Tools

    structure StrId = StrId(structure Timestamp = Tools.Timestamp
			    structure Crash = Tools.Crash
			   )

    structure Ident = Ident(structure StrId = StrId
			    structure Timestamp = Tools.Timestamp
			    structure Crash = Tools.Crash
			   )

    structure SigId = SigId()
          and FunId = FunId()
          and TyVar = TyVar(structure Crash = Tools.Crash)
	  and Lab = Lab()
    	  and SCon = SCon()
    	  and Var = Var(structure Ident = Ident)
    	  and Con = Con(structure Ident = Ident)
    	  and Excon = Excon(structure Ident = Ident)
    	  and TyCon = TyCon(structure StrId = StrId
			    structure Crash = Tools.Crash
			   )

    structure TyName = TyName(structure TyCon = TyCon
			      structure Timestamp = Tools.Timestamp
			      structure Flags = Tools.Flags
			     )

    local
      structure Unconstrained =
	StatObject(structure SortedFinMap  = Tools.SortedFinMap
		   structure Ident = Ident
		   structure Var = Var
		   structure Con = Con
		   structure Excon = Excon
		   structure Lab = Lab
		   structure SCon = SCon
		   structure TyName = TyName
		   structure TyCon = TyCon
		   structure TyVar = TyVar
		   structure Timestamp = Tools.Timestamp
		   structure ListHacks = Tools.ListHacks
		   structure Flags = Tools.Flags
		   structure Report = Tools.Report
		   structure PP = Tools.PrettyPrint
		   structure Crash = Tools.Crash
		  )
    in
      structure StatObject: STATOBJECT = Unconstrained
      structure StatObjectProp: STATOBJECT_PROP = Unconstrained
    end

   (* LexBasics is needed by SourceInfo, as well as all the parsing
      stuff. *)

    structure LexBasics = LexBasics(structure BasicIO = Tools.BasicIO
				    structure Report = Tools.Report
				    structure PP = Tools.PrettyPrint
				    structure Flags = Tools.Flags
				    structure Crash = Tools.Crash
				   )

    structure AllInfo =
      struct
	structure DFInfo = DFInfo(structure PP = Tools.PrettyPrint)

	structure SourceInfo = SourceInfo(structure LexBasics = LexBasics
					  structure PP = Tools.PrettyPrint
					  structure Crash = Tools.Crash
					 )

	structure ErrorInfo =
	  ErrorInfo(structure StatObjectProp = StatObjectProp
		    structure Lab   = Lab
		    structure Con   = Con
		    structure Excon = Excon
		    structure TyCon = TyCon
		    structure SigId = SigId
		    structure StrId = StrId
		    structure FunId = FunId
		    structure Report = Tools.Report
		   )

	structure TypeInfo =
	  TypeInfo(type lab = Lab.lab
		   type longcon = Con.longcon
		   structure PP = Tools.PrettyPrint
		  )

	structure OverloadingInfo =
	  OverloadingInfo(structure StatObjectProp = StatObjectProp
			  structure PP = Tools.PrettyPrint
			 )

	structure GrammarInfo = 
	  GrammarInfo(structure SourceInfo = SourceInfo
		      structure ErrorInfo = ErrorInfo
		      structure TypeInfo = TypeInfo
		      structure OverloadingInfo = OverloadingInfo
		      structure DFInfo = DFInfo
		      structure PP = Tools.PrettyPrint
		      structure Crash = Tools.Crash
		     )

      end
  end;

(*$TOPDEC_PARSING:
	BASICS DEC_GRAMMAR TOPDEC_GRAMMAR PPDECGRAMMAR PPTOPDECGRAMMAR
	INFIX_BASIS PARSE
 *)

signature TOPDEC_PARSING =
  sig
    structure Basics: BASICS
    structure PreElabDecGrammar: DEC_GRAMMAR
      sharing type PreElabDecGrammar.longid = Basics.Ident.longid

    structure PreElabTopdecGrammar: TOPDEC_GRAMMAR
      sharing type PreElabTopdecGrammar.dec = PreElabDecGrammar.dec
	  and type PreElabTopdecGrammar.con = PreElabDecGrammar.con
	    	   = Basics.Con.con
	  and type PreElabTopdecGrammar.excon = PreElabDecGrammar.excon
	    	   = Basics.Excon.excon
	  and type PreElabTopdecGrammar.tycon = PreElabDecGrammar.tycon
		   = Basics.TyCon.tycon
	  and type PreElabTopdecGrammar.tyvar = PreElabDecGrammar.tyvar
		   = Basics.TyVar.SyntaxTyVar
	  and type PreElabDecGrammar.scon = Basics.SCon.scon
	  and type PreElabDecGrammar.lab = Basics.Lab.lab
	  and type PreElabTopdecGrammar.id = PreElabDecGrammar.id
	    	   = Basics.Ident.id

	  and type PreElabTopdecGrammar.ty = PreElabDecGrammar.ty
	  and type PreElabTopdecGrammar.info = PreElabDecGrammar.info
		   = Basics.AllInfo.GrammarInfo.PreElabGrammarInfo
	  and type PreElabTopdecGrammar.sigid = Basics.SigId.sigid
	  and type PreElabTopdecGrammar.strid = Basics.StrId.strid
	  and type PreElabTopdecGrammar.funid = Basics.FunId.funid
	  and type PreElabTopdecGrammar.longstrid = PreElabDecGrammar.longstrid
	    	   = Basics.StrId.longstrid
	  and type PreElabTopdecGrammar.longtycon = PreElabDecGrammar.longtycon
	    	   = Basics.TyCon.longtycon

    structure PPPreElabDecGrammar: PPDECGRAMMAR
      sharing PPPreElabDecGrammar.G = PreElabDecGrammar
	  and type PPPreElabDecGrammar.StringTree
	    	   = Basics.Tools.PrettyPrint.StringTree

    structure PPPreElabTopdecGrammar : PPTOPDECGRAMMAR
      sharing PPPreElabTopdecGrammar.G = PreElabTopdecGrammar
          and type PPPreElabTopdecGrammar.StringTree
	           = Basics.Tools.PrettyPrint.StringTree

    structure InfixBasis: INFIX_BASIS

    structure Parse: PARSE
      sharing type Parse.topdec = PreElabTopdecGrammar.topdec
	  and type Parse.InfixBasis = InfixBasis.Basis
  end;

(*$TopdecParsing:
	BASICS DecGrammar TopdecGrammar PPDecGrammar
	PPTopdecGrammar InfixBasis Parse TOPDEC_PARSING
 *)

functor TopdecParsing(structure Basics: BASICS): TOPDEC_PARSING =
  struct
    structure Basics = Basics
    structure Tools = Basics.Tools
    structure AllInfo = Basics.AllInfo

    structure PreElabDecGrammar =
      DecGrammar(structure GrammarInfo =
		   struct
		     type GrammarInfo = AllInfo.GrammarInfo.PreElabGrammarInfo
		   end

		 structure Lab = Basics.Lab
		 structure SCon = Basics.SCon
		 structure Con = Basics.Con
		 structure TyVar = Basics.TyVar
		 structure TyCon = Basics.TyCon
		 structure Excon = Basics.Excon
		 structure StrId = Basics.StrId

		 structure Ident = Basics.Ident
		 structure Id = Basics.Ident
		)

    structure PreElabTopdecGrammar =
      TopdecGrammar(structure DecGrammar = PreElabDecGrammar

		    type strid = Basics.StrId.strid
		    type longstrid = Basics.StrId.longstrid
		    type funid = Basics.FunId.funid
		    type sigid = Basics.SigId.sigid
		    type id = Basics.Ident.id
		    type tyvar = Basics.TyVar.SyntaxTyVar
		    type tycon = Basics.TyCon.tycon
		    type longtycon = Basics.TyCon.longtycon
		    type con = Basics.Con.con
		    type excon = Basics.Excon.excon
		    type GrammarInfo = AllInfo.GrammarInfo.PreElabGrammarInfo

		    structure WithInfo = PreElabDecGrammar
					(* Provide `WithInfo' datatype. *)
		   )

    structure PPPreElabDecGrammar =
      PPDecGrammar(structure DecGrammar = PreElabDecGrammar
		   structure SCon = Basics.SCon
		   structure Lab = Basics.Lab
		   structure Con = Basics.Con
		   structure Excon = Basics.Excon
		   structure Ident = Basics.Ident
		   structure Id = Basics.Ident
		   structure TyVar = Basics.TyVar
		   structure TyCon = Basics.TyCon
		   structure StrId = Basics.StrId
		   structure PP = Tools.PrettyPrint
		   structure Crash = Tools.Crash
		  )

    structure PPPreElabTopdecGrammar =
      PPTopdecGrammar(structure TopdecGrammar = PreElabTopdecGrammar
		      structure StrId = Basics.StrId
		      structure FunId = Basics.FunId
		      structure SigId = Basics.SigId
		      structure Id = Basics.Ident
		      structure Con = Basics.Con
		      structure Excon = Basics.Excon
		      structure TyCon = Basics.TyCon
		      structure PP = Tools.PrettyPrint
		      structure PPDecGrammar = PPPreElabDecGrammar
		     )

    structure InfixBasis = InfixBasis(structure Ident = Basics.Ident
				      structure FinMap = Tools.FinMap
				      structure Report = Tools.Report
				      structure PP = Tools.PrettyPrint
				     )

    structure Parse =
      Parse(structure TopdecGrammar = PreElabTopdecGrammar
	    structure DecGrammar = PreElabDecGrammar
	    structure Con = Basics.Con
	    structure Excon = Basics.Excon
	    structure SCon = Basics.SCon
	    structure Lab = Basics.Lab
	    structure TyCon = Basics.TyCon
	    structure TyVar = Basics.TyVar
	    structure Ident = Basics.Ident
	    structure StrId = Basics.StrId
	    structure FunId = Basics.FunId
	    structure SigId = Basics.SigId
	    structure LexBasics = Basics.LexBasics
	    structure GrammarInfo = AllInfo.GrammarInfo
	    structure SourceInfo = AllInfo.SourceInfo
	    structure DFInfo = AllInfo.DFInfo
	    structure InfixBasis = InfixBasis
	    structure PPDecGrammar = PPPreElabDecGrammar
	    structure Report = Tools.Report
	    structure PP = Tools.PrettyPrint
	    structure FinMap = Tools.FinMap
	    structure BasicIO = Tools.BasicIO
	    structure Flags = Tools.Flags
	    structure Crash = Tools.Crash
	   )
  end;

(*$STATIC_OBJECTS:
      BASICS ENVIRONMENTS ENVIRONMENTS_PROP MODULE_STATOBJECT
      MODULE_ENVIRONMENTS
*)

signature STATIC_OBJECTS =
  sig
    structure Basics : BASICS

    structure Environments : ENVIRONMENTS
      sharing type Environments.Type         = Basics.StatObject.Type
          and type Environments.TyVar        = Basics.StatObject.TyVar
	  and type Environments.TypeScheme   = Basics.StatObject.TypeScheme
	  and type Environments.TypeFcn      = Basics.StatObject.TypeFcn
	  and type Environments.id           = Basics.Ident.id
	  and type Environments.con          = Basics.Con.con
	  and type Environments.longid       = Basics.Ident.longid
	  and type Environments.excon        = Basics.Excon.excon
	  and type Environments.Substitution = Basics.StatObject.Substitution
	  and type Environments.tycon        = Basics.TyCon.tycon
	  and type Environments.longtycon    = Basics.TyCon.longtycon
	  and type Environments.longstrid    = Basics.StrId.longstrid
	  and type Environments.StringTree   = Basics.Tools.PrettyPrint.StringTree

    structure EnvironmentsProp : ENVIRONMENTS_PROP
      sharing type EnvironmentsProp.TypeFcn = Environments.TypeFcn
	  and type EnvironmentsProp.TyStr   = Environments.TyStr
	  and type EnvironmentsProp.Context = Environments.Context
	  and type EnvironmentsProp.Env     = Environments.Env
	  and type EnvironmentsProp.con     = Basics.Con.con
	  and type EnvironmentsProp.strid   = Basics.StrId.strid
	  and type EnvironmentsProp.TyName  = Basics.TyName.TyName
	  and type EnvironmentsProp.StringTree =
	    Basics.Tools.PrettyPrint.StringTree

    structure ModuleStatObject : MODULE_STATOBJECT
      sharing type ModuleStatObject.SyntaxTyVar = Basics.TyVar.SyntaxTyVar
	  and type ModuleStatObject.longstrid = Basics.StrId.longstrid
	  and type ModuleStatObject.StrNameSet = EnvironmentsProp.StrNameSet
	  and type ModuleStatObject.StrName = EnvironmentsProp.StrName
	  and type ModuleStatObject.NameSet = EnvironmentsProp.NameSet
	  and type ModuleStatObject.TyName = Basics.TyName.TyName
	  and type ModuleStatObject.TypeFcn = Basics.AllInfo.ErrorInfo.TypeFcn
	  and type ModuleStatObject.Env = EnvironmentsProp.Env
	  and type ModuleStatObject.Str = EnvironmentsProp.Str
	  and type ModuleStatObject.TypeFcn = Basics.StatObjectProp.TypeFcn
	  and type ModuleStatObject.ConEnv = EnvironmentsProp.ConEnv
	  and type ModuleStatObject.Type = Basics.StatObjectProp.Type
	  and type ModuleStatObject.tycon = Basics.TyCon.tycon
	  and type ModuleStatObject.longtycon = Basics.TyCon.longtycon
	  and type ModuleStatObject.var = Basics.Var.var
	  and type ModuleStatObject.id = Basics.Ident.id
	  and type ModuleStatObject.StringTree
	           = Basics.Tools.PrettyPrint.StringTree
	  and type ModuleStatObject.ErrorInfo = Basics.AllInfo.ErrorInfo.info

    structure ModuleEnvironments : MODULE_ENVIRONMENTS
      sharing type ModuleEnvironments.Realisation = ModuleStatObject.Realisation
          and type ModuleEnvironments.longstrid = Basics.StrId.longstrid
          and type ModuleEnvironments.longtycon = Basics.TyCon.longtycon
	  and type ModuleEnvironments.TypeScheme = ModuleStatObject.TypeScheme
	  and type ModuleEnvironments.TypeFcn = ModuleStatObject.TypeFcn
	  and type ModuleEnvironments.NameSet = EnvironmentsProp.NameSet
	  and type ModuleEnvironments.Context = EnvironmentsProp.Context
	  and type ModuleEnvironments.StrName = EnvironmentsProp.StrName
	  and type ModuleEnvironments.ConEnv = EnvironmentsProp.ConEnv
	  and type ModuleEnvironments.FunSig = ModuleStatObject.FunSig
	  and type ModuleEnvironments.StrEnv = EnvironmentsProp.StrEnv
	  and type ModuleEnvironments.TyStr = ModuleStatObject.TyStr
	  and type ModuleEnvironments.TyVar = Basics.StatObject.TyVar
	  and type ModuleEnvironments.Type = ModuleStatObject.Type
          and type ModuleEnvironments.Tail = Basics.AllInfo.ErrorInfo.Tail
	  and type ModuleEnvironments.id = Basics.Ident.id
	  and type ModuleEnvironments.strid = Basics.StrId.strid
	  and type ModuleEnvironments.sigid = Basics.SigId.sigid
	  and type ModuleEnvironments.funid = Basics.FunId.funid
	  and type ModuleEnvironments.tycon = Basics.TyCon.tycon
	  and type ModuleEnvironments.excon = Basics.Excon.excon
	  and type ModuleEnvironments.Str = EnvironmentsProp.Str
	  and type ModuleEnvironments.Env = EnvironmentsProp.Env
	  and type ModuleEnvironments.Sig = ModuleStatObject.Sig
	  and type ModuleEnvironments.con = Basics.Con.con
	  and type ModuleEnvironments.id = Basics.Ident.id
	  and type ModuleEnvironments.StringTree
	    = Basics.Tools.PrettyPrint.StringTree
  end;

(*$StaticObjects:
	TOPDEC_PARSING Environments ENVIRONMENTS ENVIRONMENTS_PROP
	ModuleStatObject ModuleEnvironments STATIC_OBJECTS
 *)

functor StaticObjects(structure TopdecParsing: TOPDEC_PARSING
			sharing type TopdecParsing.PreElabDecGrammar.longid 
			              = TopdecParsing.Basics.Ident.longid)
	  : STATIC_OBJECTS =
  struct
    structure Basics = TopdecParsing.Basics
    structure Tools  = Basics.Tools

    local
      structure Unconstrained =
	Environments(structure DecGrammar = TopdecParsing.PreElabDecGrammar
		     structure Ident = Basics.Ident
		     structure Con = Basics.Con
		     structure Var = Basics.Var
		     structure Excon = Basics.Excon
		     structure TyCon = Basics.TyCon
		     structure StrId = Basics.StrId
		     structure StatObject = Basics.StatObjectProp
		     structure TyName = Basics.TyName
		     structure PP = Tools.PrettyPrint
		     structure SortedFinMap = Tools.SortedFinMap
		     structure FinMap = Tools.FinMap
		     structure Timestamp = Tools.Timestamp
		     structure Report = Tools.Report
		     structure Flags = Tools.Flags
		     structure ListHacks = Tools.ListHacks
		     structure Crash = Tools.Crash
		    )
    in
      structure Environments : ENVIRONMENTS = Unconstrained
      structure EnvironmentsProp : ENVIRONMENTS_PROP = Unconstrained
    end

    structure ModuleStatObject =
      ModuleStatObject(structure StrId      = Basics.StrId
		       structure SigId      = Basics.SigId
		       structure FunId      = Basics.FunId
		       structure TyCon      = Basics.TyCon
		       structure TyName     = Basics.TyName
		       structure StatObject = Basics.StatObjectProp
		       structure C          = EnvironmentsProp
		       structure ErrorInfo  = Basics.AllInfo.ErrorInfo
		       structure FinMap     = Tools.FinMap
		       structure PP         = Tools.PrettyPrint
		       structure Flags      = Tools.Flags
		       structure Crash      = Tools.Crash
		      )

    structure ModuleEnvironments =
      ModuleEnvironments(structure StrId             = Basics.StrId
			 structure SigId             = Basics.SigId
			 structure FunId             = Basics.FunId
			 structure TyCon             = Basics.TyCon
			 structure FinMap            = Tools.FinMap
			 structure StatObject        = Basics.StatObjectProp
			 structure C                 = EnvironmentsProp
			 structure ModuleStatObject  = ModuleStatObject
			 structure PP                = Tools.PrettyPrint
			 structure Report	     = Tools.Report
			 structure Flags             = Tools.Flags
			 structure ListHacks	     = Tools.ListHacks
			 structure Crash             = Tools.Crash

			 structure Tail_T = Basics.AllInfo.ErrorInfo
					(* Just so we can share type Tail. *)
			)
  end;

(*$ELABORATION:
	BASICS STATIC_OBJECTS ELABTOPDEC RESIDENT
	DEC_GRAMMAR PPDECGRAMMAR TOPDEC_GRAMMAR PPTOPDECGRAMMAR
 *)

signature ELABORATION =
  sig
    structure Basics : BASICS

    structure StaticObjects : STATIC_OBJECTS
      sharing StaticObjects.Basics = Basics

    structure ElabTopdec : ELABTOPDEC

    structure ResIdent : RESIDENT
      sharing type ResIdent.longvar = Basics.Var.longvar
	  and type ResIdent.longcon = Basics.Con.longcon
	  and type ResIdent.longexcon = Basics.Excon.longexcon

    structure PostElabDecGrammar : DEC_GRAMMAR
      sharing type PostElabDecGrammar.lab = Basics.Lab.lab
	  and type PostElabDecGrammar.scon = Basics.SCon.scon
	  and type PostElabDecGrammar.excon = Basics.Excon.excon
	  and type PostElabDecGrammar.con = Basics.Con.con
 	  and type PostElabDecGrammar.id = Basics.Var.var
 	  and type PostElabDecGrammar.longid = ResIdent.longid
	  and type PostElabDecGrammar.longstrid = Basics.StrId.longstrid
	  and type PostElabDecGrammar.info
		   = Basics.AllInfo.GrammarInfo.PostElabGrammarInfo

    structure PPPostElabDecGrammar : PPDECGRAMMAR
      sharing PPPostElabDecGrammar.G = PostElabDecGrammar
	  and type PPPostElabDecGrammar.StringTree
	    = Basics.Tools.PrettyPrint.StringTree

    structure PostElabTopdecGrammar : TOPDEC_GRAMMAR
      sharing type PostElabTopdecGrammar.id = Basics.Var.var
	  and type PostElabTopdecGrammar.excon = Basics.Excon.excon
	  and type PostElabTopdecGrammar.strid = Basics.StrId.strid
	  and type PostElabTopdecGrammar.longstrid = Basics.StrId.longstrid
	  and type PostElabTopdecGrammar.sigid = Basics.SigId.sigid
	  and type PostElabTopdecGrammar.funid = Basics.FunId.funid
	  and type PostElabTopdecGrammar.con = Basics.Con.con
	  and type PostElabTopdecGrammar.dec = PostElabDecGrammar.dec
	  and type PostElabTopdecGrammar.info
	    	   = Basics.AllInfo.GrammarInfo.PostElabGrammarInfo
	  and type ElabTopdec.PostElabTopdec = PostElabTopdecGrammar.topdec

    structure PPPostElabTopdecGrammar : PPTOPDECGRAMMAR
      sharing PPPostElabTopdecGrammar.G = PostElabTopdecGrammar
	  and type PPPostElabTopdecGrammar.StringTree
	    = Basics.Tools.PrettyPrint.StringTree
  end;

(*$Elaboration:
	TOPDEC_PARSING STATIC_OBJECTS ResIdent DecGrammar PPDecGrammar
	TopdecGrammar PPTopdecGrammar ElabTopdec ElabDec ModuleUnify
	ELABORATION
 *)

functor Elaboration(structure TopdecParsing : TOPDEC_PARSING
		     sharing type TopdecParsing.PreElabDecGrammar.lab 
		                  = TopdecParsing.Basics.Lab.lab
		    structure StaticObjects : STATIC_OBJECTS
		     sharing StaticObjects.Basics = TopdecParsing.Basics
		         and type StaticObjects.Environments.valbind
				  = TopdecParsing.PreElabDecGrammar.valbind
                         and type StaticObjects.ModuleStatObject.ty
				  = TopdecParsing.PreElabDecGrammar.ty
                         and type StaticObjects.Environments.pat
			          = TopdecParsing.PreElabDecGrammar.pat
			 and type StaticObjects.Environments.id
				  = TopdecParsing.PreElabDecGrammar.id
			 and type StaticObjects.Environments.longid
				  = TopdecParsing.PreElabDecGrammar.longid
				  = StaticObjects.Basics.Ident.longid 
			 and type StaticObjects.ModuleStatObject.TyVar 
			          = TopdecParsing.Basics.AllInfo.ErrorInfo.TyVar
		   ): ELABORATION =
  struct
    local
      structure Basics     = TopdecParsing.Basics
      structure Tools      = Basics.Tools
      structure AllInfo    = Basics.AllInfo
    in
      structure StaticObjects = StaticObjects
      open StaticObjects

      structure ResIdent = ResIdent(structure Ident = Basics.Ident
				    structure Var   = Basics.Var
				    structure Con   = Basics.Con
				    structure Excon = Basics.Excon
				   )

      structure ResId =
	struct
	  type id = Basics.Var.var
	  val pr_id = Basics.Var.pr_var
	end

      structure PostElabDecGrammar =
	DecGrammar(structure GrammarInfo =
		     struct
		       type GrammarInfo =
			 AllInfo.GrammarInfo.PostElabGrammarInfo
		     end

		   structure Lab         = Basics.Lab
		   structure SCon        = Basics.SCon
		   structure Con         = Basics.Con
		   structure TyVar       = Basics.TyVar
		   structure TyCon       = Basics.TyCon
		   structure Excon       = Basics.Excon		    
		   structure StrId       = Basics.StrId

		   structure Ident       = ResIdent
		   structure Id          = ResId
		  )

      structure PPPostElabDecGrammar =
	PPDecGrammar(structure DecGrammar = PostElabDecGrammar
		     structure SCon  = Basics.SCon
		     structure Lab   = Basics.Lab
		     structure Con   = Basics.Con
		     structure Excon = Basics.Excon
		     structure Ident = ResIdent
		     structure Id    = ResId
		     structure TyVar = Basics.TyVar
		     structure TyCon = Basics.TyCon
		     structure StrId = Basics.StrId
		     structure PP    = Tools.PrettyPrint
		     structure Crash = Tools.Crash
		    )

      structure PostElabTopdecGrammar =
	TopdecGrammar(structure DecGrammar =  PostElabDecGrammar

		      type strid     = Basics.StrId.strid
		      type longstrid = Basics.StrId.longstrid
		      type funid     = Basics.FunId.funid
		      type sigid     = Basics.SigId.sigid

		      type id        = Basics.Var.var
		      type tyvar     = Basics.TyVar.SyntaxTyVar
		      type tycon     = Basics.TyCon.tycon
		      type longtycon = Basics.TyCon.longtycon
		      type con       = Basics.Con.con
		      type excon     = Basics.Excon.excon
		      type GrammarInfo = AllInfo.GrammarInfo.PostElabGrammarInfo

		      structure WithInfo = PostElabDecGrammar
					(* Provide `WithInfo' datatype. *)
		     )

      structure Id =
	struct
	  type id = Basics.Var.var
	  val pr_id = Basics.Var.pr_var
	end

      structure PPPostElabTopdecGrammar =
	PPTopdecGrammar(structure TopdecGrammar = PostElabTopdecGrammar
			structure StrId = Basics.StrId
			structure FunId = Basics.FunId
			structure SigId = Basics.SigId
			structure Id    = Id
			structure Con   = Basics.Con
			structure Excon = Basics.Excon
			structure TyCon = Basics.TyCon
			structure PP    = Tools.PrettyPrint
			structure PPDecGrammar = PPPostElabDecGrammar
		       )

      structure ElabTopdec =
	ElabTopdec(structure IG = TopdecParsing.PreElabTopdecGrammar
		   structure OG = PostElabTopdecGrammar
		   structure ModuleStatObject   = ModuleStatObject
		   structure ModuleEnvironments = ModuleEnvironments

		   structure ElabDec =
		     ElabDec(structure IG = TopdecParsing.PreElabDecGrammar
			     structure OG = PostElabDecGrammar
			     structure R = ResIdent
			     structure Environments = Environments
			     structure Con = Basics.Con
			     structure Lab = Basics.Lab
			     structure Var = Basics.Var
			     structure StatObject = Basics.StatObject
			     structure ErrorInfo = AllInfo.ErrorInfo
			     structure SourceInfo = AllInfo.SourceInfo
			     structure TypeInfo = AllInfo.TypeInfo
			     structure OverloadingInfo = AllInfo.OverloadingInfo
			     structure GrammarInfo = AllInfo.GrammarInfo
			     structure PPInDecGrammar =
			       TopdecParsing.PPPreElabDecGrammar
			     structure Report = Tools.Report
			     structure PP = Tools.PrettyPrint
			     structure Flags = Tools.Flags
			     structure Crash = Tools.Crash
			    )

		   structure U =
		     ModuleUnify(structure StrId = Basics.StrId
				 structure TyCon = Basics.TyCon
				 structure FinMap = Tools.FinMap
				 structure O =
				   StaticObjects.ModuleStatObject
				 structure Env =
				   StaticObjects.ModuleEnvironments

				 structure ErrorInfo = AllInfo.ErrorInfo
				 structure ListHacks = Tools.ListHacks
				 structure Crash = Tools.Crash
				)

		   structure StrId = Basics.StrId
		   structure SigId = Basics.SigId
		   structure ErrorInfo = AllInfo.ErrorInfo
		   structure GrammarInfo = AllInfo.GrammarInfo
		   structure BasicIO = Tools.BasicIO
		   structure Crash = Tools.Crash
		  )
    end
  end;

(*$EXECUTION: ELABORATION DYNAMIC_BASIS EVALTOPDEC VAL_PRINT*)

signature EXECUTION =
  sig
    structure Elaboration: ELABORATION
    structure DynamicBasis: DYNAMIC_BASIS

    structure EvalTopdec: EVALTOPDEC
      sharing type EvalTopdec.DynamicBasis = DynamicBasis.Basis
	  and type EvalTopdec.topdec = Elaboration.ElabTopdec.PostElabTopdec

    structure ValPrint: VAL_PRINT
  end;

(*$Execution: ELABORATION Evaluation EXECUTION*)

functor Execution(structure Elaboration : ELABORATION
		  val USE: string -> unit
		 ) : EXECUTION =
  struct
    structure Elaboration = Elaboration
    structure ResIdent    = Elaboration.ResIdent
    structure Basics      = Elaboration.StaticObjects.Basics
    structure Tools       = Basics.Tools
    structure AllInfo     = Basics.AllInfo

    structure Evaluation =
      Evaluation(structure Lab = Basics.Lab
		 structure Con = Basics.Con
		 structure SCon = Basics.SCon
		 structure Excon = Basics.Excon
		 structure Var = Basics.Var
		 structure StrId = Basics.StrId
		 structure SigId = Basics.SigId
		 structure FunId = Basics.FunId
		 structure ResIdent = ResIdent
		 structure DecGrammar = Elaboration.PostElabDecGrammar
		 structure PPDecGrammar = Elaboration.PPPostElabDecGrammar
		 structure TopdecGrammar = Elaboration.PostElabTopdecGrammar
		 structure PPTopdecGrammar = Elaboration.PPPostElabTopdecGrammar
		 structure GrammarInfo = AllInfo.GrammarInfo
		 structure SourceInfo = AllInfo.SourceInfo
		 structure DFInfo = AllInfo.DFInfo
		 structure TypeInfo = AllInfo.TypeInfo
		 structure OverloadingInfo = AllInfo.OverloadingInfo
		 structure StatObject = Basics.StatObject
		 structure FinMap = Tools.FinMap
		 structure SortedFinMap = Tools.SortedFinMap
		 structure BasicIO = Tools.BasicIO
		 structure Report = Tools.Report
		 structure Flags = Tools.Flags
		 structure PP = Tools.PrettyPrint
		 structure Report = Tools.Report
		 structure Crash = Tools.Crash
		 val USE = USE
		)

    open Evaluation
  end;

(*$Linking:
	Tools Basics TopdecParsing StaticObjects Elaboration Execution
 *)

(* Note how USE is getting passed up all the way from Apply. We can only
   tie the knot properly once it's propagated to functor Interpret(). *)

functor Linking(val USE: string -> unit) =
  struct
    structure Tools   = Tools()
    structure Basics  = Basics(structure Tools = Tools)
    structure AllInfo = Basics.AllInfo
    structure TopdecParsing  = TopdecParsing(structure Basics = Basics)

    structure StaticObjects =
      StaticObjects(structure TopdecParsing = TopdecParsing)

    structure Elaboration =
      Elaboration(structure TopdecParsing = TopdecParsing
		  structure StaticObjects = StaticObjects
		 )

    structure Execution = Execution(structure Elaboration = Elaboration
				    val USE = USE
				   )
  end;

(* KitCompiler does the functor application to build up the
   heirarchy of structures, and builds the other stuff not directly
   relevant to the build heirarchy. It provides a convenient
   top-level interface. *)

(*$KitCompiler:
	Linking ErrorTraverse TopLevelReport Basis Interpreter
 *)

functor KitCompiler(val prelude: string)
  : sig
      val parse: unit -> unit
      val elab: unit -> unit
      val eval: unit -> unit

      val parseFile: string -> unit
      val elabFile: string -> unit
      val evalFile: string -> unit
    end = 
  struct
    local
     (* We tie the `use' knot with a reference. Cleaner than parameterising
        all the evaluation functions, I feel. And I'm in charge of the Kit. *)

      val useRef: (string -> unit) ref = ref(fn _ => ())

      structure Linking = Linking(fun USE s = (!useRef) s)
      open Linking

     (* Other arguments to Interpreter: *)
      structure Basis =
	Basis(structure InfixBasis = TopdecParsing.InfixBasis
	      structure ModuleEnvironments = StaticObjects.ModuleEnvironments
	      structure DynamicBasis = Execution.DynamicBasis
	     )

      structure Interpreter =
	Interpreter(structure Basis = Basis
		    structure Parse = TopdecParsing.Parse
		    structure ElabTopdec = Elaboration.ElabTopdec

		    structure PPPreElabTopdecGrammar =
		      TopdecParsing.PPPreElabTopdecGrammar

		    structure PPPostElabTopdecGrammar =
		      Elaboration.PPPostElabTopdecGrammar

		    structure EvalTopdec = Execution.EvalTopdec

		    structure ErrorTraverse =
		      ErrorTraverse(structure TopdecGrammar =
				      Elaboration.PostElabTopdecGrammar

				    structure DecGrammar =
				      Elaboration.PostElabDecGrammar

				    structure GrammarInfo = AllInfo.GrammarInfo
				    structure SourceInfo = AllInfo.SourceInfo
				    structure ErrorInfo = AllInfo.ErrorInfo
				    structure Report = Tools.Report
				    structure Crash = Tools.Crash
				   )

		    structure InfixBasis = TopdecParsing.InfixBasis

		    structure TopLevelReport =
		      TopLevelReport(structure FunId = Basics.FunId
				     structure SigId = Basics.SigId
				     structure StrId = Basics.StrId
				     structure Ident = Basics.Ident
				     structure Basis = Basis

				     structure InfixBasis =
				       TopdecParsing.InfixBasis

				     structure StatObjectProp =
				       Basics.StatObjectProp

				     structure EnvironmentsProp =
				       StaticObjects.EnvironmentsProp

				     structure ModuleStatObject =
				       StaticObjects.ModuleStatObject

				     structure ModuleEnvironments =
				      StaticObjects.ModuleEnvironments

				     structure ValPrint = Execution.ValPrint
				     structure Report = Tools.Report
				     structure Crash = Tools.Crash
				    )

		    structure BasicIO = Tools.BasicIO
		    structure Report = Tools.Report
		    structure PP = Tools.PrettyPrint
		    structure Flags = Tools.Flags
		    structure Crash = Tools.Crash
	)

      open Interpreter

      val _ = (useRef := interpretFile EVALUATE)
		(* Tie the knot for `use'. Note that nested evaluation is
		   always in EVALUATE mode. It must be, otherwise we'd never
		   have evaluated the `use' in the first place...! *)
    in
      val _ = interpretFile EVALUATE prelude

      val parse = interpretStdIn PARSE_ONLY
      val elab = interpretStdIn ELABORATE_ONLY
      val eval = interpretStdIn EVALUATE

      val parseFile = interpretFile PARSE_ONLY
      val elabFile = interpretFile ELABORATE_ONLY
      val evalFile = interpretFile EVALUATE
    end
  end;

(*$K: KitCompiler*)
structure K = KitCompiler(val prelude = "Common/Prelude.sml");
open K;
