/*******************************************************************************
+
+  LEDA  2.2.0                                                 03-05-1992
+
+
+  window.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef WINDOW_H
#define WINDOW_H

#include <LEDA/basic.h>
#include <LEDA/segment.h>

#ifndef __TURBOC__
#include <LEDA/plane.h>
#else
#include <LEDA/segment.h>
#endif


declare(list,string)

typedef double (*draw_func_ptr) (double);
typedef void   (*redraw_func_ptr) ();
typedef void   (*mouse_action_func_ptr) (double,double);

enum color        {white, black, red, green, blue, yellow, violet, orange};
enum line_style   {solid, dashed, dotted};
enum text_mode    {transparent, opaque};
enum drawing_mode {src_mode, xor_mode};



class window {

int state;

public:

 window(float width, float height, float xpos, float ypos);
 window(float width, float height);
 window();
~window();

void init(double x0,double x1,double y0,int gmode = 0);

operator int() { return state; }


void         set_redraw(redraw_func_ptr f);
bool         set_font(char* s);
void         set_grid_mode(int i);
line_style   set_line_style(line_style s) ;
int          set_line_width(int w);
drawing_mode set_mode(drawing_mode m);
int          set_node_width(int w);
text_mode    set_text_mode(text_mode m);
void         set_frame_label(string s);
void         reset_frame_label();
void         set_flush(bool b);


int          get_line_width();
int          get_node_width();
line_style   get_line_style();
text_mode    get_text_mode();
drawing_mode get_mode();
int          mono();

double xmin();
double xmax();
double ymin();
double ymax();
double scale();

int xpix(double x);
int ypix(double x);

double pixnum(int p);
double pixels(int p) { return pixnum(p); };

void flush();

// drawing a single pixel

void draw_pix(double x, double y, color c = black );
void draw_pix(point p, color c = black );



// drawing points

void draw_point(double x0,double y0,color c = black);
void draw_point(point p,color c = black);


// drawing nodes

void draw_node(double x0,double y0,color c = black) ;
void draw_node(point p, color c = black);
void draw_filled_node(double x0,double y0,color c = black);
void draw_filled_node(point p, color c = black);
void draw_text_node(double x,double y,string s,color c = black);
void draw_text_node(point p ,string s,color c = black);
void draw_int_node(double x,double y,int i,color c = black);
void draw_int_node(point p ,int i,color c = black);


// drawing segments

void draw_segment(double x1, double y1, double x2, double y2, color c = black );
void draw_segment(point p, point q, color c = black );
void draw_segment(segment s, color c = black );


// drawing arrows

void draw_arrow(double x1, double y1, double x2, double y2, color c = black );
void draw_arrow(point p, point q, color c = black );
void draw_arrow(segment s, color c = black );

// drawing edges

void draw_edge(double x1, double y1, double x2, double y2, color c = black );
void draw_edge(point p, point q, color c = black );
void draw_edge(segment s, color c = black );

void draw_edge_arrow(double x1, double y1, double x2, double y2, color c = black );
void draw_edge_arrow(point p, point q, color c = black );
void draw_edge_arrow(segment s, color c = black );


// drawing lines

void draw_hline(double y, color c = black );
void draw_vline(double x, color c = black );
void draw_line(double x1, double y1, double x2, double y2, color c = black );
void draw_line(point p, point q, color c=black) ;
void draw_line(segment s, color c=black) ;



//circles

void draw_circle(double x,double y,double r,color c = black);
void draw_circle(point p,double r,color c = black);

void draw_disc(double x,double y,double r,color c = black);
void draw_disc(point p,double r,color c = black);


//ellipses

void draw_ellipse(double x,double y,double a, double b, color c = black);
void draw_ellipse(point p, double a, double b, color c = black);


//drawing polygons 

void draw_polygon(list(point) lp, color c = black );
void draw_filled_polygon(list(point) lp, color c = black );



void draw_rectangle(double a, double  b, double c, double d, color col = black);
void draw_filled_rectangle(double a, double  b, double c, double d, color col = black);


// miscellaneous

void clear(color c = white);
void show_window();


void copy_rect(double x1, double y1, double x2, double y2, double x, double y) ;
void move_rect(double x1, double y1, double x2, double y2, double x, double y) ;
void copy(double x1, double y1, double x2, double y2, int i=0) ;
void cut(double x1, double y1, double x2, double y2, int i=0) ;
void paste(int i, double x, double y);
void paste(double x, double y);
void clear_buf(int i=0);


// text

void draw_text(double x, double y, string s, color c = black);
void draw_text(point p, string s, color c = black);
void draw_ctext(double x, double y, string s, color c = black);
void draw_ctext(point p, string s, color c = black);


// drawing functions

void plot_xy(double x0, double x1, draw_func_ptr f, color c = black);
void plot_yx(double y0, double y1, draw_func_ptr f, color c = black);



// mouse input

int read_mouse();
int get_button();

int read_mouse(double&, double&);
int read_mouse(point&);

int read_mouse_seg(double, double, double&, double&);
int read_mouse_seg(point, point&);

int read_mouse_rect(double, double, double&, double&);
int read_mouse_rect(point, point&);

int read_mouse_circle(double, double, double&, double&);
int read_mouse_circle(point, point&);

int read_mouse_action(mouse_action_func_ptr, double&, double&);
int read_mouse_action(mouse_action_func_ptr, point&);

int     confirm(string s);
void    acknowledge(string s);
void    notice(string s);

int     read_panel(string, int, string*);
int     read_vpanel(string, int, string*);

string  read_string(string);
double  read_real(string);
int     read_int(string);

void    message(string);
void    del_message();



// read, draw and I/O operators

window& read(point&);
window& draw(point p, color c = black);
window& operator>>(point&);
window& operator<<(point p)   { draw(p); return *this; }

window& read(segment&);
window& draw(segment s, color c = black);
window& operator>>(segment&);
window& operator<<(segment s) { draw(s); return *this; }


#ifndef __TURBOC__

// too much for Turbo C++

void draw_line(line l, color c=black);
void draw_disc(circle C,color c = black);
void draw_circle(circle C,color c = black);
void draw_polygon(polygon P, color c = black );
void draw_filled_polygon(polygon P,color c = black );


window& read(line&);
window& draw(line l, color c=black);
window& operator>>(line&);
window& operator<<(line l)    { draw(l); return *this; }

window& read(circle&);
window& draw(circle C,color c = black);
window& operator>>(circle&);
window& operator<<(circle C)  { draw(C); return *this; }

window& read(polygon&);
window& draw(polygon P, color c = black );
window& operator>>(polygon&);
window& operator<<(polygon P) { draw(P); return *this; }

#endif



}; // end of class window
  


//------------------------------------------------------------------------------
//   PANELS
//------------------------------------------------------------------------------


struct panel {

void* panel_ptr;

 panel();
 panel(string s);
~panel();


void label(string s);
void text_item(string s);

void string_item(string s, string& x);
void string_item(string label,string& x,list(string)& L);

void choice_item(string header,int& x,list(string)& L);
void choice_item(string header,int& x,string,string);
void choice_item(string header,int& x,string,string,string);
void choice_item(string header,int& x,string,string,string,string);
void choice_item(string header,int& x,string,string,string,string,string);

void int_item(string s,int& x);
void int_item(string s,int& x, int l, int h, int step);
void int_item(string s,int& x, int l, int h);

void bool_item(string s, int& x);
void real_item(string s, double& x);
void double_item(string s, double& x);

void color_item(string s, color& x);
void lstyle_item(string s, line_style& x);

void button(string s);

void new_button_line();
void new_button_line(list(string)&);

int  open();                      // center on screen
int  open(int x,int y);           // open at (x,y)

int  open(window& W);              // center on window
int  open(window& W, int x,int y);

int  open(list(string) buttons)   { new_button_line(buttons); return open(); }

};

#endif
