/**************************************************************
 * Send any comments or questions to: OTSO-Bug@tel.vtt.fi
 *
 * Name: /home/users/otso/official/otso/include/SCCS/s.group.hxx
 * Vers: 5.3    Time: 92/09/10, 16:01:09
 **************************************************************/

/***************************************************************
* Copyright (c) 1992      Technical Research Centre of Finland (VTT)
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that this notice and the reference to this notice appearing in each software
* module be retained unaltered, and that the name of any contributors shall not
* be used in advertising or publicity pertaining to distribution of the software
* without specific written prior permission.  No contributor makes any
* representations about the suitability of this software for any purpose.
* It is provided "as is" without any express or limited warranty.
*
*			NO WARRANTY
*
* ALL CONTRIBUTORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS.  IN NO
* EVENT SHALL ANY CONTRIBUTOR BE LIABLE FOR ANY SPECIAL, PUNITIVE, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA, OR PROFITS, WHETHER IN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE OR PERFORMANCE
* OF THIS SOFTWARE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THIS
* SOFTWARE IS WITH YOU.  SHOULD THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
* As used above, "contributor" includes, but is not limited to :
*        The Technical Research Centre of Finland
***************************************************************/


/**********************************************************************
* FILE
*	group.hxx
**********************************************************************/


typedef Runner* GE;	//Group element, must be a pointer
typedef void*   GP;	//Stores the current position in a Group for iteration,
                        //the actual type depends on derived Group class.

/*********************************************************************
Class Group is an abstract base class for Runner pointer containers.
The Group element type (GE) is Runner*.
Derived classes are specialized container classes
(such as Fifo, Lifo, Heap, Ring, Priorvec) that define which element
is taken from the Group next.
These derived classes must share basic properties
(defined virtually in the base class Group), so they can be used
interchangeably without major re-writes of user code that
uses a specific group :

(1) Uniform Container Contents : all contain Runner pointers,
which can of course point to any object derived from Runner. 

(2) Uniform Access of Container contents : put() and get().

(3) Iteration thru the elements of a group in a group-independent fashion:
.nf

    GP aid;	//GP is void*, stores the current position in Group
    Group* g = ...
    for (Runner* r = g->first(aid); !g->over(aid); r = g->next(aid)) {
        //...use r...
    }

.fi
Think twice before using iteration, put() and get() are the basic 
services provided by Groups.

Most of the derived Groups are "dynamic": 
put() increases the number of elements, and get() decreases it.
Ring, however, is an exception: put() does nothing, 
and get() gives a Runner* but does not extract it from the Ring.
Therefore there are "stronger" operations for inserting and extracting
elements to/from any kind of Group (Ring and all others):
putAnyway() and getAnyway().
These operations always increase/decrease the number of elements.

.SH BUGS
Probably we should have put for 1, 2, 3, etc. parameters, not
default parameters => put(&) instead of put(*) ???

Type-casting the pointer returned by get() is dangerous.  
Write a container class of your own or take a look
at class VoidGroup if you need Groups of other types than Runners.

If the number of elements is big, print() should print only the first
few element names, " ... ", and the last element name.

get() should return the extracted Runner*.
***********************************************************************/

class Group
 : public Multi
#if SIMULATING
 , public StatObj	// *MJS* If simulating
#endif
{
#if SIMULATING
  friend class SOGroup;	// *MJS*
#endif
  friend class Ring;

#if (STATISTICS || SIMULATING)
 private:
                // *MJS* These are for statistics gathering.
                // They can be used for simulation or for
                // taking measurements in real systems.
  boolean       counting;
  Time          waited;
  uint32        initCount;
  uint32        putCount;
  uint32        getCount;
  uint32        zeroWaitCount;
  SampledDisc   waitTimes;
  SampledCont   qLen;   // integral for queue length
#endif

 public:

/***************
.SH Group services
***************/
  Group&          put(const GE r);
                        	// *MJS* changed the meaning:
                        	// update statistics and call possibly
				// redefined method : myPut().
				// Only Group has put, derived classes
				// redefine their putInterface.
  Group&          get(GE& r);
                        	// *MJS* changed the meaning:
                        	// update statistics and call possibly
				// redefined method : myGet().
				// Only Group has get, derived classes
				// redefine their getInterface.

  virtual Group&  myPut(const GE r) = 0;
				// Add one Runner into the Group. Return *this.
				// Ring is an exception.  Derived classes modify put()'s
				// actions by redefining virtual myPut().
  virtual Group&  myGet(GE& r) = 0;
				// Remove one Runner* from Group to r.
				// r = &dummyRunner if empty group.
				// Returns *this.  Ring is an exception.
				// Derived classes modify get()'s actions by redefining
				// virtual myGet().

  Group&	  put(const GE o1,
		      const GE o2,
		      const GE o3 = 0, 
		      const GE o4 = 0,
		      const GE o5 = 0, 
		      const GE o6 = 0);
                                 //Add runners to the group,
                                 //return *this.  Ring is an exception.
  virtual GE	  head();
                                 //Without removing, return 
                                 //the Runner that get() would return.
  sint32  size() const;	 	 //Number of elements (Runner*'s)
                                 //currently in the Group.
  virtual sint32  dynamicSize() const;
                                 //Number of elements that can be removed
                                 //by get().  Usually 
                                 //dynamicSize() == size(),
                                 //but 0 for Ring, and for PriorVecs the
                                 //sum of dynamicSizes of elements Groups.
  virtual Group& putAnyway(const GE r);
                                 //Add one Runner into the Group, 
                                 //even if it is permanent by nature 
                                 //(get() does not
                                 //remove elements) such as Ring.
                                 //Returns *this.
				 // ? *MJS* update these too ?
  virtual Group& getAnyway(GE& r);
                                 //Remove one Runner* from Group to r,
                                 //even if the Group is permanent by nature 
                                 //(get() does not remove elements) 
                                 //such as Ring.
                                 //r = &dummyRunner if empty group.
                                 //Returns *this.
				 // ? *MJS* update these too ?

/**************
.SH Iteration
**************/
  virtual GE	  first(GP&) = 0;//
  virtual GE	  next(GP&) = 0; //
  virtual boolean over(const GP);//

/**************
.SH Miscellaneous
**************/
  virtual void	  print(Ostream& os);
                                 //Prints '(', the space separated name()'s 
                                 //of Group elements, and ')' to os.
  virtual void 	  ask(Istream& is);
  void		  printMsgs(Ostream* out, boolean help);
  virtual void	  reset(); 	 //reset() elements one by one, make the Group
                                 //empty. 
  virtual boolean isEmpty(); 	 //true if no elements, or isEmpty() is true
                                 //for all the elements.

  virtual String  className() const
                                 {return "Group";}

/***********************
.SH Statistics Gathering
************************/
#if SIMULATING
  				// *MJS* added the StatObj functions:
  redefined void        startSimExp();
  redefined void        finishSimExp();
  redefined SimObs*     createSimObs();
#endif


                 ~Group();	 //does nothing but makes compiled code smaller
 protected:
                  Group();
  sint32	  size_;	 //Number of elements (Runner*'s)
                                 //currently in the Group.
 private:
  static GP 	  dummyGP;
};


/******************************************************************* 
Iterates through Q once beginning from first.
At each step, runner points to the member.
*******************************************************************/
#define FOR_EACH_runner_IN(Q) \
  GP ddL; \
  for (GE runner = (Q).first(ddL); !(Q).over(ddL); runner = (Q).next(ddL))


/********************************************************************
A Link attaches Group elements.  It is used in derived class implementations.
Each Group has its own list of Links,
so that any Runner may belong to several independent Groups.
The elements do not contain any pointers to other elements.
.SH BUGS
It might be interesting to define a special memory allocation algorithm
for Link and see how it affects performance.
**********************************************************************/

class Link {
 public:
  GE	element;	//
  Link*	next;		//
  Link(const GE e, Link* n): element(e), next(n) {}	//
};


#if SIMULATING
  // *MJS* added class SOGroup :
  class SOGroup : public SimObs {
   private:
    String		gClassName;
    uint32		initCount;
    uint32		putCount;
    uint32		getCount;
    uint32		zeroWaitCount;
    SampledDisc		waitTimes;
    SampledCont		qLen;
   public:
    SOGroup(Group* g = 0);
   ~SOGroup();					//makes compiled code smaller
    redefined void	print(Ostream& os);
  };
#endif  /* SIMULATING */

