/*
** PREPARE.C - This is the module containing the code for SQL for
** preparing SQL Commands and other functions prior to execution.
**
** (c) 1996 by Dirk Ohme - all rights reserved
*/

#ifdef OS2
#  include <os2.h>
#endif
#ifdef WIN
#  include <dos.h>
#endif

#include "cli.h"

/*---------------------------------------------------------------------------*/
/*      Allocate a SQL statement                                             */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLAllocStmt(
        LPDBC   lpdbc,
        HSTMT FAR *phstmt)
{
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLAllocStmt] lpdbc = $%08lX", lpdbc );
        if( NULL == lpdbc )
        {
                LogEntry( LOG_ERROR,      "[SQLAllocStmt] NULL == lpdbc" );
                LogEntry( LOG_RETURNCODE, "[SQLAllocStmt] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }

        if( NULL == phstmt )
        {
                strcpy( lpdbc->szSqlState, "S1009" );
                lpdbc->pszSqlMsg = "[SQLAllocStmt] NULL == phstmt";
                LogEntry( LOG_ERROR, lpdbc->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLAllocStmt] SQL_ERROR (%s)",
                          lpdbc->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** allocate memory
        */
        *phstmt=malloc( sizeof(STMT) );
        if( SQL_NULL_HSTMT == *phstmt )
        {
                strcpy( lpdbc->szSqlState, "S1001" );
                lpdbc->pszSqlMsg = "[SQLAllocStmt] memory allocation failure";
                LogEntry( LOG_ERROR, lpdbc->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLAllocStmt] SQL_ERROR (%s)",
                          lpdbc->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** initialize memory
        */
        LogEntry( LOG_RETURNCODE, "[SQLAllocStmt] *phstmt = $%08lX", *phstmt );
        memset( *phstmt, 0, sizeof(STMT) );

        ((SQLHSTMT)(*phstmt))->pSqlDbc        = lpdbc;
        ((SQLHSTMT)(*phstmt))->szSqlState     = lpdbc->szSqlState;
        ((SQLHSTMT)(*phstmt))->szMsqlErrorMsg = &lpdbc->szMsqlErrorMsg[0];

        /*
        ** increase number of open statements
        */
        lpdbc->ciActive++;

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLAllocStmt] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Free a SQL Statement                                                 */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLFreeStmt(
        LPSTMT  lpstmt,
        UWORD   fOption)
{
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLFreeStmt] lpstmt  = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "              fOption = %s (%d)",
                  (SQL_CLOSE == fOption)          ? "SQL_CLOSE"
                  : (SQL_DROP == fOption)         ? "SQL_DROP"
                  : (SQL_UNBIND == fOption)       ? "SQL_UNBIND"
                  : (SQL_RESET_PARAMS == fOption) ? "SQL_RESET_PARAMS"
                  : "???",
                  lpstmt );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLFreeStmt] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLFreeStmt] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }
        if( SQL_CLOSE        != fOption ||
            SQL_DROP         != fOption ||
            SQL_UNBIND       != fOption ||
            SQL_RESET_PARAMS != fOption )
        {
        }

        /*
        ** reset parameters
        */
        switch( fOption )
        {
                case SQL_CLOSE:
                case SQL_DROP:
                case SQL_UNBIND:
                case SQL_RESET_PARAMS:
                        strcpy( lpstmt->szSqlState, "00000" );
                        lpstmt->pszSqlMsg = NULL;
                        /* do nothing */
                        break;
                default:
                        strcpy( lpstmt->szSqlState, "S1092" );
                        lpstmt->pszSqlMsg =
                            "[SQLFreeStmt] invalid fOption specified";
                        LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                        LogEntry( LOG_RETURNCODE,
                                  "[SQLFreeStmt] SQL_ERROR (%s)",
                                  lpstmt->szSqlState );
                        return SQL_ERROR;
        } /* switch */

        /*
        ** unbind parameters
        */
        switch( fOption )
        {
                case SQL_CLOSE:
                case SQL_DROP:
                case SQL_UNBIND:
                        /* do nothing */
                default:
                        break;
        } /* switch */

        /*
        ** drop statement parameters
        */
        switch( fOption )
        {
                case SQL_CLOSE:
                case SQL_DROP:
                        /*---| free query |---*/
                        if( NULL != lpstmt->pszQuery )
                        {
                                free( lpstmt->pszQuery );
                                lpstmt->pszQuery = NULL;
                        }

                        /*---| free result handle |---*/
                        if( NULL != lpstmt->pMsqlResult )
                        {
                                msqlFreeResult( lpstmt->pMsqlResult );
                                lpstmt->pMsqlResult = NULL;
                        }

                        /*---| decrease number of active statements |---*/
                        lpstmt->pSqlDbc->ciActive--;
                default:
                        break;
        } /* switch */

        /*
        ** withdraw statement handle
        */
        if( SQL_CLOSE == fOption )
        {
                free( lpstmt );
        }

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLFreeStmt] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Perform a Prepare on the SQL statement                               */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLPrepare(
        LPSTMT  lpstmt,
        UCHAR FAR *szSqlStr,
        SDWORD  cbSqlStr)
{                                                /*--------------------------*/
        int     ci;                              /* counter                  */
                                                 /*--------------------------*/
        /*
        ** check parameter
        */
        LogEntry( LOG_STATUS, "[SQLPrepare] lpstmt   = $%08lX", lpstmt );
        LogEntry( LOG_STATUS, "             szSqlStr = $%08lX", szSqlStr );
        LogEntry( LOG_STATUS, "             cbSqlStr =  %5d",   cbSqlStr );
        if( NULL == lpstmt )
        {
                LogEntry( LOG_ERROR,      "[SQLPrepare] NULL == lpstmt" );
                LogEntry( LOG_RETURNCODE, "[SQLPrepare] SQL_INVALID_HANDLE" );
                return SQL_INVALID_HANDLE;
        }
        if( NULL == szSqlStr )
        {
                strcpy( lpstmt->szSqlState, "S1009" );
                lpstmt->pszSqlMsg = "[SQLPrepare] NULL == szSqlStr";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLPrepare] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** check statement handle
        */
        if( NULL != lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "24000" );
                lpstmt->pszSqlMsg = "[SQLPrepare] statement already in use";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLPrepare] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        /*
        ** allocate memory and prepare SQL statement
        */
        ci = (SQL_NTS != cbSqlStr) ? cbSqlStr : strlen(szSqlStr);

        lpstmt->pszQuery = malloc( ci + 1 );
        if( NULL == lpstmt->pszQuery )
        {
                strcpy( lpstmt->szSqlState, "S1001" );
                lpstmt->pszSqlMsg = "[SQLPrepare] memory allocation error";
                LogEntry( LOG_ERROR, lpstmt->pszSqlMsg );
                LogEntry( LOG_RETURNCODE, "[SQLPrepare] SQL_ERROR (%s)",
                          lpstmt->szSqlState );
                return SQL_ERROR;
        }

        memset( lpstmt->pszQuery, 0, ci + 1 );
        strncpy( lpstmt->pszQuery, szSqlStr, ci );

        /*
        ** return success
        */
        LogEntry( LOG_RETURNCODE, "[SQLPrepare] SQL_SUCCESS" );
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Set parameters on a statement handle                                 */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLSetParam(
        LPSTMT  lpstmt,
        UWORD   ipar,
        SWORD   fCType,
        SWORD   fSqlType,
        UDWORD  cbColDef,
        SWORD   ibScale,
        PTR     rgbValue,
        SDWORD FAR *pcbValue)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Returns the description of a parameter marker.                       */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLDescribeParam(
        LPSTMT  lpstmt,
        UWORD   ipar,
        SWORD FAR *pfSqlType,
        UDWORD FAR *pcbColDef,
        SWORD FAR *pibScale,
        SWORD FAR *pfNullable)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Sets multiple values (arrays) for the set of parameter markers.      */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLParamOptions(
        LPSTMT  lpstmt,
        UDWORD  crow,
        UDWORD FAR *pirow)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Returns the number of parameter markers.                             */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLNumParams(
        LPSTMT  lpstmt,
    SWORD FAR *pcpar)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Sets options that control the behavior of cursors.                   */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLSetScrollOptions(
        LPSTMT  lpstmt,
        UWORD   fConcurrency,
        SDWORD  crowKeyset,
        UWORD   crowRowset)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Set the cursor name on a statement handle                            */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLSetCursorName(
        LPSTMT  lpstmt,
        UCHAR FAR *szCursor,
        SWORD   cbCursor)
{
        return SQL_SUCCESS;
}


/*---------------------------------------------------------------------------*/
/*      Return the cursor name for a statement handle                        */
/*---------------------------------------------------------------------------*/
RETCODE SQL_API SQLGetCursorName(
        LPSTMT  lpstmt,
        UCHAR FAR *szCursor,
        SWORD   cbCursorMax,
        SWORD FAR *pcbCursor)
{
        return SQL_SUCCESS;
}

/*===| end of file |=========================================================*/
