/*
 * in.cfingerd.c --
 *	Return Cool Finger Daemon information.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: in.cfingerd.c,v 1.30 1995/08/17 01:22:20 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/param.h>
#include <sys/stat.h>

#include "../config.h"

#include "packet.h"
#include "util.h"
#include "os.h"
#include "getservhost.h"
#include "fingerpaths.h"
#include "error.h"
#include "flock.h"

#ifdef USE_HPCLUSTER
#define CLUSTERCONF "/etc/clusterconf"
#include <cluster.h>
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* How old the hostdata and clientstatus files can get before we suspect
   something is wrong. */
#ifndef MIN_UPDATE_INTERVAL
#define MIN_UPDATE_INTERVAL (2 * DEFAULT_POLL_INTERVAL)
#endif

/* Time we allow ourselves to run before aborting */
#ifndef SELF_DESTRUCT_TIMEOUT
#define SELF_DESTRUCT_TIMEOUT	(2 * DEFAULT_POLL_INTERVAL)
#endif 

/* If this is non-zero then all instances of users are reported on
   instead of having one unique entry for each user. */
int all_users = 0;

/* print entire host/user database (if running on server host) */
int all_hosts = 0;

/* print ASCII output for debugging purposes */
int debugging = 0;

/* collect lastlog entries as well */
int lastlog = 0;

/* **************************************************************** */
/*								    */
/*			Client Finger Daemon			    */
/*								    */
/* **************************************************************** */

static void
ship_packets(packets)
    FINGER_PACKET **packets;
{
  register int i;

  for (i = 0; packets[i]; i++) {
    if ( debugging )
	print_packet(packets[i], stdout);
    else
      {
        packets[i]->login_time = htonl(packets[i]->login_time);
        packets[i]->idle_time = htonl(packets[i]->idle_time);
        fwrite ((char *)packets[i], sizeof (FINGER_PACKET), 1, stdout);
      }
  }

  fflush (stdout);
}

/* This gets invoked when called by the GNU Finger server host.  The only
   purpose of this code is to return local finger packets in binary. */
/*ARGSUSED*/
main (argc, argv)
     int argc;
     char **argv;
{
  FINGER_PACKET **packets;
  char arg[128];
  char peername[MAXHOSTNAMELEN + 1];

#ifdef USE_SYSLOG
  default_error_handling (argv[0], LOG_SYS);
#else
  /* prevent error messages from garbling binary output */
  default_error_handling (argv[0], LOG_NONE);
#endif

  /* Sometimes things get very slow or hang due to file servers not
     responding, or we run forever due to a looping bug.
     The polling server times out in these case -- we just commit suicide
     to avoid cluttering the client machine with daemons */
  alarm(SELF_DESTRUCT_TIMEOUT);

  if (!local_socket (fileno(stdin), peername, sizeof(peername), NULL, 0))
    {
      handle_error(WARNING, "non-local connection attempted from %s\n",
			peername);
      exit(2);
    }

  if (!fgets (arg, sizeof (arg), stdin))
      exit (1);

  /* parse options */
  {
    char *p, *q;

    for ( p = arg; *p; p = q )
      {
        for ( ; *p && isspace(*p); p++ );
        for ( q = p; *q && !isspace(*q); q++ );
        if ( *q )
          *q++ = '\0';

        if ( strcmp(p, "-debug") == 0 )
	    debugging = 1;
        else if ( strcmp(p, "-lastlog") == 0 )
	    lastlog = 1;
        else if ( strcmp(p, "-all-users") == 0 )
	    all_users = 1;
        else if ( strcmp(p, "-all-hosts") == 0 )
	    all_hosts = 1;
	else if (*p)
	    handle_error(WARNING, "unknown option %s", p);
      }
  }

  if (all_hosts)
    {
      char *serverhost;
      int f;

      /* Are we the finger daemon running on the server host? */
      if (!(serverhost = getservhost (GETSERV_LOCAL|GETSERV_NOERROR)) ||
          !same_hostname (get_full_hostname (), serverhost))
	{
	  handle_error(WARNING, "-all-hosts polling attempt from %s", peername);
	  exit(1);
	}

      if (lastlog)
        {
	  warn_if_not_recent (USERDATA);

          if ((f = open (USERDATA, O_RDONLY)) < 0 ||
	      flock (f, LOCK_SH) < 0)
	    file_error(FATAL, USERDATA);
	  packets = read_packets(f, 0);
        }
      else
        {
	  warn_if_not_recent (HOSTDATA);

          if ((f = open (HOSTDATA, O_RDONLY)) < 0 ||
	      flock (f, LOCK_SH) < 0)
	    file_error(FATAL, HOSTDATA);
	  packets = read_packets(f, 0);
        }

      if (!packets)
        exit (1);

      ship_packets (packets);
    }
  else
    {
      if (lastlog)
	packets = get_lastlog_data (NULL);
      else
	packets = get_finger_data (NULL, all_users);

      if (!packets)
	handle_error(FATAL, "couldn't get finger packets");

      ship_packets (packets);

#ifdef USE_HPCLUSTER
      {
	FILE *cconf;
	struct cct_entry *ccent;

	/* We open our own stream to /etc/clusterconf so we are not thrown
	   off by any lookups on lib/os.c */
	cconf = fopen(CLUSTERCONF, "r");
	if (!cconf)
	  {
	    file_error(WARNING, CLUSTERCONF);
	    exit (0);
	  }

	while (ccent = fgetccent(cconf))
	  {
	    char cnode[MAXHOSTNAMELEN];

	    /* don't do local host again */
	    if (same_hostname(ccent->cnode_name, get_full_hostname ()))
	      continue;

	    /* We have to copy the string because its storage will be reused
	       during processing in lib/os.c. */
	    strcpy(cnode, ccent->cnode_name);

	    if (lastlog)
	      packets = get_lastlog_data (cnode);
	    else
	      packets = get_finger_data (cnode, all_users);

	    if (!packets)
	      handle_error(FATAL, "couldn't get finger packets for %s", cnode);

	    ship_packets (packets);
	    free_array (packets);
	  }

	fclose(cconf);
      }
#endif /* USE_HPCLUSTER */
    }

  exit (0);
}

/* Simply complain if FILENAME hasn't been written in a while.  This can be
   an indication that the finger server has wedged or otherwise stopped
   running.  Return non-zero if the file hasn't been written recently. */
int
warn_if_not_recent (filename)
     char *filename;
{
  int result = 0;
  struct stat finfo;

  if (stat (filename, &finfo) != -1)
    {
      time_t last_change = time ((time_t *)0) - finfo.st_mtime;

      if (last_change > MIN_UPDATE_INTERVAL)
	{
	  char *itime = idle_time_string (last_change, 0);

	  handle_error (ALERT, "%s: file has not changed in %s",
		   filename, itime);

	  free (itime);
	  result = 1;
	}
    }
  else
    {
      /* don't report error here, will do so later anyway */
      /* file_error (WARNING, filename); */
      result = 1;
    }

  return (result);
}

