/*
 * fingerd.c --
 *	Poll machines for information on who is using them.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: fingerd.c,v 1.63 1994/10/14 03:18:15 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <ctype.h>

#include "../config.h"

#include "general.h"
#include "client.h"
#include "fingerpaths.h"
#include "getservhost.h"
#include "packet.h"
#include "os.h"
#include "tcp.h"
#include "error.h"
#include "util.h"
#include "flock.h"

#ifndef DEFAULT_POLL_INTERVAL
#define DEFAULT_POLL_INTERVAL 120
#endif

#ifndef DEFAULT_POLL_COUNT
#define DEFAULT_POLL_COUNT 100
#endif

#ifndef DEFAULT_CONNECT_TIMEOUT
#define DEFAULT_CONNECT_TIMEOUT 1000000
#endif

#ifndef DEFAULT_POLL_TIMEOUT
#define DEFAULT_POLL_TIMEOUT 5000000
#endif

#ifndef SIGRET
#define SIGRET void
#endif

/* The database containing info from all polled hosts. */
char *hostdata;

/* The database containing info about specific users. */
char *userdata;

/* The database of the status of polled clients. */
char *hoststat;

/* The amount of time (in seconds) we sleep between each poll. */
int time_between_polls = DEFAULT_POLL_INTERVAL;

/* The number of machines polled each time. */
int poll_count = DEFAULT_POLL_COUNT;

/* We keep the database of inet numbers in core.  This appears to be the
   most effecient use of available resources. */

/* The client status file descriptor */
int clients_file = -1;

/* Array of clients.  Each entry contains a HOSTENT, as returned
   by gethostbyaddr (), and a HOSTSTATUS.  The list is terminated
   with a NULL pointer. */
CLIENT **clients = (CLIENT **)NULL;

/* Offset into the clients array. */
int client_index;

/* Number of slots created for the CLIENTS array. */
int clients_size = 0;

/* Flag set by SIGHUP */
int init_clients = 0;

/* Flag variables. */

/* Non-zero means report on all users, even if the same one appears twice. */
int all_users = 0;

/* Non-zero means print debugging info. */
int debugging = 0;
int poll_debugging = 0;

/* The output stream for debugging. */
FILE *debug_output = stderr;

/* The hostdata file descriptor */
int host_file = -1;

/* Our in RAM database of user/host mappings. */
FINGER_PACKET **host_packets = (FINGER_PACKET **)NULL;

/* Number of slots allocated to the above array. */
int host_packets_size = 0;

/* Number of slots filled in the above array. */
int host_packets_len;

/* The userdata file descriptor */
int user_file = -1;

/* Our in RAM database of last user records. */
FINGER_PACKET **user_packets = (FINGER_PACKET **)NULL;

/* Number of slots allocated to the above array. */
int user_packets_size = 0;

int true_cmp(), host_cmp(), bogus_packet();

/* The daemon which polls clients.  Possible arguments on startup:
   -debug	print debugging info.
   -poll-debug	print polling debugging info.
   -connect-timeout	set connection timeout
   -poll-timeout	set polling timeout 
   -poll-interval	set sleep between polls
   -all-users	get info for all users, not just unique ones. */
main (argc, argv)
     int argc;
     char **argv;
{
  char *serverhost;
  int arg_index = 1;

  allow_time_outs = 1;
  connect_timeout = DEFAULT_CONNECT_TIMEOUT;
  read_timeout = DEFAULT_POLL_TIMEOUT;

  /* Parse arguments. */
  for (arg_index = 1; arg_index != argc; arg_index++)
    {
      if (strcmp (argv[arg_index], "-debug") == 0)
	{
	  debugging = 1;
	  continue;
	}
      else if (strcmp (argv[arg_index], "-poll-debug") == 0)
	{
	  poll_debugging = 1;
	  continue;
	}
      else if (strcmp (argv[arg_index], "-connect-timeout") == 0)
	{
	  if ( arg_index < argc - 1 )
	    {
	      connect_timeout = atoi(argv[++arg_index]);
	      continue;
	    }
	}
      else if (strcmp (argv[arg_index], "-poll-timeout") == 0)
	{
	  if ( arg_index < argc - 1 )
	    {
	      read_timeout = atoi(argv[++arg_index]);
	      continue;
	    }
	}
      else if (strcmp (argv[arg_index], "-poll-interval") == 0)
	{
	  if ( arg_index < argc - 1 )
	    {
	      time_between_polls = atoi(argv[++arg_index]);
	      continue;
	    }
	}
      usage ();
    }

  /* Errors go to stderr if debugging, otherwise to syslog */
  if (!debugging)
    default_error_handling (argv[0], LOG_SYS);

  serverhost = getservhost (GETSERV_LOCAL);
  if (!serverhost)
    exit (1);

  if (!same_hostname (get_full_hostname (), serverhost))
    handle_error (FATAL, "%s is not the Finger server.  %s is.",
		  get_full_hostname (), serverhost);

  if (!debugging)
    {
      int pid;

      pid = fork ();

      if (pid == -1)
	handle_error (FATAL, "Can't fork a child!");
      else if (pid > 0)
	{
	  fprintf(stderr, "%s: daemon starting (pid %d)\n", argv[0], pid);
	  exit (0);
	}

      /* Turn off the controlling terminal in the child. */
      {
#if defined(SYSV) || defined(sun) || defined(ultrix)	/* possibly others */
	setsid();
#else
	int tty = open ("/dev/tty", O_RDWR, 0666);
	
	if (tty)
	  ioctl (tty, TIOCNOTTY, 0);

	close (tty);
#endif
      }

      /* Using syslog, so we can close stdio streams.  This will also
         detach the daemon from the shell. */
      fclose(stdin);
      fclose(stdout);
      fclose(stderr);
    }
  else
    {
      allow_time_outs = 0;
    }

  debug (debugging || poll_debugging,
	"connect timeout = %.6f secs%s", connect_timeout/1000000.0,
			 allow_time_outs ? "" : " (disabled for debugging)");
  debug (debugging || poll_debugging,
	"poll timeout    = %.6f secs", read_timeout/1000000.0);
  debug (debugging || poll_debugging,
	"poll interval   = %d secs", time_between_polls);

  /* Initialize the data base and defaults. */
  initialize_server ();

  while (1)
    {
      if (setjmp (top_level))
	{
	  clean_up_for_exit ();
	  exit (2);
	}

      poll_some_clients ();
      sleep (time_between_polls);
    }
}

/* What to do just before abnormal exit. */
clean_up_for_exit ()
{
}

/* Tell people how to start the finger daemon. */
usage ()
{
  fprintf (stderr, "usage: %s [options]\n", progname);
  fprintf (stderr, "options:\n\
	-debug			full debugging, don't fork daemon\n\
	-poll-debug		track polling cycle\n\
	-connect-timeout value	set connection timeout (in micro-secs, 0 = none)\n\
	-poll-timeout value	set polling timeout (in micro-secs, 0 = none)\n\
	-poll-interval value	set interval between polls (in secs)\n");

  exit (1);
}

static SIGRET
terminate_server(signo)
  int signo;
{
  close(user_file);
  close(host_file);
  if (clients_file >= 0)
      close(clients_file);

  handle_error (FATAL, "going down on signal %d", signo);
}

static SIGRET
switch_debug(signo)
  int signo;
{
#ifdef SYSV
  signal (signo, switch_debug);
#endif

  if (signo == SIGUSR1)
    poll_debugging = 1;
  else if (signo == SIGUSR2)
    poll_debugging = 0;
}

static SIGRET
reinit_server(signo)
  int signo;
{
#ifdef SYSV
  signal (signo, reinit_server);
#endif

  handle_error (WARNING, "reinitializing on signal %d", signo);
  init_clients = 1;
}

/* Initialize the server. */
initialize_server ()
{
  initialize_pathnames ();
  initialize_clients ();
  initialize_fingerdata ();
    
  if (!host_packets)
    {
      host_packets =
	(FINGER_PACKET **)xmalloc ((host_packets_size = 40)
				   * sizeof (FINGER_PACKET *));
      host_packets[0] = (FINGER_PACKET *)NULL;
      host_packets_len = 0;
    }

  signal (SIGHUP, reinit_server);
  signal (SIGINT, terminate_server);
  signal (SIGQUIT, terminate_server);
  signal (SIGTERM, terminate_server);
  signal (SIGUSR1, switch_debug);
  signal (SIGUSR2, switch_debug);
}

/* Set global pathname variables which specify the location of 
   the client data, the clients to poll, etc. */
initialize_pathnames ()
{
  char *getenv ();

  hostdata = HOSTDATA;
  userdata = USERDATA;
  hoststat = HOSTSTAT;
}

/* Initialize the list of pollable clients from the config file
   Called by fingerd_initialize (), and when the daemon gets a SIGHUP. */
initialize_clients ()
{
  initialize_clients_internal ();

  if (clients_file >= 0)
      close(clients_file);

  /* reopen clientstatus file and write initial contents */
  clients_file = open (hoststat, O_WRONLY | O_CREAT | O_TRUNC, 0666);
  if (clients_file < 0 ||
      flock (clients_file, LOCK_EX) < 0 || 
      write_host_status (clients, clients_file) < 0)
      file_error (ALERT, hoststat);

  if (clients_file >= 0)
      flock (clients_file, LOCK_UN);

  /* nuke all host packets -- may be from old clients */
  if (host_packets)
    {
      remove_packets (host_packets, true_cmp, NULL);
      host_packets_len = 0;
    }
}

/* Initialize the list of pollable clients from host config file.

   The file contains specifications of clients to poll.  The
   specifications may be in one of the following forms:

	a) The name of an internet host.
	b) The internet address of a host.
	c) A subnet specification, preceded by the keyword ":subnet".
	   You should be able to do ranges of machines with this,
	   but the only syntax that I can think of looks like:

	      :subnet 128.111.47.0 - 10

	   which would specify the machines from .0 to .10.
	d) A gateway host, preceeded by the keyword ":gateway".
	   This should be another host running a finger server.
	   fingerd will obtain all host and user data from that host
	   and merge it with it's own data.  It is thus possible to
	   build a hierarchical or heterarchical network of finger domains.
	e) The keyword :local, causing the local host to be queried by
	   fingerd itself (i.e., without going through in.cfingerd).
	f) A comment, which is a line whose first character is
	   a pound sign.  The comment ends at the end of the line.
 */
initialize_clients_internal ()
{
  char *line;
  
  /* Close current hostconfig file to make sure we get the new one in
     case the ino changed. */
  close_host_config ();
  (void)open_host_config ();

  if (clients)
    {
      free_array(clients);
      clients = (CLIENT **)NULL;
      clients_size = 0;
    }
    
  client_index = 0;

  for (; line = get_config_entry("client", NULL); free(line))
    {
      int status = 0;
      char *start, *end;
      struct hostent *hostinfo;
      struct in_addr address;

      /* If special directive, do what it says. */
      if (*line == ':')
	{
	  start = line + 1;
	  /* Find end of directive. */
	  for (end = start; *end && !isspace(*end); end++);

	  if (strnicmp (start, "subnet", end - start) == 0)
	    {
	      warning ("client: can't hack subnets yet");
	      continue;
	    }
	  else if (strnicmp (start, "gateway", end - start) == 0)
	    {
	      status |= CLIENT_GATEWAY;

	      /* Skip to first character of hostname */
	      for (start = end; *start && isspace (*start); start++);

	      if (!*start)
	        {
	          warning ("client: missing gateway host");
	          continue;
	        }
	    }
	  else if (strnicmp (start, "local", end - start) == 0)
	    {
	      status |= CLIENT_LOCAL;
	      address.s_addr = 0;

	      /* Skip to first character of hostname */
	      for (start = end; *start && isspace (*start); start++);

	      /* If no hostname follows, assume local machine */
	      if (!*start)
		{
		  add_client (get_full_hostname (), address, status);
		  continue;
		}
	    }
	  else
	    {
	      warning ("client: bad directive: %s", line);
	      continue;
	    }
	}
      else
        start = line;

      /* This is a regular host specification.  It is either an internet
	 address, or a hostname. */
      if (isdigit (start[0]) &&
	  (address.s_addr = inet_addr (start)) != -1)
	{
	  if (hostinfo = gethostbyaddr ((char *)&address,
					sizeof(address), AF_INET))
	    start = hostinfo->h_name;
	}
      else
	{ 
 	  if (hostinfo = gethostbyname (start))
	    bcopy (hostinfo->h_addr, &address, sizeof(address));
	}

      if (!hostinfo)
	warning ("%s: unknown host", start);
      else
	add_client (start, address, status);
    }

  if (client_index == 0)
    handle_error (FATAL, "no clients found -- exiting");

  client_index = 0;
}

add_client (name, address, status)
     char *name;		/* name as we want it recorded */
     struct in_addr address;
     int status;
{
  CLIENT *new;

  if (client_index + 2 > clients_size)
    {
      if (clients == (CLIENT **)NULL)
	clients =
	    (CLIENT **)xmalloc ((clients_size = 20) * sizeof (CLIENT *));
      else
	clients = (CLIENT **)xrealloc
	    (clients, (clients_size += 20) * sizeof (CLIENT *));
    }

  new = (CLIENT *)xmalloc (sizeof (CLIENT));
  bzero ((char *)new, sizeof (CLIENT));
  strcpy (new->hostname, name);
  new->address = address;
  new->status = status;
  new->users = 0;
  new->idle_time = 0;
  new->times_polled = 0;

  clients[client_index] = new;
  clients[++client_index] = (CLIENT *)NULL;
}

/* Make sure that the files named in HOSTDATA and USERDATA exist.
   Open them and read old USERDATA into memory. */
initialize_fingerdata ()
{

  if (host_file < 0)
    {
      host_file = open (hostdata, O_WRONLY | O_CREAT, 0666);
      if (host_file < 0)
          file_error (FATAL, hostdata);
    }

  if (user_file < 0)
    {
      int bogus;

      user_file = open (userdata, O_RDWR | O_CREAT, 0666);
      if (user_file < 0)
          file_error (FATAL, userdata);

      if (user_packets)
          free_array(user_packets);

      user_packets = read_packets(user_file, 0);
      if (! user_packets)
          file_error (FATAL, userdata);

      user_packets_size = array_len(user_packets);
      debug (debugging || poll_debugging,
          "%d old user packets from %s", user_packets_size, userdata);

      /* check for bogus packets */
      bogus = remove_packets(user_packets, bogus_packet, NULL);
      if (bogus > 0)
	  handle_error (WARNING, "%d bogus packets in old userdata", bogus);

      /* sort by user and remove duplicates */
      sort_packets (user_packets);
      bogus = uniq_packets (user_packets);
      if (bogus > 0)
	  handle_error (WARNING, "%d duplicate users in old userdata", bogus);

      /* read_packets() has closed user_file */
      user_file = open (userdata, O_WRONLY | O_CREAT, 0666);
      if (user_file < 0)
          file_error (FATAL, userdata);

      /* Rewrite the userdata file to bring it in sync with our memory image. */
      if (flock (user_file, LOCK_EX) < 0 ||
          write_packets (user_packets, user_file) < 0)
          file_error (ALERT, userdata);

      flock (user_file, LOCK_UN);
    }
}

/* Poll POLL_COUNT hosts from CLIENTS, starting at CLIENT_INDEX.
   Leave CLIENT_INDEX pointing at the next host to poll. */
poll_some_clients ()
{
  register int i, j;
  CLIENT *client;

  /* if SIGHUP told us to reinitialize clients list, do it here */
  if (init_clients)
    {
      reset_ttylocs ();
      initialize_clients ();
      init_clients = 0;
    }

  for (i = 0, j = client_index; i < poll_count; i++)
    {
      client = clients[j];

      /* If we have reached the end of the list, wrap around and start
	 at the top again.  If the list is shorter than POLL_COUNT, then
	 just reset to start of list. */
      if (!client)
	{
	  if (client_index == 0)
	    {
	      j = 0;
	      break;
	    }
	  else
	    {
	      client_index = j = 0;
	      continue;
	    }
	}
      else
	j++;

      if (! (client->status & CLIENT_LASTLOG))
          poll_client (client, 1);

      poll_client (client, 0);
  }

  client_index = j;

  /* After doing a set of hosts, save the information on disk.  This is
     the way that we pass information from the poller to the reply server. */
  if (flock (host_file, LOCK_EX) < 0 ||
      write_packets (host_packets, host_file) < 0)
      file_error (ALERT, hostdata);

  flock (host_file, LOCK_UN);

  /* Also update the client status file. */
  if (clients_file >= 0 &&
      (flock (clients_file, LOCK_EX) < 0 ||
       write_host_status (clients, clients_file) < 0))
      file_error (ALERT, hoststat);

  if (clients_file >= 0)
      flock (clients_file, LOCK_UN);

  /* update user data mtime even if nothing was written to it */
#ifdef SYSV
  utime(USERDATA, NULL);
#else
  utimes(USERDATA, NULL);
#endif
}

/* Poll a single CLIENT.  Add information to the global data
   base of logged in users. */
poll_client (client, lastlog)
     CLIENT *client;
     int lastlog;		/* request lastlog info */
{
  FINGER_PACKET **packets;
  int i;
  int gateway = (client->status & CLIENT_GATEWAY);
  time_t timestamp = time((time_t *)NULL);

  debug (debugging || poll_debugging,
	"polling %s%s%s", gateway ? "gateway " : "host ",
			  client->hostname,
			  lastlog ? " for lastlog" : "");

  if (client->status & CLIENT_LOCAL)
    {
      if (lastlog)
	packets = get_lastlog_data (client->hostname);
      else
	packets = get_finger_data (client->hostname, 0);
    }
  else
    {
      if (lastlog)
	{
	  int old_timeout = read_timeout;

	  /* increase timeout for lastlog request */
	  read_timeout = 2 * read_timeout;
	  packets = finger_at_address (&client->address,
				  gateway ? "-all-hosts -lastlog" : "-lastlog");
	  read_timeout = old_timeout;
	}
      else
	packets = finger_at_address (&client->address,
				gateway ? "-all-hosts" : "");
    }

  if (packets && packets[0])
    {
      int bogus;
      char *lasthost = NULL;

      /* If the client was down before, log that it's back up */
      if (client->idle_time != 0 && !(client->status & CLIENT_UP))
	warning ("%s is back up", client->hostname);

      /* check for bogus packets */
      bogus = remove_packets(packets, bogus_packet, NULL);
      if (bogus > 0)
	  handle_error (WARNING, "got %d bogus packets from %s",
				 bogus, client->hostname);

      /* Since we have just polled this client, the information is more
         current than anything that we have saved.  So remove all information
	 about all hosts this client tells us about */
      if (!lastlog)
        for (i = 0; packets[i]; i++)
	  /* packets belonging to the same host come usually back-to-back
	     -- avoid unnecessary removals in this case */
	  if (!lasthost || strcmp(lasthost, packets[i]->host) != 0)
	    {
	      lasthost = packets[i]->host;
	      host_packets_len -= remove_host_packets (host_packets, lasthost);
	    }
    }
  else if (packets)
    {
      /* Polling returned no packets.  This is not supposed to happen since
         at least a dummy packet is usually sent.  Probably a timeout.
      */
      warning("No packets from %s -- timeout may be too short",
              client->hostname);
    }
  else
    {
      FINGER_PACKET dummy;

     /* Log the error (reason for not getting the data) since it
	may be indicative of some problem.  But do so only the
	first time the host is noticed as being down down.
      */
      if (client->idle_time == 0 || (client->status & CLIENT_UP))
	file_error (WARNING, client->hostname);

      /* There was an error from the host, probably because it's down
	 or its server is down.  So we remove the entries about this host.
	 We don't remove packets immediately (may just be a fluke, e.g.,
	 because of a load peek), but only after the call fails a second time.
      */
      if (!(client->status & CLIENT_UP))	/* was down before */
	{
	  host_packets_len -=
		remove_host_packets (host_packets, client->hostname);

	  /* To expurge any user packets from higher level finger servers,
	     we create a dummy packet in this host's name and add it to the
	     host data. Dummy packets are marked by an empty login name. */
	  dummy.name[0] = '\0';
	  strncpy(dummy.real_name, HOST_DOWN, sizeof(dummy.real_name));
	  strncpy(dummy.host, client->hostname, sizeof(dummy.host));
	  dummy.login_time = 0;
	  dummy.idle_time = 0;
	  dummy.ttyname[0] = '\0';
	  dummy.ttyloc[0] = '\0';
	  dummy.what[0] = '\0';
	  record_user_info (&dummy, timestamp, lastlog, gateway);
	}
    }

  /* If there was an error, then this client is deemed down.
     Same if local polling explicitly tells us the machine is down */
  if (!packets ||
      packets[0] &&
      !packets[0]->name[0] &&
      strcmp(packets[0]->real_name, HOST_DOWN) == 0 &&
      same_hostname(packets[0]->host, client->hostname))
    {
      /* record start of down time */
      if (client->idle_time == 0 || (client->status & CLIENT_UP))
	client->idle_time = timestamp;

      client->status &= ~CLIENT_UP;
    }
  else
    {
      client->status |= CLIENT_UP;
      client->times_polled++;

      if (lastlog)
        client->status |= CLIENT_LASTLOG;
    }

  /* Remember the information about each user at this machine. */
  if (packets)
    {
      int users = 0;
      time_t machine_idle_time = (time_t)-1;
 
      for (i = 0; packets[i]; i++)
	{
	  if (debugging)
	    {
	      print_packet (packets[i], debug_output);
	      fflush(debug_output);
	    }

	  /* ignore dummy packets in user count */
	  if (packets[i]->name[0] != '\0')
	    {
	      if (machine_idle_time == (time_t)-1 ||
		  packets[i]->idle_time < machine_idle_time)
	        machine_idle_time = packets[i]->idle_time;

	      users++;
	    }

	  record_user_info (packets[i], timestamp, lastlog, gateway);
	}

      debug (debugging || poll_debugging, 
	    "%d packets from %s", i, client->hostname);

      client->users = users;
      client->idle_time = machine_idle_time;
      free_array (packets);
    }
   else
    {
      client->users = 0;
    }
}

/* **************************************************************** */
/*								    */
/*		     Keeping Track of User/Host			    */
/*								    */
/* **************************************************************** */

/* Record the finger information found in PACKET in our database of
   such packets. */
record_user_info (packet, timestamp, lastlog, gateway)
     FINGER_PACKET *packet;
     time_t timestamp;		/* when packet is recorded */
     int lastlog;		/* packet is a lastlog record */
     int gateway;		/* packet comes from a gateway */
{
  if (!lastlog)
    {
      /* We already junked all packets from this host, so we know we can
         just add this one at the end. */
      if (host_packets_len + 2 > host_packets_size)
	host_packets =
	  (FINGER_PACKET **)xrealloc (host_packets,
				      (host_packets_size *= 2)
				      * sizeof (FINGER_PACKET *));

      host_packets[host_packets_len] =
	    (FINGER_PACKET *)xmalloc (sizeof (FINGER_PACKET));
      bcopy (packet, host_packets[host_packets_len], sizeof (FINGER_PACKET));
      host_packets[++host_packets_len] = (FINGER_PACKET *)NULL;
    }

  /* Update the disk file which contains the last known user locations. */
  if (packet->name[0] != '\0')
  {
    register int i;
    long offset;

    /* use idle_time in user packets as time stamps */
    if (lastlog)
      {
        if (! gateway)
	  packet->idle_time = packet->login_time;
      }
    else
        packet->idle_time = timestamp - packet->idle_time;

    /* find user packet to update */
    for (i = 0; user_packets[i]; i++)
      {
	if ((strcmp (user_packets[i]->name, packet->name) == 0))
	  {
	    /* see if this needs updating */
#ifdef KEEP_LAST_LOGIN
	    if (user_packets[i]->login_time < packet->login_time)
#else
	    if (user_packets[i]->idle_time < packet->idle_time)
#endif
	      {
		bcopy (packet, user_packets[i], sizeof (FINGER_PACKET));
		offset = i * (long)sizeof(FINGER_PACKET);
		if (flock (user_file, LOCK_EX) < 0 ||
		    lseek (user_file, offset, 0) != offset)
		    file_error(FATAL, userdata);
		write (user_file, packet, sizeof (FINGER_PACKET));
		flock (user_file, LOCK_UN);
	      }
	    break;
	  }
      }
  
    if (!user_packets[i]) /* not found */
      {
	if (i + 2 > user_packets_size)
	  user_packets =
	    (FINGER_PACKET **)xrealloc (user_packets,
				  (user_packets_size += 10)
				  * sizeof (FINGER_PACKET *));

	user_packets[i] = (FINGER_PACKET *)xmalloc (sizeof (FINGER_PACKET));
	bcopy (packet, user_packets[i], sizeof (FINGER_PACKET));
	user_packets[i + 1] = (FINGER_PACKET *)NULL;

	offset = i * (long)sizeof(FINGER_PACKET);
	if (flock (user_file, LOCK_EX) < 0 ||
	    lseek (user_file, offset, 0) != offset)
	    file_error(FATAL, userdata);
	write (user_file, packet, sizeof (FINGER_PACKET));
        flock (user_file, LOCK_UN);
      }
  }
}

/* Return non-zero if PACKET->host == HOST. */
int
host_cmp (packet, host)
     FINGER_PACKET *packet;
     char *host;
{
  return (same_hostname (packet->host, host));
}

/*ARGSUSED*/
int
true_cmp (packet, host)
     FINGER_PACKET *packet;
     char *host;
{
  return 1;
}

/* check packet for sanity */
int
bogus_packet (packet)
     FINGER_PACKET *packet;
{
  char *p;

  /* whitespace or non-ascii in user name */
  for (p = packet->name; *p; p++)
    {
      if (isspace(*p) || !isascii(*p))
	return (1);
    }

  /* empty host name */
  if (! *packet->host)
    return (1);

  /* whitespace or non-ascii in host name */
  for (p = packet->host; *p; p++)
    {
      if (isspace(*p) || !isascii(*p))
	return (1);
    }

  /* non-dummy packet with empty tty name */
  if (*packet->name && ! *packet->ttyname)
    return (1);

  /* whitespace or non-ascii in tty name */
  for (p = packet->ttyname; *p; p++)
    {
      if (isspace(*p) || !isascii(*p))
	return (1);
    }

  return (0);
}

/* Remove all entries from the LIST which are from HOST. */
int
remove_host_packets (list, host)
     FINGER_PACKET **list;
     char *host;
{
  return remove_packets (list, host_cmp, host);
}

/* Write out the contents of the CLIENTS array.  This lets us keep track
   of the state of the clients that are being polled.
   Returns -1 if any I/O error occurred. */
int
write_host_status (clients, filedes)
  CLIENT **clients;
  int filedes;
{
  register int i;

  /* write file from start */
  if (lseek(filedes, 0L, 0) != 0)
      return -1;

  if (clients)
    {
      for (i = 0; clients[i]; i++)
	if (write (filedes, clients[i], sizeof (CLIENT)) != sizeof(CLIENT))
	  return -1;
    }

  /* through away any old data */
  if (ftruncate(filedes, (long)(i * sizeof (CLIENT))) < 0)
      return -1;

  return 0;
}

