/*
 * util.c --
 *	Simple utility functions that everyone uses.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: util.c,v 1.27 1994/04/27 07:45:59 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <pwd.h>

#include "general.h"
#include "util.h"
#include "error.h"
#include "fingerpaths.h"

/* **************************************************************** */
/*								    */
/*			General Utility				    */
/*								    */
/* **************************************************************** */

/* Array processing. */

/* Return the length of the null terminated array of pointers. */
int
array_len (array)
     char **array;
{
  register int i;

  for (i = 0; array[i]; i++);
  return (i);
}

/* Free the contents of the null terminated array, and
   then the array itself. */
void
free_array (array)
     char **array;
{
  register int i;

  for (i = 0; array[i]; i++)
    free (array[i]);

  free (array);
}

/* Table for fast toupper() */
char toupper_table[256] = {
      0,   1,   2,   3,   4,   5,   6,   7,
      8,   9,  10,  11,  12,  13,  14,  15,
     16,  17,  18,  19,  20,  21,  22,  23,
     24,  25,  26,  27,  28,  29,  30,  31,
     32,  33,  34,  35,  36,  37,  38,  39,
     40,  41,  42,  43,  44,  45,  46,  47,
     48,  49,  50,  51,  52,  53,  54,  55,
     56,  57,  58,  59,  60,  61,  62,  63,
     64,  65,  66,  67,  68,  69,  70,  71,
     72,  73,  74,  75,  76,  77,  78,  79,
     80,  81,  82,  83,  84,  85,  86,  87,
     88,  89,  90,  91,  92,  93,  94,  95,
     96,  65,  66,  67,  68,  69,  70,  71,
     72,  73,  74,  75,  76,  77,  78,  79,
     80,  81,  82,  83,  84,  85,  86,  87,
     88,  89,  90, 123, 124, 125, 126, 127,
    128, 129, 130, 131, 132, 133, 134, 135,
    136, 137, 138, 139, 140, 141, 142, 143,
    144, 145, 146, 147, 148, 149, 150, 151,
    152, 153, 154, 155, 156, 157, 158, 159,
    160, 161, 162, 163, 164, 165, 166, 167,
    168, 169, 170, 171, 172, 173, 174, 175,
    176, 177, 178, 179, 180, 181, 182, 183,
    184, 185, 186, 187, 188, 189, 190, 191,
    192, 193, 194, 195, 196, 197, 198, 199,
    200, 201, 202, 203, 204, 205, 206, 207,
    208, 209, 210, 211, 212, 213, 214, 215,
    216, 217, 218, 219, 220, 221, 222, 223,
    224, 225, 226, 227, 228, 229, 230, 231,
    232, 233, 234, 235, 236, 237, 238, 239,
    240, 241, 242, 243, 244, 245, 246, 247,
    248, 249, 250, 251, 252, 253, 254, 255
};

/* Whoops, Unix doesn't have strnicmp.  String functions. */

/* Compare at most COUNT characters from string1 to string2.  Case
   doesn't matter. */
int
strnicmp (string1, string2, count)  /* same as strncasecmp() in <string.h> */
     char *string1, *string2;
{
  while (--count >= 0 && toupper(*string1) == toupper(*string2++))
	if (*string1++ == '\0') return 0 ;

  return (count < 0 ? 0 : toupper(*string1) - toupper(*--string2));
}

/* strcmp (), but caseless. */
int
stricmp (string1, string2)	/* same as strcasecmp() in <string.h> */
     char *string1, *string2;
{

  while (toupper(*string1) == toupper(*string2++))
	if ( *string1++ == '\0' ) return 0;

  return toupper(*string1) - toupper(*--string2);
}

/* Determine if s2 occurs in s1.  If so, return a pointer to the
   match in s1.  The compare is case insensitive. */
char *
strindex (s1, s2)
     register char *s1, *s2;
{
  register int i, l = strlen (s2);
  register int len = strlen (s1);

  for (i = 0; (len - i) >= l; i++)
    if (strnicmp (&s1[i], s2, l) == 0)
      return (s1 + i);
  return ((char *)NULL);
}

/* Return a new string which is the concatenation of PATH and FILE. */
char *
path_concat (path, file)
     char *path, *file;
{
  char *output = (char *)xmalloc (2 + strlen (path) + strlen (file));
  register int i;

  strcpy (output, path);

  for (i = strlen (output) - 1; i > -1; i--)
    if (output[i] != ' ')
      break;

  output[i + 1] = '\0';

  if (output[0] && (output[strlen (output) - 1] != '/'))
    strcat (output, "/");
  strcat (output, file);
  return (output);
}

/* **************************************************************** */
/*								    */
/*			Generate Idle Time			    */
/*								    */
/* **************************************************************** */

/* Number of seconds below which the terminal is not idle. */
time_t idle_time_threshold = 60;

/* Macros for formatting possibly pluralized strings to time_buffer. */
#define ADD_SHORT(var, str) \
	sprintf (time_buffer, "%d %s", var, str)
#define ADD_LONG(var, str) \
	sprintf (time_buffer + strlen (time_buffer), "%d %s%s, ", \
	         var, str, (var) > 1 ? "s" : "")

/* Return a static string which is the English representation of the
   amount of idle time present in ITIME.  Note that ITIME is a time_t. */
char *
idle_time_string (itime, short_format)
     time_t itime;
     int short_format;
{
  int seconds, minutes, hours, days, weeks, months, years;
  char time_buffer[128];

  time_buffer[0] = '\0';

  if (itime >= idle_time_threshold)
    {
      seconds = itime % 60;
      minutes = itime / 60;

      hours = minutes / 60; minutes %= 60;
      days = hours / 24; hours %= 24;
      weeks = days / 7; days %= 7;
      months = weeks / 4; weeks %= 4;
      years = months / 12; months %= 12;

      if (short_format)
	{
	  if (years)       ADD_SHORT (years, "yr"); 
	  else if (months) ADD_SHORT (months, "mo");
	  else if (weeks)  ADD_SHORT (weeks, "wk");
	  else if (days)   ADD_SHORT (days, "day");
	  else	sprintf (time_buffer, "%2d:%02d", hours, minutes);
	}
      else
        {
	  if (years)	ADD_LONG (years, "year");
	  if (months)	ADD_LONG (months, "month");
	  if (weeks)	ADD_LONG (weeks, "week");
	  if (days)	ADD_LONG (days, "day");
	  sprintf (time_buffer + strlen (time_buffer), "%2d:%02d:%02d",
		   hours, minutes, seconds);
        }
    }

  return ((char *)strcpy
	  ((char *)xmalloc (1 + strlen (time_buffer)), time_buffer));
}

#ifndef USE_DBMALLOC

static void
memory_error_and_abort (nbytes)
     int nbytes;
{
  handle_error(FATAL, "(re)alloc error. Cannot allocate %d bytes.", nbytes);
}
  
extern char *malloc(), *realloc();

void *
xmalloc (nbytes)
     int nbytes;
{
  char *temp = malloc (nbytes);

  if (!temp)
    memory_error_and_abort (nbytes);

  return ((void *)temp);
}

void *
xrealloc (pointer, nbytes)
     char *pointer;
     int nbytes;
{
  char *temp = realloc (pointer, nbytes);

  if (!temp)
    memory_error_and_abort (nbytes);

  return ((void *)temp);
}
#endif /* USE_DBMALLOC */

/* Return the real person name of ENTRY. */
#define SAMENAME '&'
#define ASTERISK '*'
char *
pw_real_name (entry)
     struct passwd *entry;
{
  static char *real_name_buffer = (char *)NULL;
  char *t, *s;
  int size = 0;	/* number of `&' seen in the name field */

  if (real_name_buffer)
    free (real_name_buffer);

  real_name_buffer = (char *)xmalloc ((size = strlen (entry->pw_gecos) + 1));

  s = entry->pw_gecos;
  t = real_name_buffer;

  /* ignore leading asterisk
     (XXX: what was this for again?) */
  if (*s == ASTERISK)
    s++;

  /* skip leading whitespace characters, which some people
     insist on putting in here ... */
  while (isspace(*s))
    s++;

  /* copy name */
  while (*s && *s != ';' && *s != ',')
    {
      if (*s == SAMENAME)
	{
	  char *lp = entry->pw_name;
	  int old_offset = t - real_name_buffer;

	  /* extend buffer for inlining */
	  real_name_buffer = (char *)xrealloc(real_name_buffer,
				  (size += strlen(entry->pw_name)));
	  t = real_name_buffer + old_offset;

	  /* inline the login name */
	  if (islower(*lp))
	    *t++ = toupper(*lp++);
	  while (*t++ = *lp++) ;
	  t--;
	  s++;
        }
       else
	*t++ = *s++;
    }
  *t = '\0';

  return (real_name_buffer);
}

/* Global config file pointer (to avoid multiple opening) */
static FILE *config_file = NULL;

/* Open standard config file.  Abort on failure. */
FILE *
open_host_config ()
{
  if (config_file)
    rewind (config_file);
  else
    {
      if (! (config_file = fopen (HOSTCONFIGFILE, "r")))
	file_error (FATAL, HOSTCONFIGFILE);
    }
   
  return (config_file);
}

/* Close standard config file. */
void
close_host_config ()
{
  if (config_file)
    {
      (void) fclose (config_file);
      config_file = NULL;
    }
}

/* Config file reading and lookup.
   If KEY is non-null and non-empty string, the FILE is for the next entry
   matching the KEY and the associated entry is returned.
   If KEY is null or empty, the next entry is returned with it's key as
   a string in the format <key>\0<entry>\0.
   If now matching entry is found, NULL is returned.
   If FILE is null, the standard config file is used. */
char *
get_config_entry (key, file)
    char *key;
    FILE *file;
{
  char buffer[256];
  char *entry = NULL;
  int nokey = (!key || !*key);

  if (!file)
    {
      if (!config_file)
	(void) open_host_config ();

      file = config_file;
    }

  while (fgets(buffer, sizeof(buffer), file))
    {
      char *kstart, *kend;
      char *start, *end;
	
      /* skip whitespace */
      for (kstart = buffer; *kstart && isspace(*kstart); kstart++);
	  
      /* skip empty and comment lines */
      if (!*kstart || *kstart == '#')
	continue;
	  
      /* find end of key */
      for (kend = kstart + 1; *kend && !isspace(*kend); kend++);

      /* skip if not our key */
      if (!nokey && strnicmp(key, kstart, kend - kstart) != 0)
        continue;

      /* trim white space at beginning and end of entry */
      for (start = kend; *start && isspace(*start); start++);
	  
      if (!*start)
	end = start;
      else
        {
          for (end = start + strlen(start) - 1;
	       end != start && isspace(*end); end--);
          if (end != start) end++;
          *end = '\0';
	}
	  
      if (nokey)
        {
          entry = xmalloc((kend - kstart) + (end - start) + 2);
          strncpy(entry, kstart, kend - kstart);
	  entry[kend - kstart] = '\0';
          strcpy(entry + (kend - kstart) + 1, start);
	}
      else
        {
          entry = xmalloc(end - start + 1);
          strcpy(entry, start);
	}
      break;
    }
  
  if (!entry)
    rewind (file);

  return (entry);
}

/* Matching of host addresses against hostconfig entries.
   The HOST (which is typically either an IP number or a domain host name)
   is matched against entries with attribute KEY in the config FILE.
   A match is when
	- the entry is the empty string
	- the entry and HOST are identical
	- the entry ends in a dot and is a prefix of HOST
	- the entry starts with a dot and is a suffix of HOST
   An entry preceded by `!' is matched (without the leading character),
   and if successful returns a negative result.
   Comparisons are case-insentive.
   The FILE rewound to its beginning (to facilitate repeated lookups).
   If FILE is NULL, the standard config file is used. */
int
match_host_config (key, host, file)
    char *key;
    char *host;
    FILE *file;
{
  char *entry;
  int hostlen = strlen (host);
  int result = 0, negate = 0;

  if (!file)
    file = open_host_config ();
  else
    rewind (file);

  while (!result && (entry = get_config_entry (key, file)))
    {
      int len = strlen (entry);

      if (*entry == '!')
	{
	  negate = 1;
	  entry++;
	  len--;
	}
      else
	negate = 0;
   
      if (*entry == '\0')
	result = 1;
      if (entry[0] == '.')
	{
	  if (hostlen > len && stricmp (host + hostlen - len, entry) == 0)
	    result = 1;
	}
      else if (len > 0 && entry[len - 1] == '.')
	{
	  if (strnicmp (host, entry, len) == 0)
	    result = 1;
	}
      else
	{
	  if (stricmp (host, entry) == 0)
	    result = 1;
	}

      if (negate)
	entry--;

      free (entry);
    }

  rewind (file);
  return (!negate && result);
}

