/*
 * userinfo.c --
 *	The default method of printing out the verbose info
 *	on a given user.  Another choice might be to print out values found
 *	in a database where each record contained interesting info about
 *	a user.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: icsi_info.c,v 1.41 1995/08/31 02:21:47 stolcke Exp $ ICSI (Berkeley)";
#endif

#include "../config.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef LOCAL_LOGINS
#include <lastlog.h>
#endif
#include <sys/stat.h>
#include <pwd.h>
#include <fcntl.h>
#include <sys/file.h>
#ifndef R_OK
#include <unistd.h>
#endif

#include "general.h"
#include "call.h"
#include "packet.h"
#include "bitmap.h"
#include "util.h"
#include "error.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif
#ifndef S_ISREG
#define	S_ISREG(m)	(((m)&S_IFMT) == S_IFREG)
#endif

#ifndef MAILDIR
#define MAILDIR "/usr/spool/mail"
#endif

/* Location of the lastlog file.  This is for regular fingering. */
#ifdef LOCAL_LOGINS
#ifndef LASTLOG_FILE
#define LASTLOG_FILE    "/usr/adm/lastlog"
#endif
#endif

/* Width of ctime(3) generated string excluding final newline */
#define CTIME_WIDTH 24

/*
 *  The code for decoding the GECOS field is taken from the UCB version 
 * of finger.  Take it or leave it.
 */
#define ASTERISK        '*'             /* ignore this in real name */
#define COMMA           ','             /* separator in pw_gecos field */
#define CORY            'C'             /* cory hall office */
#define EVANS           'E'             /* evans hall office */

static char *pers_office, *pers_officephone, *pers_homephone, *pers_random;

/*
 *  very hacky section of code to format phone numbers.  filled with
 *  magic constants like 4, 7 and 10.
 */

static char *
phone(s, len, alldigits)
	register char *s;
	int len;
	char alldigits;
{
	char fonebuf[15];
	register char *p = fonebuf;
	register i;

	if (!alldigits)
		return (strcpy(xmalloc(len + 1), s));
	switch (len) {
	case 4:
		*p++ = ' ';
		*p++ = 'x';
		*p++ = '2';
		*p++ = '-';
		for (i = 0; i < 4; i++)
			*p++ = *s++;
		break;
	case 5:
		*p++ = ' ';
		*p++ = 'x';
		*p++ = *s++;
		*p++ = '-';
		for (i = 0; i < 4; i++)
			*p++ = *s++;
		break;
	case 7:
		for (i = 0; i < 3; i++)
			*p++ = *s++;
		*p++ = '-';
		for (i = 0; i < 4; i++)
			*p++ = *s++;
		break;
	case 10:
		for (i = 0; i < 3; i++)
			*p++ = *s++;
		*p++ = '-';
		for (i = 0; i < 3; i++)
			*p++ = *s++;
		*p++ = '-';
		for (i = 0; i < 4; i++)
			*p++ = *s++;
		break;
	case 0:
		return 0;
	default:
		return (strcpy(xmalloc(len + 1), s));
	}
	*p++ = 0;
	return (strcpy(xmalloc(p - fonebuf), fonebuf));
}

/*
 * decode the information in the gecos field of /etc/passwd
 */
static
decode(entry)
	struct passwd *entry;
{
	char buffer[256];
	register char *bp, *gp;
	int alldigits;
	int hasspace;
	int len;

	pers_office = NULL;
	pers_officephone = NULL;
	pers_homephone = NULL;
	pers_random = NULL;

	gp = entry->pw_gecos;
	bp = buffer;
	if (*gp == ASTERISK)
		gp++;
	while (*gp && *gp != COMMA)
				/* skip name -- handled by pw_real_name() */
		gp++;
	if (*gp == COMMA) {				/* office */
		gp++;
		hasspace = 0;
		bp = buffer;
		while (*gp && *gp != COMMA) {
			*bp = *gp++;
			if (*bp == ' ')
				hasspace = 1;
			/* leave 5 for Cory and Evans expansion */
			if (bp < buffer + sizeof buffer - 6)
				bp++;
		}
		*bp = 0;
		len = bp - buffer;
		bp--;			/* point to last character */
		if (hasspace || len == 0)
			len++;
		else if (*bp == CORY) {
			strcpy(bp, " Cory");
			len += 5;
		} else if (*bp == EVANS) {
			strcpy(bp, " Evans");
			len += 6;
		} else
			len++;
		if (len > 1)
			pers_office = strcpy(xmalloc(len), buffer);
	}
	if (*gp == COMMA) {				/* office phone */
		gp++;
		bp = buffer;
		alldigits = 1;
		while (*gp && *gp != COMMA) {
			*bp = *gp++;
			if (!isdigit(*bp))
				alldigits = 0;
			if (bp < buffer + sizeof buffer - 1)
				bp++;
		}
		*bp = 0;
		pers_officephone = phone(buffer, bp - buffer, alldigits);
	}
	if (*gp == COMMA) {				/* home phone */
		gp++;
		bp = buffer;
		alldigits = 1;
		while (*gp && *gp != COMMA) {
			*bp = *gp++;
			if (!isdigit(*bp))
				alldigits = 0;
			if (bp < buffer + sizeof buffer - 1)
				bp++;
		}
		*bp = 0;
		pers_homephone = phone(buffer, bp - buffer, alldigits);
	}
}

/* check that a file is plain */
static int
is_plain(path)
     char *path;
{
     struct stat sbuf;

     if (stat(path, &sbuf) < 0)
	return 0;
     else if (S_ISREG(sbuf.st_mode))
	return 1;
     else {
	handle_error(WARNING, "Warning: %s: not a plain file", path);
	return 0;
     }
}

/* Print out information about the user in ENTRY on STREAM.
   PACKETS contains some login state from the local machine. */
/*ARGSUSED*/
display_finger_info (entry, stream, packets)
     struct passwd *entry;
     FILE *stream;
     FINGER_PACKET **packets;
{
  struct stat finfo;
  extern char peer_name[];	/* from in.fingerd.c */
  char *mail_file = path_concat (MAILDIR, entry->pw_name);
  char *proj_file = path_concat (entry->pw_dir, ".project");
  char *plan_file = path_concat (entry->pw_dir, ".plan");
  int hasface = 0;
  
  /* Check for face bitmap */
  {
    char *mugshot = mugshot_filename (entry->pw_name, entry);

    if (mugshot && (access (mugshot, R_OK) == 0))
      hasface = 1;

    if (mugshot)
      free (mugshot);
  }

  /*
   * The header output is designed to be reminiscent of the UCB format
   */
  fprintf(stream, "Login name: %-8s %-19s", entry->pw_name,
                   (hasface == 1 ? "(bitmap available)" : ""));
  fprintf(stream, "In real life: %s", pw_real_name (entry));
  fprintf(stream, "\r\n");

  /* GECOS info */
  decode(entry);
  if (pers_office) {
      if (pers_officephone) {
	      pers_office = (char *)xrealloc(pers_office, 
			strlen(pers_office) + strlen(pers_officephone) + 3);
	      strcat(pers_office, ", ");
	      strcat(pers_office, pers_officephone);
      }
      fprintf(stream, "Office: %-32s", pers_office);
      if (pers_homephone)
	      fprintf(stream, "Home phone: %s", pers_homephone);
      else if (pers_random)
	      fprintf(stream, "%s", pers_random);
      fprintf(stream, "\r\n");
  } else if (pers_officephone) {
      fprintf(stream, "Phone: %s", pers_officephone);
      if (pers_homephone)
	      fprintf(stream, ", %s", pers_homephone);
      if (pers_random)
	      fprintf(stream, ", %s", pers_random);
      fprintf(stream, "\r\n");
  } else if (pers_homephone) {
      fprintf(stream, "Phone: %s", pers_homephone);
      if (pers_random)
	      fprintf(stream, ", %s", pers_random);
      fprintf(stream, "\r\n");
  } else if (pers_random)
      fprintf(stream, "%s\r\n", pers_random);

  /* user environment info */
  fprintf(stream, "Directory: %-29s", entry->pw_dir);
  if ( *entry->pw_shell )
	fprintf(stream, "Shell: %-s", entry->pw_shell);
  fprintf(stream, "\r\n");

#ifndef LOCAL_LOGINS 
  /* print a list of network logins for the user */
  finger_default(entry->pw_name, INFINGERD, stream);
#else /* LOCAL_LOGINS */
  {
    int packets_output = 0;

    /* If the user is logged in, show the login packets. */
    if (packets)
      {
	int i;

	for (i = 0; packets[i]; i++)
	  {
	    if (strcmp (entry->pw_name, packets[i]->name) == 0)
	      {
		ascii_packet (packets[i], stream, packets_output == 0);
		packets_output++;
	      }
	  }
      }

    /* If the user is not currently logged in, get the last known
       login time. */
    if (packets_output == 0)
      {
	struct lastlog logent;
	int file;

	if ((file = open (LASTLOG_FILE, O_RDONLY)) >= 0)
	  {
	    if ((lseek (file, (long) (entry->pw_uid * sizeof (logent)), 0)
								    != -1) &&
		(read (file, &logent, sizeof (logent)) == sizeof (logent)) &&
		logent.ll_time != 0 )
	      {
		fprintf (stream, "Last login on %s ",
			 logent.ll_line[0] ? logent.ll_line : "(no tty)");
  #define HMAX sizeof(logent.ll_host)
		if (*logent.ll_host) {
		  fprintf (stream, "from %.*s, ", HMAX, logent.ll_host);
		}
		fprintf (stream, "on %.*s.\r\n", CTIME_WIDTH,
		                 ctime (&(logent.ll_time)));
	      }
	    else
	      fprintf (stream, "Never logged in.\r\n");

	    close (file);
	  }
      }
  }
#endif /* !LOCAL_LOGINS */

  /* check mail forwarding */
  if (mail_expand(entry->pw_name, stream, peer_name))
    /* prints mail forwarding, if any */;
  else if (stat (mail_file, &finfo) < 0 || finfo.st_size == 0)
    fprintf (stream, "No mail.\r\n");
  else
    {
      /* On System 5 (well, HP/UX at least), atime of mailbox seems to
	 be set equal to mtime on mail delivery.  So there is no way to
	 detect `New mail'.  The same is true on BSD systems when the
	 mailbox is first created. */
      if (finfo.st_atime == finfo.st_mtime)
	{
	  fprintf (stream, "Unread mail since %.*s.\r\n", CTIME_WIDTH,
                           ctime (&finfo.st_mtime));
	}
      else if (finfo.st_atime < finfo.st_mtime)
	{
	  char *temp_string;

	  fprintf (stream, "New mail since %.*s.\r\n", CTIME_WIDTH,
	                   ctime (&finfo.st_mtime));
	  temp_string =
	    idle_time_string ((time_t)time ((time_t *)0) - finfo.st_atime, 0);
	  if (*temp_string)
	    fprintf (stream, "Has not read mail for %s.\r\n", temp_string);
	  free (temp_string);
	}
      else
	{
	  fprintf (stream, "No unread mail.\r\n");
	}
    }


  /* Maybe do the .project file. */
  {
    FILE *istream;
    char c;

    if (is_plain(proj_file) &&
	(istream = fopen(proj_file, "r")))
      {
	fprintf (stream, "Project: ");
	while ((c = getc(istream)) != EOF) {
		if (c == '\n') break;
		if (!(isprint(c) || isspace(c))) c ^= 100;
		putc(c, stream);
	}
	fprintf (stream, "\r\n");
	fclose (istream);
      }
  }

  /* Maybe do the .plan file. */
  {
    FILE *istream;
    char buff[256];

    if (is_plain(plan_file) &&
	(istream = fopen(plan_file, "r")))
      {
#ifdef CPP_CMD
        int first = getc(istream);
#endif
	fprintf (stream, "Plan:\r\n");
#ifdef CPP_CMD
        if (first != '#')
	  {
	    ungetc(first, istream);
#endif /* CPP_CMD */
	    while (fgets (buff, sizeof(buff), istream))
	      {	
		int linelen = strlen (buff);

		/* make sure line if CR-NL terminated */
		if (linelen > 0 && buff[linelen - 1] == '\n')
		   buff[linelen - 1] = '\0';
	        fprintf (stream, "%s\r\n", buff);
	      }
	    fclose (istream);
#ifdef CPP_CMD
	  }
	else 
	  {
    	    extern int local_connect;
	    time_t now = time((time_t *)NULL);
	    struct tm *tm = localtime(&now);

	    fclose (istream);
	    if (tm)
	      sprintf(buff, "%s \
-DLOCAL=%d -DYEAR=%d -DMONTH=%d -DDAY=%d -DHOUR=%d -DMINUTE=%d %s",
				CPP_CMD,
				local_connect,
				tm->tm_year,
				tm->tm_mon + 1,
				tm->tm_mday,
				tm->tm_hour,
				tm->tm_min,
				plan_file);

	    if (tm && chdir(entry->pw_dir) == 0 &&
	        (istream = popen(buff, "r")))
	      {
		first = 1;
	        while (fgets (buff, sizeof(buff), istream))
		  if (first)
		    first = 0;
	          else
		    {
		      int linelen = strlen (buff);

		      /* make sure line if CR-NL terminated */
		      if (linelen > 0 && buff[linelen - 1] == '\n')
			 buff[linelen - 1] = '\0';
	              fprintf (stream, "%s\r\n", buff);
		    }
	        pclose (istream);
	      }
	  }
#endif /* CPP_CMD */
      }
    else
      fprintf (stream, "No plan.\r\n");
  }

  free (mail_file);
  free (plan_file);
  free (proj_file);
  fflush (stream);
}
