/*
 * userface.c --
 *	Default methods of displaying a users face.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: defface.c,v 1.8 1994/10/11 19:26:11 stolcke Exp $ ICSI (Berkeley)";
#endif

#include "../config.h"

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_DIRENT
#include <dirent.h>
#else
#include <sys/dir.h>
#endif

#include "general.h"
#include "bitmap.h"

/* Return the face bits of USER.  If we can't, return a NULL pointer. */
BITMAP *
site_read_face (user)
     char *user;
{
  BITMAP *result = (BITMAP *)NULL;

#if defined (MIT_MUGSHOTS)
  {
    extern BITMAP *read_lispmug ();
    result = read_lispmug (user);
  }
#endif

#if defined (FACE_MUGSHOTS)
  {
    extern BITMAP *read_face ();
    result = read_face (user);
  }
#endif

#if defined (BITMAP_MUGSHOTS)
  {
    extern BITMAP *read_bitmap ();
    result = read_bitmap (user);
  }
#endif

#if defined (X11_MUGSHOTS)
  {
    extern BITMAP *read_x_bitmap ();
    result = read_x_bitmap (user);
  }
#endif

#if defined (RAST_MUGSHOTS)
  {
    extern BITMAP *read_rasterfile ();
    result = read_rasterfile (user);
  }
#endif

  return (result);
}

/* Create an array of the mugshot files in DIR. */
char **
mugshot_files_in_dir (dir)
     char *dir;
{
  DIR *directory;
  char name[256], **result = (char **)NULL;
  int result_index, result_size;
#ifdef HAVE_DIRENT
  struct dirent *entry = (struct dirent *)NULL;
#else
  struct direct *entry = (struct direct *)NULL;
#endif
  int dir_is_dot;

  result_index = result_size = 0;

  directory = opendir (dir);

  if (!directory)
    return ((char **)NULL);

  dir_is_dot = (strcmp (dir, ".") == 0);

  while (entry = readdir (directory))
    {
#ifdef HAVE_DIRENT
      int len = strlen(entry->d_name);
#else
      int len = entry->d_namlen;
#endif

      if (entry->d_name[0] == '.')
	continue;

      strncpy (name, entry->d_name, len);
      name[len] = '\0';

      /* Skip files that end in a `~'. */
      if (len && name[len - 1] == '~')
	continue;

      /* If current directory, then skip files that do not end in ".mug". */
      if (dir_is_dot)
	{
	  char *tail = strrchr (name, '.');

	  if (!tail || strcmp (tail, ".mug") != 0)
	    continue;
	}

      if (result_index + 1 >= result_size)
	{
	  if (result)
	    {
	      result = (char **)
		xrealloc (result, (result_size += 50) * sizeof (char **));
	    }
	  else
	    {
	      result = (char **)
		xmalloc ((result_size = 50) * sizeof (char **));
	    }
	}

      result[result_index] = (char *)xmalloc (1 + strlen (name));
      strcpy (result[result_index], name);
      result[++result_index] = (char *)NULL;
    }

  closedir (directory);
  return (result);
}

/* List the available faces on STREAM.  Return the number of
   faces listed. */
int
site_list_faces (stream)
     FILE *stream;
{
  char *dir, *path;
  char *getenv();
  int path_index = 0;
  int total_mugshots = 0;

  path = getenv ("MUGSHOT_PATH");

  if (!path)
    path = MUGSHOT_PATH;

  while (dir = (char *)extract_colon_unit (path, &path_index))
    {
      register int i;
      char **array;

      if (!*dir)
	{
	  free (dir);
	  dir = savestring (".");
	}

      array = mugshot_files_in_dir (dir);

      for (i = 0; array && array[i]; i++)
	{
	  fprintf (stream, "%s\n", array[i]);
	  fflush (stream);
	  free (array[i]);
	}

      free (dir);
      if (array)
	free (array);

      total_mugshots += i;
    }

  return (total_mugshots);
}

/* Save this user's face in a file. */
int
site_save_face (user, face)
     char *user;
     BITMAP *face;
{
  int result = -1;

#if defined (MIT_MUGSHOTS)
  {
    extern int save_lispmug ();
    result = save_lispmug (user, face);
  }
#endif

#if defined (BITMAP_MUGSHOTS)
  {
    extern int save_bitmap ();
    result = save_bitmap (user, face);
  }
#endif

#if defined (X11_MUGSHOTS)
  {
    extern int save_x_bitmap ();
    result = save_x_bitmap (user, face);
  }
#endif

#if defined (RAST_MUGSHOTS)
  {
    extern int save_rasterfile ();
    result = save_rasterfile (user, face);
  }
#endif

  return (result);
}

