/*
 * showface.c --
 *	Display a user's face.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: showface.c,v 1.10 1992/09/03 18:40:42 stolcke Exp $ ICSI (Berkeley)";
#endif

/* We make site_show_face() a seperate file to allow it to be linked
   separately -- it pulls all of the X and SunView libs in. */

#include "../config.h"

#include <stdio.h>
#include <string.h>

#include "bitmap.h"
#include "error.h"
#include "util.h"

extern char *getenv();

/* Show the FACE of USER on the local display. */
int
site_show_face (user, face)
     char *user;
     BITMAP *face;
{
#ifdef HAVE_X11
  if (getenv ("DISPLAY"))
    {
      if (!fork ())
	exit (x_show_face (user, face));
      return (0);
    }
#endif /* HAVE_X11 */
#ifdef HAVE_SUNVIEW
  if (getenv ("WINDOW_PARENT"))
    {
      if (!fork ())
	exit (sun_show_face (user, face));
      return (0);
    }
#endif /* HAVE_SUNVIEW */
  warning ("Need X11 or SunView to show face");
  return (-1);
}

#ifdef PRINT_WIDGET

/* Inches to Points conversion. */
#define	ITOP(in)	((int)((in) * 72.0))

/* LaserWriter X bias. */
#define	XTRANS_FUDGE	(1.0 / 16.0)

/* Width and height of the paper. */
#define w_page 8.5
#define h_page 11.0

#define	HI_NIBBLE(i)	(((i) >> 4) & 0xf)
#define	LO_NIBBLE(i)	((i) & 0xf)
#define ADJUST 0.932

/* Dump a face image to PostScript printer */
static int
print_image (name, image, width, height, depth, printer_name)
     char *name;
     byte *image;
     int width, height, depth;
     char *printer_name;
{
  FILE *output_stream;
  char *filename;
  register int i, *hexp;
  int hexdig[16];
  int error = 0;
  double aspect;
  double w_img = 7.5, h_img = 9.0;
  int result;

  aspect = width;
  aspect /= height;

  /*
   * Adjust PostScript image size to preserve aspect ratio.
   */
  if (aspect >= w_img / h_img)
    h_img = w_img / aspect;
  else
    w_img = h_img * aspect;

  switch (depth)
    {
    case 1:
      if (width % 16 != 0)
	width += 16 - width % 16;
      break;

    case 8:
      width += width & 1;
      break;
    }

  /* We have to reverse the image bit ordering as well as invert the bits.
     The fast way is with a translation table. */

  hexdig[15] = 'f';
  hexdig[14] = '7';
  hexdig[13] = 'b';
  hexdig[12] = '3';
  hexdig[11] = 'd';
  hexdig[10] = '5';
  hexdig[9]  = '9';
  hexdig[8]  = '1';
  hexdig[7]  = 'e';
  hexdig[6]  = '6';
  hexdig[5]  = 'a';
  hexdig[4]  = '2';
  hexdig[3]  = 'c';
  hexdig[2]  = '4';
  hexdig[1]  = '8';
  hexdig[0]  = '0';

  /* Create a temporary filename for postscript output. */
  filename = (char *)xmalloc (20 + strlen (name));
  sprintf (filename, "/tmp/%s-face", name);
  output_stream = fopen (filename, "w");

  if (!output_stream)
    {
      file_error (WARNING, filename);
      return (-1);
    }

  /* Prepare to print the file. */

  /* Postscript header. */
  fprintf (output_stream, "%%!\n");

  /* Center the image on the output page. */
  fprintf (output_stream, "%d %d translate\n",
	   ITOP (((w_page - w_img) / 2.0) + XTRANS_FUDGE),
	   ITOP ((h_page - h_img) / 2.0));

  fprintf (output_stream, "%d %d scale\n",
	   ITOP (w_img), ITOP (ADJUST * h_img));

  /* I/O buffer for readhexstring. */
  fprintf (output_stream, "/linebuf %d string def\n", width / (8 / depth));

  /* Say how big the image is. */
  fprintf (output_stream, "%ld %ld %d\n", width, height, depth);
  fprintf (output_stream, "[%ld 0 0 -%ld 0 %ld]\n", width, height, height);
  fprintf (output_stream, "{currentfile linebuf readhexstring pop}\n");

  /* Print the image data itself. */
  fprintf (output_stream, "image\n");

  hexp = hexdig;

  if (fflush (output_stream) == EOF)
    error = 1;

  for (i = 0; !error && i < (width * height * depth) / 8; i++)
    {
      if ((putc (hexp[LO_NIBBLE (~image[i])], output_stream) == EOF) ||
	  (putc (hexp[HI_NIBBLE (~image[i])], output_stream) == EOF))
	error = 1;
    }

  /* Show the page. */
  if (!error)
    fprintf (output_stream, "showpage\n");

  fclose (output_stream);

  /* Call the printer. */
  {
    char *command;

    command = (char *)xmalloc (20 + strlen (filename) + strlen (printer_name));

    sprintf (command, PRINT_COMMAND, printer_name, filename);
    result = system (command);
    free (command);
  }
  unlink (filename);
  free (filename);

  return (result);
}

int
site_print_face (user, face)
     char *user;
     BITMAP *face;
{
  char *printer;

  if ( !(printer = getenv("PRINTER")) )
	printer = DEFAULT_PRINTER;
	
  return print_image (user, face->bits, face->width, face->height, 1, printer);
}
#endif /* PRINT_WIDGET */

