/*
 * mail.c --
 *	SMTP interface for GNU finger
 *
 * Copyright (C) 1992 Free Software Foundation, Inc.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: mail.c,v 1.15 1994/01/11 02:14:00 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/param.h>

#include <netdb.h>
#include <netinet/in.h>
#if !defined (hpux)
#include <arpa/inet.h>
#endif

#include "../config.h"

#include "general.h"
#include "tcp.h"
#include "fingerpaths.h"
#include "util.h"
#include "os.h"
#include "error.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

#define FORWMSG "Mail forwarded to "
#define DISPWIDTH 50

/* The command send to SMTP agent to expand a mail alias. */
#ifndef SMTP_EXPAND_CMD
#define SMTP_EXPAND_CMD	"EXPN"
#endif

/* Return a new string which is the name of the SMTP server host.  If
   this cannot be determined, return a NULL pointer instead. */

static char *
getsmtphost ()
{
  char *smtphost = NULL;

  (void)open_host_config ();

  smtphost = get_config_entry ("smtphost", NULL);

  if (smtphost && !*smtphost)
    smtphost = LOCALHOST;

  return (smtphost);
}

/* Like FGETS, except reads until CR-LF. Doesn't return anything,
   either. Returns on LF, removes all occurences of CR (the LF is also
   removed). */

static void
xfgets (buf, bufsize, stream)
  char *buf;
  int bufsize;
  FILE *stream;
{
  register char *cp;

  for (cp = buf; cp < buf + bufsize - 1; cp++)
    switch (*cp = fgetc (stream))
      {
      case '\n':

	*cp = 0;
	return;

      case '\r':
	cp--;
	break;

#if 0
      default:

	fprintf (stdout, "%c", *cp); fflush (stdout);
#endif
      }

  buf[bufsize-1] = 0;
}

static FILE *smtpin, *smtpout;
static char *smtphost;

/* Open SMTP connection to mailhost. Fills in the SMPTIN and SMTPOUT
   streams and SMTPHOST, returning 0, or returns -1 to indicate error.
   PEER_NAME, if non-null, gives name of host on whose behalf the lookup
   is performed. */
int
smtp_connect (peer_name)
  char *peer_name;
{
  int sockfd;
  struct in_addr addr;
  struct hostent *host;
  char rbuf[1024];

  /* Get SMTP host name */
  if (!(smtphost = getsmtphost ()))
    return (-1);

  /* Connect to SMTP */
  if (isdigit (*smtphost))
    {
      addr.s_addr = inet_addr (smtphost);
      host = gethostbyaddr ((char *)&addr, sizeof(addr), AF_INET);
    }
  else
    host = gethostbyname (smtphost);

  if (!host)
    {
      warning ("SMTP host %s doesn't exist\n", smtphost);
      return (-1);
    }
      
  bcopy (host->h_addr, &addr, sizeof(addr));

  if ((sockfd = tcp_to_service ("smtp", &addr)) < 0)
    {
      warning ("failed to SMTP connect to %s\n", smtphost);
      return (-1);
    }

  smtpin = fdopen (sockfd, "r");
  smtpout = fdopen (dup (sockfd), "w");
  if (!smtpin || !smtpout)
    {
      if (smtpin)
	fclose (smtpin);
      return (-1);
    }

  /* skip over multiline greeting */
  do
    xfgets (rbuf, sizeof rbuf, smtpin);
  while (strncmp (rbuf, "220-", 4) == 0);

  /* Send HELO packet to identify ourselves (ignore reply) */
  if (!peer_name)
    fprintf (smtpout, "HELO %s@%s\n", progname ? progname : "finger",
				get_full_hostname ());
  else
    fprintf (smtpout, "HELO %s@%s@%s\n", progname ? progname : "finger",
				get_full_hostname (), peer_name);
  fflush (smtpout);

  /* skip all the HELO response */
  do
    xfgets (rbuf, sizeof rbuf, smtpin);
  while (strncmp (rbuf, "250-", 4) == 0);

  return (0);
}

/* Close SMTP connection to mailhost */
int
smtp_disconnect ()
{
  /* Be nice and terminate SMTP connection properly. */
  fprintf (smtpout, "QUIT\n");
  fflush (smtpout);

  fclose (smtpin);
  fclose (smtpout);

  return (0);
}

/* Print forwarding address of USER on STREAM. Zero is returned and
   nothing printed if no forwarding has been set up. */

int
mail_expand (login_name, stream, peer_name)
  char *login_name;
  FILE *stream;
  char *peer_name;
{
  int success = 0;
  int login_len = strlen(login_name);
  char rbuf[1024];

  if (smtp_connect (peer_name) < 0)
     return 0;

  /* Send EXPN packet and print response to STREAM */
  fprintf (smtpout, "%s %s\n", SMTP_EXPAND_CMD, login_name);
  fflush (smtpout);

  xfgets (rbuf, sizeof rbuf, smtpin);

  /* The inet addr seems to always be of the form "250.+<...>"  */
  if (rbuf[0] == '2' && rbuf[1] == '5' && rbuf[2] == '0' && rbuf[3])
    for (;;)
      {
	if (rbuf[strlen(rbuf)-1] == '>')
	  {
	    char *mailaddr;
	    int tailpos = strlen (rbuf) - 1;
	    
	    rbuf[tailpos] = '\0';
	    
	    for (mailaddr = rbuf + tailpos;
		 mailaddr > rbuf+4 && *mailaddr != '<'; mailaddr--);
	    mailaddr++;
	    
	    /* strip quotes */
 	    if (*mailaddr == '\\')
	      mailaddr++;
 	    else if (*mailaddr == '"')
	      {
		mailaddr++;
		if (strchr(mailaddr, '"'))
		  *strchr(mailaddr,  '"') = '\0';
	      }

	    if (*mailaddr != '/' &&	/* not a file */
		*mailaddr != '|' &&	/* not a program */
	        stricmp (mailaddr, login_name) != 0 &&
					/* expansion != name or name@smtphost */
		! (strnicmp (mailaddr, login_name, login_len) == 0
		   && mailaddr[login_len] == '@'
		   && same_hostip(mailaddr + login_len + 1, smtphost)))
	      {
		fprintf (stream, "%s%s", !success ? FORWMSG : ", ", mailaddr);
		success = 1;
	      }
	  }
	else
	  {
	    /* Don't know how record is formatted - just print it in its
	       entirety. */
	    fprintf (stream, "%s%s", !success ? FORWMSG : ", ", rbuf+4);
	    success = 1;
	  }
	
	if (rbuf[3] == '-')
	  xfgets (rbuf, sizeof rbuf, smtpin);
	else
	  break;
      }
  if (success)
    fprintf (stream, ".\r\n");

  smtp_disconnect ();
  return success;
}


/* List expansion of ALIAS on STREAM. Zero is returned and nothing
   printed if no matching alias exists. Otherwise the expansion is
   printed and a non-zero result returned. */

int
mail_list (alias, stream, peer_name)
  char *alias;
  FILE *stream;
  char *peer_name;
{
  char rbuf[1024];
  int success = 1;

  if (smtp_connect (peer_name) < 0)
    return (0);

  /* Send EXPN packet and print response to STREAM */
  fprintf (smtpout, "%s %s\n", SMTP_EXPAND_CMD, alias);
  fflush (smtpout);

  xfgets (rbuf, sizeof rbuf, smtpin);

  /* Loop responses */
  if (rbuf[0] == '2' && rbuf[1] == '5' && rbuf[2] == '0' && rbuf[3])
    {
      if (rbuf[3] == ' ')
	fprintf (stream, "The alias %s expands to %s.\r\n", alias, rbuf+4);
      else
	{
	  fprintf (stream, "%s is an alias for the following:\r\n", alias);

	  for (;;)
	    {
	      fprintf (stream, "    %s%s\r\n", rbuf+4, rbuf[3] == '-' ? "," : "\r\n");

	      if (rbuf[3] == '-')
		xfgets (rbuf, sizeof rbuf, smtpin);
	      else
		break;
	    }
	}
    }
  else
    /* No such alias */
    success = 0;

  smtp_disconnect ();
  return success;
}

#ifdef TEST
main (argc, argv)
  int argc;
  char *argv[];
{
  int i;

  for (i = 1; i < argc; i++)
   {
     if (!mail_expand(argv[i], stdout, NULL))
       printf("%s has no expansion\n", argv[i]);
     if (!mail_list(argv[i], stdout, NULL))
       printf("%s is not a mail list\n", argv[i]);
   }
   exit (0);
}
#endif /* TEST */
