/*
 * error.c --
 *	Simple error handling.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: error.c,v 1.13 1994/03/24 02:14:54 stolcke Exp $ ICSI (Berkeley)";
#endif

#include "../config.h"

#include <syslog.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#ifdef USE_SYSLOG
#include <syslog.h>
#endif

#include "general.h"
#include "error.h"

/* These can be filled in `manually' by callers, but the easiest way
   is to call default_error_handling (argv[0], stderr). */
char *progname = NULL;

jmp_buf top_level;

static FILE *errlog = stderr;

/* **************************************************************** */
/*								    */
/*			Error Handler				    */
/*								    */
/* **************************************************************** */

void
default_error_handling (program_name, logfile)
     char *program_name;
     FILE *logfile;
{
  if (strrchr (program_name, '/') != (char *)NULL)
    {
      program_name = strrchr (program_name, '/');
      program_name++;
    }

  progname = savestring (program_name);

  errlog = logfile;
#ifdef USE_SYSLOG
  if (errlog == LOG_SYS)
#ifndef LOG_DAEMON /* 4.2BSD syslog */
    openlog (progname, LOG_PID);
#else
    openlog (progname, LOG_PID|LOG_CONS, USE_SYSLOG);
#endif
#endif /* USE_SYSLOG */

  if (setjmp (top_level))
    exit (1);
}

/* Hack to handle previous bad setjmp (). */
void
longjmperror ()
{
  exit (1);
}

/* Handle some error. */
/*VARARGS2*/
void
handle_error (severity, format, arg1, arg2, arg3)
     int severity;
     char *format, *arg1, *arg2, *arg3;
{
#ifdef USE_SYSLOG
  if (errlog == LOG_SYS)
    {
      syslog (severity == FATAL ? LOG_ALERT :
	      (severity == ALERT ? LOG_ALERT :
	       (severity == ERROR ? LOG_ERR :
	        (severity == WARNING ? LOG_WARNING :
                 (severity == DEBUG ? LOG_DEBUG : LOG_NOTICE)))), 
              format, arg1, arg2, arg3);
    }
  else
#endif /* USE_SYSLOG */
       if (errlog != LOG_NONE)
    {
      if (progname && *progname)
        fprintf (stderr, "%s: ", progname);
      fprintf (stderr, format, arg1, arg2, arg3);
      if (format[strlen(format) - 1] != '\n')
        fprintf (stderr, "\n");
      fflush (stderr);
    }

  switch (severity)
    {
    case FATAL:
      exit (1);

    case ERROR:
      longjmp (top_level, 1);

    case ALERT:
    case WARNING:
    case DEBUG:
      break;

    default:
      handle_error(FATAL, "\
handle_error () called with bad severity (%d).\n\
Who knows what else is wrong with the code?  Stopping.", severity);
      exit (2);
    }
}

/* Handle a file error.  You pass severity and filename. If FILENAME is null,
   then don't print it. */
void
file_error (severity, filename)
     int severity;
     char *filename;
{
  extern sys_nerr;
  extern char *sys_errlist[];

  char *error_text;

  if (errno) {
    if (errno < sys_nerr)
      error_text = sys_errlist[errno];
    else
      error_text = "(large errno?)";
  } else {
    error_text = "(Hmm, no error?)";
  }

  if (filename == (char *)NULL)
    handle_error (severity, "%s", error_text, NULL, NULL);
  else
    handle_error (severity, "%s: %s", filename, error_text, NULL);
}

/*VARARGS1*/
void
warning (format, arg1, arg2, arg3)
     char *format, *arg1, *arg2, *arg3;
{
  handle_error (WARNING, format, arg1, arg2, arg3);
}

/*VARARGS2*/
void
debug (debugging, format, arg1, arg2, arg3)
     int debugging;
     char *format, *arg1, *arg2, *arg3;
{
  if (debugging)
    handle_error (DEBUG, format, arg1, arg2, arg3);
}
